package cn.lalaki.pub.tasks;

import cn.lalaki.pub.internal.BaseCentralPortalExtension;

import org.gradle.api.DefaultTask;
import org.gradle.api.tasks.TaskAction;

import java.io.File;
import java.io.IOException;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.util.Objects;

import okhttp3.Credentials;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;

/**
 * Task for publishing to Central Portal via Central API.
 * <p>
 * Created on yyyy-MM-dd
 *
 * @author thebugmc
 * @since BasePublishToCentralPortalTask
 */
public abstract class BasePublishToCentralPortalTask extends DefaultTask {
    /**
     * Central API URL for publishingType being {@code AUTOMATIC}.
     */
    private ChecksumTask checksumTask;

    /**
     * Mark what file should be uploaded.
     */
    public void upload(ChecksumTask checksumTask) {
        if (this.checksumTask != null && checksumTask != this.checksumTask) {
            throw new IllegalArgumentException("Checksum task already set to " + this.checksumTask);
        }
        dependsOn(checksumTask);
        this.checksumTask = checksumTask;
    }

    /**
     * Uploads the file via the Central API.
     *
     * @throws IOException If any errors happen during upload.
     */
    @TaskAction
    public void sendRequest() throws IOException {
        var target = getProject();
        var extension = target.getExtensions().getByType(BaseCentralPortalExtension.class);
        var username = extension.getUsername().getOrElse(Objects.toString(target.findProperty("centralPortal.username")));
        var password = extension.getPassword().getOrElse(Objects.toString(target.findProperty("centralPortal.password")));
        if (username.isEmpty() || password.isEmpty()) {
            var valueIsNull = username.isEmpty() ? "username" : "password";
            throw new IOException("Missing PublishToCentralPortal's `(" + valueIsNull + ")`, `centralPortal." + valueIsNull + "` value" + " and `centralPortal." + valueIsNull + "` property");
        }
        publishToMaven(username, password, extension.getPublishingType().getOrElse(BaseCentralPortalExtension.PublishingType.AUTOMATIC).name(), URLEncoder.encode(getProject().getGroup() + ":" + extension.getName().orElse(target.getRootProject().getName()).get() + ":" + getProject().getVersion(), Charset.defaultCharset()), checksumTask.outputFile());
    }

    private void publishToMaven(String username, String password, String publishingType, String bundleName, File outputFile) throws IOException {
        String mPublisherApi = "https://central.sonatype.com/api/v1/publisher/upload?publishingType=";
        try (var resp = new OkHttpClient().newCall(new Request.Builder()
                .url(mPublisherApi + publishingType + "&name=" + bundleName)
                .addHeader("Authorization", Credentials.basic(username, password).replace("Basic", "Bearer"))
                .post(new MultipartBody.Builder().setType(MultipartBody.FORM).addFormDataPart("bundle", "bundle.zip", RequestBody.create(Files.readAllBytes(outputFile.toPath()))).build()).build()).execute()) {
            if (!resp.isSuccessful()) {
                throw new IOException("Error " + resp.code() + ": " + resp.message());
            }
        }
    }
}