package net.thebugmc.gradle.sonatypepublisher;

import org.gradle.api.DefaultTask;
import org.gradle.api.provider.Property;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.Optional;
import org.gradle.api.tasks.TaskAction;

import java.io.File;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.Base64;
import java.util.Objects;

import static java.lang.System.currentTimeMillis;
import static java.nio.charset.StandardCharsets.UTF_8;

import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;

/**
 * Task for publishing to Central Portal via Central API.
 */
public abstract class PublishToCentralPortal extends DefaultTask {
    /**
     * Central API URL for publishingType being {@code AUTOMATIC}.
     */
    public static final String UPLOAD_ENDPOINT
            = "https://central.sonatype.com/api/v1/publisher/upload?publishingType=";

    private Checksum checksumTask;

    /**
     * Mark what file should be uploaded.
     */
    public void upload(Checksum checksumTask) {
        if (this.checksumTask != null && checksumTask != this.checksumTask)
            throw new IllegalArgumentException("Checksum task already set to " + this.checksumTask);

        dependsOn(checksumTask);
        this.checksumTask = checksumTask;
    }

    /**
     * User token used for the publishing.
     */
    @Input
    @Optional
    public abstract Property<String> getUsername();

    /**
     * Token passphrase used for the publishing.
     */
    @Input
    @Optional
    public abstract Property<String> getPassword();

    /**
     * Uploads the file via the Central API.
     *
     * @throws IOException If any errors happen during upload.
     */
    @TaskAction
    public void sendRequest() throws IOException {
        var target = getProject();
        var extension = target.getExtensions().getByType(CentralPortalExtension.class);
        var username = getUsername().getOrNull();
        if (username == null)
            username = extension.getUsername().getOrElse(Objects.toString(target.findProperty("centralPortal.username"), ""));
        var password = getPassword().getOrElse(extension.getPassword().getOrElse(Objects.toString(target.findProperty("centralPortal.password"), "")));
        if (username.isEmpty() || password.isEmpty()) {
            var valueIsNull = username.isEmpty() ? "username" : "password";
            throw new IOException("Missing PublishToCentralPortal's `(" + valueIsNull + ")`, `centralPortal." + valueIsNull + "` value" + " and `centralPortal." + valueIsNull + "` property");
        }
        var nameProvider = extension.getName().orElse(target.getRootProject().getName());
        publishToMaven(username, password, extension.getPublishingType().getOrElse(PublishingType.AUTOMATIC), URLEncoder.encode(getProject().getGroup() + ":" + nameProvider.get() + ":" + getProject().getVersion(), UTF_8), checksumTask.outputFile());
    }

    private void publishToMaven(String username, String password, PublishingType publishingType, String bundleName, File outputFile) throws IOException {
        var client = new OkHttpClient();
        var token = Base64.getEncoder().encodeToString((username + ":" + password).getBytes());
        var boundary = Long.toHexString(~currentTimeMillis());
        var requestBody = new MultipartBody.Builder(boundary).setType(MultipartBody.FORM).addFormDataPart("bundle", "bundle.zip", RequestBody.create(outputFile, MediaType.parse("application/octet-stream"))).build();
        var request = new Request.Builder().url(UPLOAD_ENDPOINT + publishingType + "&name=" + bundleName).post(requestBody).addHeader("Authorization", "UserToken " + token).build();
        try (var resp = client.newCall(request).execute()) {
            if (!resp.isSuccessful()) {
                throw new IOException(
                        "Error " + resp.code() + ": " + resp.message()
                );
            }
        }
    }
}