package cn.kicent.framework.base.support;

import org.springframework.beans.factory.config.YamlPropertiesFactoryBean;
import org.springframework.core.env.PropertiesPropertySource;
import org.springframework.core.env.PropertySource;
import org.springframework.core.io.support.EncodedResource;
import org.springframework.core.io.support.PropertySourceFactory;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;

import java.util.Objects;
import java.util.Properties;

/**
 * 加载 YAML 文件到 Properties 中
 *
 * @author yang.lu
 */
public class YamlPropertySourceFactory implements PropertySourceFactory {

	/**
	 * 根据提供的 YAML 资源创建 PropertySource
	 *
	 * @param name     PropertySource 名称
	 * @param resource YAML 文件资源
	 * @return PropertySource<?> 创建的 PropertySource 对象
	 * @author yang.lu
	 */
	@Override
	public @NonNull PropertySource<?> createPropertySource(@Nullable String name, @NonNull EncodedResource resource) {
		Properties properties = loadYaml(resource);
		String sourceName = (name != null)
			? name
			: Objects.requireNonNull(resource.getResource().getFilename(), "Resource filename cannot be null");
		return new PropertiesPropertySource(sourceName, properties);
	}

	/**
	 * 将 YAML 文件加载到 Properties 对象
	 *
	 * @param resource YAML 文件资源
	 * @return Properties 从 YAML 文件加载的 Properties 对象
	 * @author yang.lu
	 */
	private Properties loadYaml(EncodedResource resource) {
		YamlPropertiesFactoryBean factory = new YamlPropertiesFactoryBean();
		factory.setResources(resource.getResource());
		factory.afterPropertiesSet();
		return factory.getObject();
	}
}
