package cn.kicent.framework.base.api;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.kicent.framework.base.enums.IBaseEnum;
import cn.kicent.framework.base.enums.ResultEnum;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.NoArgsConstructor;

import java.io.Serial;
import java.io.Serializable;
import java.util.Optional;
import java.util.StringJoiner;

/**
 * 响应包装体
 *
 * @author yang.lu
 */
@Getter
@NoArgsConstructor(access = AccessLevel.PRIVATE)
@JsonPropertyOrder({"code", "message", "success", "data"})
public class Result<T> implements Serializable {

	@Serial
	private static final long serialVersionUID = 1L;

	private String code;
	private String message;
	private boolean success;
	private T data;

	private Result(IBaseEnum<String> baseEnum) {
		this(baseEnum, baseEnum.getDesc(), null);
	}

	private Result(IBaseEnum<String> baseEnum, String message) {
		this(baseEnum, message, null);
	}

	private Result(IBaseEnum<String> baseEnum, String message, T data) {
		this(baseEnum.getCode(), message, data);
	}

	private Result(String code, String message, T data) {
		this.code = code;
		this.message = message;
		this.success = ResultEnum.SUCCESS.equals(code);
		this.data = data;
	}

	public static <T> Result<T> success() {
		return success(null);
	}

	public static <T> Result<T> success(T data) {
		return new Result<>(ResultEnum.SUCCESS.getCode(), ResultEnum.SUCCESS.getDesc(), data);
	}

	public static <T> Result<T> failure(String message) {
		return new Result<>(ResultEnum.INTERNAL_SERVER_ERROR, message);
	}

	public static <T> Result<T> failure(String code, String message) {
		return new Result<>(code, message, null);
	}

	public static <T> Result<T> failure(String code, String message, T data) {
		return new Result<>(code, message, data);
	}

	public static <T> Result<T> failure(IBaseEnum<String> baseEnum) {
		return new Result<>(baseEnum);
	}

	public static <T> Result<T> failure(IBaseEnum<String> baseEnum, String message) {
		return new Result<>(baseEnum, message);
	}

	public static <T> Result<T> condition(boolean flag) {
		return flag ? success() : failure(ResultEnum.INTERNAL_SERVER_ERROR);
	}

	@Override
	public String toString() {
		final StringJoiner joiner = new StringJoiner(", ", "{", "}");

		Optional.ofNullable(code)
			.filter(StrUtil::isNotEmpty)
			.ifPresent(c -> joiner.add("\"code\":" + c));

		Optional.ofNullable(message)
			.filter(StrUtil::isNotEmpty)
			.ifPresent(m -> joiner.add("\"message\":\"" + m + "\""));

		joiner.add("\"success\":" + success);

		Optional.ofNullable(data)
			.filter(ObjectUtil::isNotNull)
			.ifPresent(d -> joiner.add("\"data\":" + d));

		return joiner.toString();
	}
}
