package cn.keayuan.util;

import android.content.Context;
import android.content.SharedPreferences;

/**
 * Created by keayuan on 2013/9/9.
 * <p>
 * SharedPreference 工具类
 * </p>
 *
 * @author keayuan001
 */
public class SpfUtil {
    private static final String DEFAULT = "spf_default";

    private SpfUtil() {
        // hide
    }

    public static SharedPreferences get(Context context) {
        return get(context, DEFAULT);
    }

    public static SharedPreferences get(Context context, String name) {
        return get(context, name, Context.MODE_PRIVATE);
    }

    public static SharedPreferences get(Context context, String name, int mode) {
        if (context != null) {
            return context.getSharedPreferences(name, mode);
        }
        return null;
    }

    /**
     * 查询某个key是否已经存在
     *
     * @param key 关键字
     */
    public static boolean contains(SharedPreferences sp, String key) {
        return require(sp).contains(key);
    }

    /**
     * @param key   关键字
     * @param value 存储值
     */
    public static void putInt(SharedPreferences sp, String key, int value) {
        require(sp).edit().putInt(key, value).apply();
    }

    /**
     * @param key          关键字
     * @param defaultValue 默认值
     */
    public static int getInt(SharedPreferences sp, String key, int defaultValue) {
        return require(sp).getInt(key, defaultValue);
    }

    /**
     * @param key 关键字
     * @return 默认返回0
     */
    public static int getInt(SharedPreferences sp, String key) {
        return getInt(sp, key, 0);
    }

    /**
     * @param key   关键字
     * @param value 存储值
     */
    public static void putBoolean(SharedPreferences sp, String key, boolean value) {
        require(sp).edit().putBoolean(key, value).apply();
    }

    /**
     * @param key 关键字
     * @return 默认false
     */
    public static boolean getBoolean(SharedPreferences sp, String key) {
        return getBoolean(sp, key, false);
    }

    /**
     * @param key          关键字
     * @param defaultValue 默认值
     */
    public static boolean getBoolean(SharedPreferences sp, String key, boolean defaultValue) {
        return require(sp).getBoolean(key, defaultValue);
    }

    /**
     * @param key   关键字
     * @param value 存储值
     */
    public static void putString(SharedPreferences sp, String key, String value) {
        require(sp).edit().putString(key, value == null ? "" : value).apply();
    }

    /**
     * @param key 关键字
     * @return 默认""
     */
    public static String getString(SharedPreferences sp, String key) {
        return getString(sp, key, "");
    }

    /**
     * @param key          关键字
     * @param defaultValue 默认值
     */
    public static String getString(SharedPreferences sp, String key, String defaultValue) {
        return require(sp).getString(key, defaultValue);
    }

    /**
     * @param key   关键字
     * @param value 存储值
     */
    public static void putFloat(SharedPreferences sp, String key, float value) {
        require(sp).edit().putFloat(key, value).apply();
    }

    /**
     * @param key 关键字
     * @return 默认返回0
     */
    public static float getFloat(SharedPreferences sp, String key) {
        return require(sp).getFloat(key, 0);
    }

    /**
     * @param key   关键字
     * @param value 存储值
     */
    public static void putLong(SharedPreferences sp, String key, long value) {
        require(sp).edit().putLong(key, value).apply();
    }

    /**
     * @param key 关键字
     * @return 默认返回0
     */
    public static long getLong(SharedPreferences sp, String key) {
        return require(sp).getLong(key, 0);
    }

    /**
     * 设置SharePreference文件中的字段的值
     *
     * @param key   关键字
     * @param value 值
     */
    public static boolean setValue(SharedPreferences sp, String key, Object value) {
        require(sp);
        if (key == null || value == null) return false;
        if (value instanceof String) {
            putString(sp, key, (String) value);
        } else if (value instanceof Integer) {
            putInt(sp, key, (Integer) value);
        } else if (value instanceof Long) {
            putLong(sp, key, (Long) value);
        } else if (value instanceof Float) {
            putFloat(sp, key, (Float) value);
        } else if (value instanceof Boolean) {
            putBoolean(sp, key, (Boolean) value);
        } else {
            return false;
        }
        return true;
    }

    /**
     * 获得SharePreference的值
     *
     * @param key          关键字
     * @param defaultValue 默认值
     * @return 获得对应key的值
     */
    @SuppressWarnings("unchecked")
    public static <T> T getValue(SharedPreferences sp, String key, T defaultValue) {
        if (key == null || sp == null) return defaultValue;
        if (defaultValue instanceof String) {
            return (T) getString(sp, key, (String) defaultValue);
        } else if (defaultValue instanceof Integer) {
            return (T) Integer.valueOf(getInt(sp, key));
        } else if (defaultValue instanceof Long) {
            return (T) Long.valueOf(getLong(sp, key));
        } else if (defaultValue instanceof Float) {
            return (T) Float.valueOf(getFloat(sp, key));
        } else if (defaultValue instanceof Boolean) {
            return (T) Boolean.valueOf(getBoolean(sp, key));
        } else {
            return null;
        }
    }

    /**
     * 移除某个key值已经对应的值
     *
     * @param key 关键字
     */
    public static void remove(SharedPreferences sp, String key) {
        require(sp).edit().remove(key).apply();
    }

    /**
     * 清除所有数据
     */
    public static void clear(SharedPreferences sp) {
        require(sp).edit().clear().apply();
    }

    private static <T> T require(T t) {
        if (t == null) throw new NullPointerException();
        return t;
    }
}
