package cn.keayuan.util;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.widget.Toast;

import java.lang.ref.WeakReference;

import cn.keayuan.util.function.BiFunction;

/**
 * Toast工具类
 */
public class ToastUtil {

    private static WeakReference<Toast> toastWeak;
    private static WeakReference<Application> contextWeak;
    private static final Handler mainHandler = new Handler(Looper.getMainLooper());
    private static BiFunction<Context, CharSequence, Toast> supplier;

    private ToastUtil() {
        // hide
    }

    public static void init(Application context) {
        init(context, null);
    }

    public static void init(Application context, BiFunction<Context, CharSequence, Toast> supplier) {
        contextWeak = new WeakReference<>(context);
        ToastUtil.supplier = supplier;
    }

    private static class Task implements Runnable {

        private final String content;
        private final boolean isLong;
        private final WeakReference<Context> weakContext;

        private Task(String content, boolean isLong) {
            this(contextWeak.get(), content, isLong);
        }

        private Task(Context context, String content, boolean isLong) {
            if (content == null) throw new NullPointerException();
            weakContext = new WeakReference<>(context);
            this.content = content;
            this.isLong = isLong;
        }

        @Override
        public void run() {
            cancel();
            Context context = weakContext.get();
            if (context == null || content == null || (context instanceof Activity && ((Activity) context).isFinishing())) {
                return;
            }
            synchronized (ToastUtil.class) {
                Toast toast;
                if (supplier != null) {
                    toast = supplier.apply(context, content);
                } else {
                    toast = new Toast(context);
                    toast.setText(content);
                }
                if (toast == null) return;
                toast.setDuration(isLong ? Toast.LENGTH_LONG : Toast.LENGTH_SHORT);
                toast.show();
                toastWeak = new WeakReference<>(toast);
            }
        }
    }

    private static void showToast(Task task) {
        mainHandler.post(task);
    }

    /**
     * Toast提示
     *
     * @param content 文案
     */
    public static void show(String content, boolean isLong) {
        showToast(new Task(content, isLong));
    }

    public static void show(String content) {
        show(content, false);
    }

    // context
    public static void show(Context context, int resId, Object... args) {
        show(context, resId, false, args);
    }

    public static void show(Context context, int resId, boolean isLong, Object... args) {
        show(context, context.getString(resId, args), isLong);
    }

    public static void show(Context context, String format) {
        show(context, format, false);
    }

    public static void show(Context context, String format, boolean isLong) {
        showToast(new Task(context, format, isLong));
    }

    public static void cancel() {
        Toast toast;
        synchronized (ToastUtil.class) {
            toast = toastWeak == null ? null : toastWeak.get();
        }
        if (toast != null) {
            toast.cancel();
        }
    }

    public static Builder create() {
        return new Builder();
    }

    public static Builder create(Context context) {
        return new Builder(context);
    }

    public static class Builder {
        private final Context context;

        private Builder(Context context) {
            if (context == null) throw new NullPointerException();
            this.context = context;
        }

        private Builder() {
            context = contextWeak == null ? null : contextWeak.get();
            if (context == null) throw new NullPointerException();
        }

        void show(String content, Object... args) {
            showToast(new Task(String.format(content, args), false));
        }

        void show(int resId, Object... args) {
            show(context.getString(resId, args));
        }

        void showLong(String content, Object... args) {
            showToast(new Task(String.format(content, args), true));
        }

        void showLong(int resId, Object... args) {
            showLong(context.getString(resId, args));
        }
    }
}
