package cn.keayuan.util;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.FutureTask;

import cn.keayuan.util.function.BiConsumer;
import cn.keayuan.util.function.Consumer;

public final class Task<V> extends FutureTask<V> {
    private final Pair<BiConsumer<Task<V>, V>, Executor> success;
    private final Pair<BiConsumer<Task<V>, Throwable>, Executor> error;
    private final Pair<Consumer<Task<V>>, Executor> cancel;
    private final Pair<Consumer<Task<V>>, Executor> complete;

    private final String tag;

    private Task(Builder<V> builder) {
        super(builder.callable);
        success = builder.success;
        error = builder.error;
        cancel = builder.cancel;
        complete = builder.complete;
        tag = builder.tag;
    }

    @Override
    protected void done() {
        if (isCancelled() && checkPair(cancel)) {
            cancel.second.execute(() -> cancel.first.accept(Task.this));
        } else {
            try {
                V v = get();
                if (checkPair(success)) {
                    success.second.execute(() -> success.first.accept(Task.this, v));
                }
            } catch (ExecutionException | InterruptedException e) {
                if (checkPair(error)) {
                    error.second.execute(() -> error.first.accept(Task.this, e));
                }
            }
        }
        if (checkPair(complete)) {
            complete.second.execute(() -> complete.first.accept(Task.this));
        }
    }

    private boolean checkPair(Pair<?, ?> pair) {
        return !(pair == null || pair.first == null || pair.second == null);
    }

    public boolean cancel() {
        return super.cancel(true);
    }

    public String getTag() {
        return tag;
    }

    public static final class Builder<V> {

        private Pair<BiConsumer<Task<V>, V>, Executor> success;
        private Pair<BiConsumer<Task<V>, Throwable>, Executor> error;
        private Pair<Consumer<Task<V>>, Executor> cancel;
        private Pair<Consumer<Task<V>>, Executor> complete;

        private final Callable<V> callable;
        private String tag;

        public Builder(Callable<V> callable) {
            this.callable = callable;
        }

        public Builder<V> tag(String tag) {
            this.tag = tag;
            return this;
        }

        public Builder<V> success(BiConsumer<Task<V>, V> consumer) {
            return success(consumer, Platform.getMainExecutor());
        }

        public Builder<V> success(BiConsumer<Task<V>, V> consumer, Executor executor) {
            success = Pair.create(consumer, executor);
            return this;
        }

        public Builder<V> error(BiConsumer<Task<V>, Throwable> consumer) {
            return error(consumer, Platform.getMainExecutor());
        }

        public Builder<V> error(BiConsumer<Task<V>, Throwable> consumer, Executor executor) {
            error = Pair.create(consumer, executor);
            return this;
        }

        public Builder<V> cancel(Consumer<Task<V>> consumer) {
            return cancel(consumer, Platform.getMainExecutor());
        }

        public Builder<V> cancel(Consumer<Task<V>> consumer, Executor executor) {
            cancel = Pair.create(consumer, executor);
            return this;
        }

        public Builder<V> complete(Consumer<Task<V>> consumer) {
            return complete(consumer, Platform.getMainExecutor());
        }

        public Builder<V> complete(Consumer<Task<V>> consumer, Executor executor) {
            complete = Pair.create(consumer, executor);
            return this;
        }

        public Task<V> build() {
            return new Task<>(this);
        }
    }

    private static class Pair<F, S> {

        public final F first;
        public final S second;

        public Pair(F first, S second) {
            this.first = first;
            this.second = second;
        }

        @Override
        public boolean equals(Object o) {
            if (!(o instanceof Pair)) {
                return false;
            }
            Pair<?, ?> p = (Pair<?, ?>) o;
            return ObjectUtils.equals(p.first, first) && ObjectUtils.equals(p.second, second);
        }

        @Override
        public int hashCode() {
            return (first == null ? 0 : first.hashCode()) ^ (second == null ? 0 : second.hashCode());
        }

        @Override
        public String toString() {
            return "Pair{" + first + " " + second + "}";
        }

        public static <A, B> Pair<A, B> create(A a, B b) {
            return new Pair<>(a, b);
        }
    }
}