package cn.keayuan.util;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

/**
 * File Utils
 * Read or write file
 * <ul>
 * <li>{@link #readFileToList(String, String)} read file to string list</li>
 * <li>{@link #writeFile(String, String, boolean)} write file from String</li>
 * <li>{@link #writeFile(String, String)} write file from String</li>
 * <li>{@link #writeFile(String, InputStream)} write file</li>
 * <li>{@link #writeFile(String, InputStream, boolean)} write file</li>
 * <li>{@link #writeFile(File, InputStream)} write file</li>
 * <li>{@link #writeFile(File, InputStream, boolean)} write file</li>
 * </ul>
 * Operate file
 * <ul>
 * <li>{@link #moveFile(File, File)} or {@link #moveFile(String, String)}</li>
 * <li>{@link #copyFile(String, String)}</li>
 * <li>{@link #getFileExtension(String)}</li>
 * <li>{@link #getFileName(String)}</li>
 * <li>{@link #getFileNameWithoutExtension(String)}</li>
 * <li>{@link #deleteFile(String)}</li>
 * <li>{@link #isFileExist(String)}</li>
 * <li>{@link #isFolderExist(String)}</li>
 * <li>{@link #makeDirs(String)}</li>
 * </ul>
 */
public class FileUtils {
    private static final String TAG = "FileUtils";

    private static final String FILE_EXTENSION_SEPARATOR = ".";

    private FileUtils() {
        throw new AssertionError();
    }

    public static String toKB(long size, int exactness) {
        double v = size / 1024d;
        return String.format("%." + exactness + "fKB", v);
    }

    public static String toMB(long size, int exactness) {
        double v = size / 1024d / 1024d;
        return String.format("%." + exactness + "fMB", v);
    }

    public static String toGB(long size, int exactness) {
        double v = size / 1024d / 1024d / 1024d;
        return String.format("%." + exactness + "fGB", v);
    }

    public static String toTB(long size, int exactness) {
        double v = size / 1024d / 1024d / 1024d / 1024d;
        return String.format("%." + exactness + "fTB", v);
    }

    public static byte[] readFile(String filePath) throws IOException {
        if (StringUtils.isEmpty(filePath)) {
            return null;
        }
        File file = new File(filePath);
        if (!file.isFile()) {
            return null;
        }

        try (InputStream is = new FileInputStream(file)) {
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            int ret;
            byte[] buff = new byte[1024 * 10];
            while ((ret = is.read(buff)) != -1) {
                os.write(buff, 0, ret);
            }
            return os.toByteArray();
        }
    }

    public static boolean writeFile(String filePath, byte[] data) {
        if (StringUtils.isEmpty(filePath)) {
            return false;
        }
        makeDirs(filePath);
        File file = new File(filePath);
        if (file.exists()) {
            file.delete();
        }
        try {
            if (!file.createNewFile()) {
                Platform.logger().e(TAG, "create new file failed");
                return false;
            }
            try (FileOutputStream os = new FileOutputStream(file)) {
                os.write(data);
                return true;
            }
        } catch (IOException e) {
            Platform.logger().e(TAG, "write file failed");
        }
        return false;
    }

    /**
     * write file
     *
     * @param filePath 目标文件
     * @param content  文件内容
     * @param append   is append, if true, write to the end of file, else clear content of file and write into it
     * @return return false if content is empty, true otherwise
     */
    public static boolean writeFile(String filePath, String content, boolean append) {
        if (StringUtils.isEmpty(content)) {
            Platform.logger().w(TAG, "write file content is empty");
            return false;
        }

        try (FileWriter fileWriter = new FileWriter(filePath, append)) {
            makeDirs(filePath);
            fileWriter.write(content);
            return true;
        } catch (IOException e) {
            Platform.logger().e(TAG, "write file failed", e);
        }
        return false;
    }

    /**
     * write file, the string will be written to the begin of the file
     *
     * @param filePath 文件路径
     * @param content  内容
     */
    public static boolean writeFile(String filePath, String content) {
        return writeFile(filePath, content, false);
    }

    /**
     * write file, the bytes will be written to the begin of the file
     *
     * @param filePath -
     * @param stream   -
     * @see #writeFile(String, InputStream, boolean)
     */
    public static boolean writeFile(String filePath, InputStream stream) {
        return writeFile(filePath, stream, false);
    }

    /**
     * write file
     *
     * @param filePath the file to be opened for writing.
     * @param stream   the input stream
     * @param append   if <code>true</code>, then bytes will be written to the end of the file rather than the beginning
     * @return return true
     */
    public static boolean writeFile(String filePath, InputStream stream, boolean append) {
        return writeFile(new File(filePath != null ? filePath : ""), stream, append);
    }

    /**
     * write file, the bytes will be written to the begin of the file
     *
     * @param file   -
     * @param stream -
     * @return -
     * @see #writeFile(File, InputStream, boolean)
     */
    public static boolean writeFile(File file, InputStream stream) {
        return writeFile(file, stream, false);
    }

    /**
     * write file
     *
     * @param file   the file to be opened for writing.
     * @param stream the input stream
     * @param append if <code>true</code>, then bytes will be written to the end of the file rather than the beginning
     * @return return true
     */
    public static boolean writeFile(File file, InputStream stream, boolean append) {
        if (!makeDirs(file.getAbsolutePath())) {
            return false;
        }
        try (OutputStream o = new FileOutputStream(file, append)) {
            byte[] data = new byte[1024 * 10];
            int length;
            while ((length = stream.read(data)) != -1) {
                o.write(data, 0, length);
            }
            o.flush();
            return true;
        } catch (IOException e) {
            e.printStackTrace();
        }
        return false;
    }

    /**
     * move file
     *
     * @param sourceFilePath 源文件
     * @param destFilePath   目标文件
     */
    public static boolean moveFile(String sourceFilePath, String destFilePath) {
        if (StringUtils.isEmpty(sourceFilePath) || StringUtils.isEmpty(destFilePath)) {
            return false;
        }
        return moveFile(new File(sourceFilePath), new File(destFilePath));
    }

    /**
     * move file
     *
     * @param srcFile  源文件
     * @param destFile 目标文件
     */
    public static boolean moveFile(File srcFile, File destFile) {
        boolean rename = srcFile.renameTo(destFile);
        if (!rename) {
            rename = copyFile(srcFile.getAbsolutePath(), destFile.getAbsolutePath());
            if (rename) {
                deleteFile(srcFile.getAbsolutePath());
            }
        }
        return rename;
    }

    /**
     * copy file
     *
     * @param sourceFilePath 源文件
     * @param destFilePath   目标文件
     */
    public static boolean copyFile(String sourceFilePath, String destFilePath) {
        try (InputStream inputStream = new FileInputStream(sourceFilePath)) {
            return writeFile(destFilePath, inputStream);
        } catch (IOException e) {
            Platform.logger().e(TAG, "copy file error " + sourceFilePath + " to " + destFilePath, e);
        }
        return false;
    }

    /**
     * read file to string list, a element of list is a line
     *
     * @param filePath    源文件
     * @param charsetName The name of a supported {@link java.nio.charset.Charset </code>charset<code>}
     * @return if file not exist, return null, else return content of file
     */
    public static List<String> readFileToList(String filePath, String charsetName) {
        if (filePath == null) {
            return null;
        }
        File file = new File(filePath);
        List<String> fileContent = new ArrayList<>();
        if (!file.isFile()) {
            return null;
        }

        try (BufferedReader reader =
                 new BufferedReader(new InputStreamReader(new FileInputStream(file), charsetName))) {
            String line;
            while ((line = reader.readLine()) != null) {
                fileContent.add(line);
            }
            return fileContent;
        } catch (IOException e) {
            Platform.logger().e(TAG, "read file failed " + filePath, e);
        }
        return null;
    }

    /**
     * get file name from path, include suffix
     * <p>
     * <pre>
     *      getFileName(null)               =   null
     *      getFileName("")                 =   ""
     *      getFileName("   ")              =   "   "
     *      getFileName("a.mp3")            =   "a.mp3"
     *      getFileName("a.b.rmvb")         =   "a.b.rmvb"
     *      getFileName("abc")              =   "abc"
     *      getFileName("c:\\")              =   ""
     *      getFileName("c:\\a")             =   "a"
     *      getFileName("c:\\a.b")           =   "a.b"
     *      getFileName("c:a.txt\\a")        =   "a"
     *      getFileName("/home/admin")      =   "admin"
     *      getFileName("/home/admin/a.txt/b.mp3")  =   "b.mp3"
     * </pre>
     *
     * @param filePath 文件路径名
     * @return file name from path, include suffix
     */
    public static String getFileName(String filePath) {
        if (StringUtils.isEmpty(filePath)) {
            return filePath;
        }

        int filePos = filePath.lastIndexOf(File.separator);
        return (filePos == -1) ? filePath : filePath.substring(filePos + 1);
    }

    /**
     * get file name from path, not include suffix
     * <p>
     * <pre>
     *      getFileNameWithoutExtension(null)               =   null
     *      getFileNameWithoutExtension("")                 =   ""
     *      getFileNameWithoutExtension("   ")              =   "   "
     *      getFileNameWithoutExtension("abc")              =   "abc"
     *      getFileNameWithoutExtension("a.mp3")            =   "a"
     *      getFileNameWithoutExtension("a.b.rmvb")         =   "a.b"
     *      getFileNameWithoutExtension("c:\\")              =   ""
     *      getFileNameWithoutExtension("c:\\a")             =   "a"
     *      getFileNameWithoutExtension("c:\\a.b")           =   "a"
     *      getFileNameWithoutExtension("c:a.txt\\a")        =   "a"
     *      getFileNameWithoutExtension("/home/admin")      =   "admin"
     *      getFileNameWithoutExtension("/home/admin/a.txt/b.mp3")  =   "b"
     * </pre>
     *
     * @param filePath 文件
     * @return file name from path, not include suffix
     */
    public static String getFileNameWithoutExtension(String filePath) {
        String fileName = getFileName(filePath);
        if (StringUtils.isEmpty(fileName)) {
            return fileName;
        }

        int exPos = fileName.lastIndexOf(FILE_EXTENSION_SEPARATOR);
        return exPos <= 0 ? fileName : fileName.substring(0, exPos);
    }

    /**
     * get folder name from path
     * <p>
     * <pre>
     *      getFolderName(null)               =   null
     *      getFolderName("")                 =   ""
     *      getFolderName("   ")              =   ""
     *      getFolderName("a.mp3")            =   ""
     *      getFolderName("a.b.rmvb")         =   ""
     *      getFolderName("abc")              =   ""
     *      getFolderName("c:\\")              =   "c:"
     *      getFolderName("c:\\a")             =   "c:"
     *      getFolderName("c:\\a.b")           =   "c:"
     *      getFolderName("c:a.txt\\a")        =   "c:a.txt"
     *      getFolderName("c:a\\b\\c\\d.txt")    =   "c:a\\b\\c"
     *      getFolderName("/home/admin")      =   "/home"
     *      getFolderName("/home/admin/a.txt/b.mp3")  =   "/home/admin/a.txt"
     * </pre>
     *
     * @param filePath 文件路径
     */
    public static String getFolderName(String filePath) {
        if (StringUtils.isEmpty(filePath)) {
            return filePath;
        }

        int filePos = filePath.lastIndexOf(File.separator);
        return (filePos == -1) ? "" : filePath.substring(0, filePos);
    }

    /**
     * get suffix of file from path
     * <p>
     * <pre>
     *      getFileExtension(null)               =   ""
     *      getFileExtension("")                 =   ""
     *      getFileExtension("   ")              =   "   "
     *      getFileExtension("a.mp3")            =   "mp3"
     *      getFileExtension("a.b.rmvb")         =   "rmvb"
     *      getFileExtension("abc")              =   ""
     *      getFileExtension("c:\\")              =   ""
     *      getFileExtension("c:\\a")             =   ""
     *      getFileExtension("c:\\a.b")           =   "b"
     *      getFileExtension("c:a.txt\\a")        =   ""
     *      getFileExtension("/home/admin")      =   ""
     *      getFileExtension("/home/admin/a.txt/b")  =   ""
     *      getFileExtension("/home/admin/a.txt/b.mp3")  =   "mp3"
     * </pre>
     *
     * @param filePath 文件路径
     */
    public static String getFileExtension(String filePath) {
        if (StringUtils.isBlank(filePath)) {
            return filePath;
        }

        int extPos = filePath.lastIndexOf(FILE_EXTENSION_SEPARATOR);
        int filePos = filePath.lastIndexOf(File.separator);
        if (extPos == -1 || filePos + 1 >= extPos) {
            return "";
        }
        return filePath.substring(extPos + 1);
    }

    /**
     * make folder
     *
     * @param filePath 路径
     */
    public static boolean makeDirs(String filePath) {
        String folderName = getFolderName(filePath);
        if (StringUtils.isEmpty(folderName)) {
            return false;
        }

        File folder = new File(folderName);
        return (folder.exists() && folder.isDirectory()) || folder.mkdirs();
    }

    /**
     * Indicates if this file represents a file on the underlying file system.
     *
     * @param filePath 文件路径
     */
    public static boolean isFileExist(String filePath) {
        if (StringUtils.isBlank(filePath)) {
            return false;
        }

        File file = new File(filePath);
        return file.exists() && file.isFile();
    }

    /**
     * Indicates if this file represents a directory on the underlying file system.
     *
     * @param directoryPath 文件路径
     */
    public static boolean isFolderExist(String directoryPath) {
        if (StringUtils.isBlank(directoryPath)) {
            return false;
        }

        File dire = new File(directoryPath);
        return dire.exists() && dire.isDirectory();
    }

    /**
     * delete file or directory
     * <ul>
     * <li>if path is null or empty, return true</li>
     * <li>if path not exist, return true</li>
     * <li>if path exist, delete recursion. return true</li>
     * </ul>
     *
     * @param path -
     * @return -
     */
    public static boolean deleteFile(String path) {
        if (StringUtils.isBlank(path)) {
            return true;
        }

        File file = new File(path);
        if (!file.exists()) {
            return true;
        }
        if (file.isFile()) {
            return file.delete();
        }
        if (!file.isDirectory()) {
            return false;
        }
        File[] list = file.listFiles();
        if (list != null) {
            for (File f : list) {
                if (f.isFile()) {
                    f.delete();
                } else if (f.isDirectory()) {
                    deleteFile(f.getAbsolutePath());
                }
            }
        }
        return file.delete();
    }

    public static long getSize(String path) {
        if (StringUtils.isBlank(path)) return 0;
        return getSize(new File(path));
    }

    public static long getSize(File file) {
        if (file == null || !file.exists() || !file.canRead()) return 0;
        if (file.isFile()) return file.length();
        long len = 0;
        if (file.isDirectory()) {
            File[] files = file.listFiles();
            if (files != null) {
                for (File f : files) {
                    len += getSize(f);
                }
            }
        }
        return len;
    }
}