package cn.keayuan.util.log;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

public class DefaultLog implements ILog {
    private static final String[] NAME = {"", "", "V", "D", "I", "W", "E", "A"};

    private static final boolean isAndroid;

    private String tag;
    private int priority = Logger.VERBOSE;
    private StringBuilder builder;
    private PLog pLog;

    private final SimpleDateFormat dateFormat = new SimpleDateFormat("MM-dd HH:mm:ss.SSS", Locale.getDefault());

    static {
        boolean hasLog;
        try {
            hasLog = Class.forName("android.os.Build$VERSION").getField("SDK_INT").getInt(null) > 0;
        } catch (ClassNotFoundException | NoSuchFieldException | IllegalAccessException e) {
            hasLog = false;
        }
        isAndroid = hasLog;
        System.out.println("isAndroid " + isAndroid);
    }

    public DefaultLog(String tag) {
        this.tag = tag;
    }

    public void setTag(String tag) {
        this.tag = tag;
    }

    public void setPriority(int priority) {
        this.priority = priority;
    }

    public void setLog(PLog log) {
        pLog = log;
    }

    @Override
    public String getTag() {
        return tag;
    }

    @Override
    public boolean isLoggable(int priority) {
        return priority >= this.priority;
    }

    @Override
    public synchronized void log(int priority, String tag, String msg, Object... obj) {
        if (priority > Logger.ASSERT || priority < Logger.VERBOSE) {
            throw new IllegalArgumentException("priority error " + priority);
        }
        if (pLog != null) {
            pLog.log(priority, tag, msg, obj);
            return;
        }
        if (builder == null) {
            builder = new StringBuilder();
        }
        builder.setLength(0);
        for (Object o : obj) {
            if (o instanceof Throwable) {
                builder.append(getStackTraceString((Throwable) o));
            } else {
                builder.append(o);
            }
            builder.append(' ');
        }
        if (isAndroid) {
            android.util.Log.println(priority, tag, msg + " " + builder.toString());
        } else {
            System.out.println(dateFormat.format(new Date()) + " " + NAME[priority] + "/" + tag + ": " + msg + " " + builder.toString());
        }
        if (builder.length() > 5 * 1024) {
            builder = null;
        }
    }

    public static String getStackTraceString(Throwable tr) {
        if (tr == null) {
            return "";
        }
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        tr.printStackTrace(pw);
        pw.flush();
        return sw.toString();
    }
}
