package cn.keayuan.util.log.disk;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;

import cn.keayuan.util.log.PrintStrategy;

/**
 * Abstract class that takes care of background threading the file log operation on Android.
 * implementing classes are free to directly perform I/O operations there.
 *
 * @author ex-keayuan001
 */
public class DiskLogStrategy implements PrintStrategy, Runnable {

    private int mMaxFileSize = 1024 * 1024;// 1M
    private String fileName = "log";

    private final String mPath;
    private final BlockingQueue<String> queue = new LinkedBlockingQueue<>();

    private Executor executor;
    private RandomAccessFile out;
    private volatile boolean isRunning;

    public DiskLogStrategy(String path) {
        this(path, 0);
    }

    public DiskLogStrategy(String path, int maxFile) {
        mPath = path;
        if (maxFile > mMaxFileSize) {
            mMaxFileSize = maxFile;
        }
    }

    public DiskLogStrategy fileName(String name) {
        fileName = name == null ? "name" : name;
        return this;
    }

    @Override
    public boolean print(int level, String tag, String message) {
        queue.offer(tag + ":" + message);
        if (executor == null) {
            synchronized (DiskLogStrategy.class) {
                executor = Executors.newSingleThreadExecutor();
            }
            if (!isRunning) {
                executor.execute(this);
            }
        }
        return true;
    }

    @Override
    public void run() {
        isRunning = true;
        while (isRunning) {
            try {
                String item = queue.take();
                if (out == null) {
                    out = getFileOutputStream(mPath, this.fileName);
                }
                out.write(item.getBytes());
                if (out.length() > mMaxFileSize) {
                    out.close();
                    out = null;
                }
            } catch (InterruptedException | FileNotFoundException e) {
                e.printStackTrace();
                try { if (out != null) out.close(); } catch (IOException ignore) { }
                isRunning = false;
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    private RandomAccessFile getFileOutputStream(String folderName, String fileName) throws FileNotFoundException {
        File folder = new File(folderName);
        if (!folder.exists()) {
            folder.mkdirs();
        }

        File newFile = new File(folder, fileName + ".log");
        if (newFile.exists()) {
            long length = newFile.length();
            if (length > mMaxFileSize) {
                File temp;
                int newFileCount = 0;
                do {
                    temp = new File(folder, String.format("%s_%s.log", fileName, newFileCount++));
                } while (temp.exists());
                newFile.renameTo(temp);
                newFile = new File(folder, fileName + ".log");
            }
        }

        return new RandomAccessFile(newFile, "rw");
    }
}
