package cn.keayuan.util.log;

import java.util.HashSet;
import java.util.Set;

import cn.keayuan.util.log.console.ConsolePrinter;
import cn.keayuan.util.log.disk.DiskPrinter;
import cn.keayuan.util.log.logcat.LogcatPrinter;

/**
 * Created by keayuan on 2021.12.28.
 *
 * @author keayuan
 */
public final class LoggerFactory {
    private static final String TAG = "LoggerFactory";

    private static boolean logcat;
    private Printer defaultPrint;

    private Format objFormat;

    final Set<Printer> logPrinters = new HashSet<>();

    String TAG_PREFIX = "global";
    LogFormat formatter;
    int level = Logger.VERBOSE;

    private LoggerFactory() {
        objFormat = new ObjectFormatter(testClass("org.json.JSONObject") ? new JsonFormatter() : null, testClass(
            "javax.xml.transform.Source") ? new XmlFormatter() : null);
        formatter = new DefaultFormat(objFormat);
        try {
            Class.forName("android.util.Log");
            Class<?> cls = Class.forName("android.os.Build$VERSION");
            logcat = cls.getField("SDK_INT").getInt(cls) > 0;
        } catch (ClassNotFoundException ignored) {
            System.out.println(TAG + ":环境不支持 logcat, 没有找到android.util.Log");
        } catch (NoSuchFieldException ignored) {
            System.out.println(TAG + ":环境不支持 logcat, 没有找到android.os.Build$VERSION.SDK_INI");
        } catch (IllegalAccessException ignored) {
            System.out.println(TAG + ":环境不支持 logcat, 没有找到android.os.Build$VERSION.SDK_INI");
        }
        defaultPrint = logcat ? LogcatPrinter.getInstance(null) : ConsolePrinter.getInstance(null);
        logPrinters.add(defaultPrint);
    }

    /**
     * 设置全局tag
     *
     * @param tagPrefix tag前缀
     */
    public void tag(String tagPrefix) {
        TAG_PREFIX = tagPrefix;
    }

    /**
     * 添加打印者
     * <ul>
     * <li>Different {@link LogcatPrinter}</li>
     * <li>Different {@link DiskPrinter}</li>
     * <li>Different {@link ConsolePrinter}</li>
     * </ul>
     *
     * @param printers 打印者
     * @see LogcatPrinter
     * @see DiskPrinter
     * @see ConsolePrinter
     */
    public void addPinter(Printer... printers) {
        for (Printer printer : printers) {
            if (printer != null) {
                if (!logcat && printer instanceof LogcatPrinter) {
                    throw new IllegalArgumentException("环境不支持 cn.sskbskdrin.log.logcat, 没有找到android.util.Log");
                }
                if (defaultPrint != null) {
                    logPrinters.remove(defaultPrint);
                    defaultPrint = null;
                }
                logPrinters.add(printer);
            }
        }
    }

    /**
     * 清除所有打印者
     */
    public void clearPrinters() {
        logPrinters.clear();
    }

    public void setLogLevel(int level) {
        this.level = level;
    }

    /**
     * 设置内容格式化方法
     *
     * @param formatter 格式化函数
     */
    public final void setFormatter(LogFormat formatter) {
        if (formatter == null) {
            throw new IllegalArgumentException("formatter Can not be null");
        }
        this.formatter = formatter;
    }

    public final void setObjectFormat(Format formatter) {
        if (formatter == null) {
            throw new IllegalArgumentException("formatter Can not be null");
        }
        this.objFormat = formatter;
    }

    public final Format getObjectFormat() {
        return this.objFormat;
    }

    public ILog getLogger(Class<?> cls) {
        return new LogImpl(cls.getSimpleName(), this);
    }

    public ILog getLogger(String tag) {
        return new LogImpl(tag, this);
    }

    private static final LoggerFactory defaultFactory = new LoggerFactory();

    public static LoggerFactory createLoggerFactory() {
        return new LoggerFactory();
    }

    public static LoggerFactory defaultFactory() {
        return defaultFactory;
    }

    private static boolean testClass(String name) {
        try {
            Class.forName(name);
            return true;
        } catch (ClassNotFoundException ignore) {
        }
        return false;
    }
}
