package cn.keayuan.util.log;

import java.util.HashSet;
import java.util.Set;

import cn.keayuan.util.log.format.DefaultFormat;
import cn.keayuan.util.log.format.Format;
import cn.keayuan.util.log.format.JsonFormatter;
import cn.keayuan.util.log.format.LogFormat;
import cn.keayuan.util.log.format.ObjectFormatter;
import cn.keayuan.util.log.format.XmlFormatter;
import cn.keayuan.util.log.printer.ConsolePrinter;
import cn.keayuan.util.log.printer.DiskPrinter;
import cn.keayuan.util.log.printer.LogcatPrinter;

/**
 * Created by keayuan on 2021.12.28.
 *
 * @author keayuan
 */
public final class LoggerFactory implements ILogProvider<ILog> {

    final Set<PLog> logPrinters = new HashSet<>();

    String TAG_PREFIX = "global";
    LogFormat formatter;
    int priority = Logger.VERBOSE;

    private LoggerFactory(String tag) {
        tag(tag);
        Format objFormat = new ObjectFormatter(testClass("org.json.JSONObject") ? new JsonFormatter() : null,
            testClass("javax.xml.transform.Source") ? new XmlFormatter() : null);
        formatter = new DefaultFormat(objFormat);
        logPrinters.add(DefaultLog.isAndroid ? new LogcatPrinter() : new ConsolePrinter(true));
    }

    /**
     * 设置全局tag
     *
     * @param tagPrefix tag前缀
     */
    public LoggerFactory tag(String tagPrefix) {
        TAG_PREFIX = tagPrefix;
        return this;
    }

    /**
     * 添加打印者
     * <ul>
     * <li>Different {@link LogcatPrinter}</li>
     * <li>Different {@link DiskPrinter}</li>
     * <li>Different {@link ConsolePrinter}</li>
     * </ul>
     *
     * @param printers 打印者
     * @see LogcatPrinter
     * @see DiskPrinter
     * @see ConsolePrinter
     */
    public LoggerFactory addPinter(PLog... printers) {
        for (PLog printer : printers) {
            if (printer != null) {
                if (!DefaultLog.isAndroid && printer instanceof LogcatPrinter) {
                    throw new IllegalArgumentException("not support LogcatPrinter");
                }
                logPrinters.add(printer);
            }
        }
        return this;
    }

    /**
     * 清除所有打印者
     */
    public LoggerFactory clearPrinters() {
        logPrinters.clear();
        return this;
    }

    public LoggerFactory setLogPriority(int priority) {
        this.priority = priority;
        return this;
    }

    /**
     * 设置内容格式化方法
     *
     * @param formatter 格式化函数
     */
    public LoggerFactory setFormatter(LogFormat formatter) {
        if (formatter == null) {
            throw new IllegalArgumentException("formatter Can not be null");
        }
        this.formatter = formatter;
        return this;
    }

    public ILog getLogger(String tag) {
        return new LogImpl(tag, this);
    }

    public static LoggerFactory createFactory(String tagPrefix) {
        return new LoggerFactory(tagPrefix);
    }

    private static final LoggerFactory defaultFactory = new LoggerFactory("default");

    public static LoggerFactory defaultFactory() {
        return defaultFactory;
    }

    private static boolean testClass(String name) {
        try {
            Class.forName(name);
            return true;
        } catch (ClassNotFoundException ignore) {
        }
        return false;
    }
}
