package cn.keayuan.util.log.printer;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;

import cn.keayuan.util.log.Logger;
import cn.keayuan.util.log.Printer;

public class DiskPrinter extends Printer implements Runnable {

    private static final String[] names = {"UNKNOWN", "UNKNOWN", "VERBOSE", "DEBUG", "INFO", "WARN", "ERROR", "ASSERT"};

    private static final String SEPARATOR = " ";

    private final Date date = new Date();
    private final SimpleDateFormat dateFormat = new SimpleDateFormat("MM-dd HH:mm:ss.SSS", Locale.US);
    private int mMaxFileSize = 1024 * 1024;// 1M
    private final String fileName;

    private final String mPath;
    private final BlockingQueue<String> queue = new LinkedBlockingQueue<>();

    private Executor executor;
    private RandomAccessFile out;
    private volatile boolean isRunning;

    public DiskPrinter(String filePath, String fileName) {
        this(filePath, fileName, 8 * 1024 * 1024);
    }

    public DiskPrinter(String filePath, String fileName, int maxFileSize) {
        mPath = filePath;
        if (maxFileSize > mMaxFileSize) {
            mMaxFileSize = maxFileSize;
        }
        this.fileName = fileName == null ? "log" : fileName;
    }

    @Override
    public String formatTag(int priority, String tag) {
        if (tag == null) {
            tag = "";
        }
        date.setTime(System.currentTimeMillis());
        return dateFormat.format(date) + SEPARATOR + Thread.currentThread()
            .getId() + SEPARATOR + logLevel(priority) + SEPARATOR + tag;
    }

    @Override
    public String format(String msg) {
        return msg + NEW_LINE;
    }

    private static String logLevel(int value) {
        if (value < 0 || value > Logger.ASSERT) return names[0];
        return names[value];
    }

    @Override
    public void print(int priority, String tag, String message) {
        queue.offer(tag + ":" + message);
        if (executor == null) {
            synchronized (this) {
                executor = Executors.newSingleThreadExecutor();
            }
            if (!isRunning) {
                executor.execute(this);
            }
        }
    }

    @Override
    public void run() {
        isRunning = true;
        while (isRunning) {
            try {
                String item = queue.take();
                if (out == null) {
                    out = getFileOutputStream(mPath, this.fileName, mMaxFileSize);
                }
                out.write(item.getBytes());
                if (out.length() > mMaxFileSize) {
                    out.close();
                    out = null;
                }
            } catch (InterruptedException | FileNotFoundException e) {
                e.printStackTrace();
                try {if (out != null) out.close();} catch (IOException ignore) {}
                isRunning = false;
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    private static RandomAccessFile getFileOutputStream(String folderName, String fileName, int maxSize) throws FileNotFoundException {
        File folder = new File(folderName);
        if (!folder.exists()) {
            folder.mkdirs();
        }

        File newFile = new File(folder, fileName + ".log");
        if (newFile.exists()) {
            long length = newFile.length();
            if (length > maxSize) {
                File temp;
                int newFileCount = 0;
                do {
                    temp = new File(folder, String.format("%s_%s.log", fileName, newFileCount++));
                } while (temp.exists());
                newFile.renameTo(temp);
                newFile = new File(folder, fileName + ".log");
            }
        }

        return new RandomAccessFile(newFile, "rw");
    }
}
