package cn.k7g.alloy.ioc.processor;

import cn.k7g.alloy.annotation.Label;
import cn.k7g.alloy.autoconfiguration.EnableOptions;
import cn.k7g.alloy.expose.WebExceptionResponseHandler;
import com.fasterxml.jackson.databind.exc.InvalidFormatException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.hibernate.validator.internal.engine.ConstraintViolationImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.web.servlet.MultipartProperties;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.multipart.MaxUploadSizeExceededException;

import javax.security.auth.login.AccountExpiredException;
import java.lang.reflect.Field;
import java.nio.file.AccessDeniedException;
import java.util.stream.Collectors;

@Slf4j
@RestControllerAdvice
@ConditionalOnBean(EnableOptions.EnhanceExceptionMessage.class)
public class EnhanceExceptionResponse {

    @Autowired
    private WebExceptionResponseHandler handler;
    @Autowired
    private MultipartProperties multipartProperties;

//    @Autowired
//    private VerifyMessageFactory verifyMessageFactory;


//    @ExceptionHandler(NoHandlerFoundException.class)
//    public Object handlerNoFoundException(NoHandlerFoundException e) {
//        return handler.handle("404 路径不存在，请检查路径是否正确", e);
//    }

    @ExceptionHandler(AccessDeniedException.class)
    public Object handleAuthorizationException(AccessDeniedException e) {
        return handler.handle("没有权限，请联系管理员授权", e);
    }

    @ExceptionHandler(AccountExpiredException.class)
    public Object handleAccountExpiredException(AccountExpiredException e) {
        return handler.handle("登录失效了", e);
    }

//    @ExceptionHandler(UsernameNotFoundException.class)
//    public Object handleUsernameNotFoundException(UsernameNotFoundException e) {
//        return handler.handle("用户不存在", e);
//    }

    @ExceptionHandler(Exception.class)
    public Object handleException(Exception e) {
        return handler.handle("系统错误", e);
    }

    @ExceptionHandler(MethodArgumentTypeMismatchException.class)
    public Object handleException(MethodArgumentTypeMismatchException e) {
        String msg = String.format("参数类型错误: [%s] - %s  详细描述： %s", e.getName(),
                e.getParameter().getParameterType().getSimpleName(),
                e.getMessage()) ;
        return handler.handle(msg, e);
    }

    @ExceptionHandler(BindException.class)
    public Object validatedBindException(BindException e) {
        StringBuilder sb = new StringBuilder();
        for (ObjectError error : e.getAllErrors()) {
            try {
                if (error.contains( ConstraintViolationImpl.class)) {
                    Field sourceField = FieldUtils.getField(error.getClass(), "source", true);
                    ConstraintViolationImpl source = (ConstraintViolationImpl) sourceField.get(error);

                    Field nameField = FieldUtils.getField(error.getClass(), "field", true);
                    String fieldName = (String) nameField.get(error);
                    Field errField = FieldUtils.getField(source.getRootBeanClass(), fieldName, true);

                    Label label = errField.getAnnotation(Label.class);
                    if (label != null && label.enhanceVerifyMessage()) {
                        sb.append(label.value() + " " + error.getDefaultMessage()).append("\r\n");
                    }
                }
            } catch (Exception ex) {
                log.debug("增强错误提示出现错误：", ex);
                sb.append(error.getDefaultMessage()).append("\r\n");
            }
        }
        return handler.handle(sb.toString(), e);
    }

    @ExceptionHandler(MethodArgumentNotValidException.class)
    public Object validExceptionHandler(MethodArgumentNotValidException e) {
        String message = e.getBindingResult().getFieldError().getDefaultMessage();
        return handler.handle(message, e);
    }


    @ExceptionHandler(MaxUploadSizeExceededException.class)
    public Object maxUploadSizeExceededException(MaxUploadSizeExceededException e) {
        long min = Math.min(multipartProperties.getMaxFileSize().toBytes(), multipartProperties.getMaxRequestSize().toBytes());
        return handler.handle("文件最大支持：" + FileUtils.byteCountToDisplaySize(min), e);
    }

    @ExceptionHandler(HttpMessageNotReadableException.class)
    public Object invalidFormatException(HttpMessageNotReadableException e) {
        if (e.getCause() instanceof InvalidFormatException) {
            InvalidFormatException cause = (InvalidFormatException) e.getCause();
            String fieldPath = cause.getPath().stream().map(o -> o.getFieldName()).filter(o -> o != null).collect(Collectors.joining("."));
            return handler.handle(String.format("%s 是 %s 类型，无法输入值为：%s", fieldPath, cause.getTargetType().getSimpleName(), cause.getValue().toString()), e);
        }
        return handler.handle(e.getMessage(), e);
    }
}
