package cn.jrack.metadata.driver;
import cn.jrack.core.util.LogUtil;
import cn.jrack.metadata.ast.Clickhouse20CreateTableStatement;
import cn.jrack.metadata.convert.ClickHouseTypeConvert;
import cn.jrack.metadata.convert.ITypeConvert;
import cn.jrack.metadata.parser.Clickhouse20StatementParser;
import cn.jrack.metadata.pojo.Table;
import cn.jrack.metadata.query.ClickHouseQuery;
import cn.jrack.metadata.query.IDBQuery;
import cn.jrack.metadata.result.SqlExplainResult;
import com.alibaba.druid.sql.ast.SQLStatement;
import com.alibaba.druid.sql.ast.statement.SQLDropTableStatement;
import com.alibaba.druid.sql.ast.statement.SQLExprTableSource;
import com.alibaba.druid.sql.ast.statement.SQLSelectStatement;
import com.alibaba.druid.sql.parser.ParserException;
import com.alibaba.druid.sql.parser.Token;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * ClickHouseDriver
 *
 * @author ZhaoYang
 **/
public class ClickHouseDriver extends AbstractJdbcDriver {

    @Override
    String getDriverClass() {
        return "ru.yandex.clickhouse.ClickHouseDriver";
    }

    @Override
    public IDBQuery getDBQuery() {
        return new ClickHouseQuery();
    }

    @Override
    public ITypeConvert getTypeConvert() {
        return new ClickHouseTypeConvert();
    }

    @Override
    public String getType() {
        return "ClickHouse";
    }

    @Override
    public String getName() {
        return "ClickHouse OLAP 数据库";
    }

    @Override
    public List<SqlExplainResult> explain(String sql) {
        String initialSql = sql;
        List<SqlExplainResult> sqlExplainResults = new ArrayList<>();
        StringBuilder explain = new StringBuilder();
        PreparedStatement preparedStatement = null;
        ResultSet results = null;
        String current = null;
        try {
            sql = sql.replaceAll("(?i)if exists", "");
            Clickhouse20StatementParser parser = new Clickhouse20StatementParser(sql);
            List<SQLStatement> stmtList = new ArrayList<>();
            parser.parseStatementList(stmtList, -1, null);
            if (parser.getLexer().token() != Token.EOF) {
                throw new ParserException("syntax error : " + sql);
            }
            for (SQLStatement item : stmtList) {
                current = item.toString();
                String type = item.getClass().getSimpleName();
                if (!(item instanceof SQLSelectStatement)) {
                    if (item instanceof Clickhouse20CreateTableStatement) {
                        Matcher m = Pattern.compile(",\\s*\\)").matcher(sql);
                        if (m.find()) {
                            sqlExplainResults.add(SqlExplainResult.fail(sql, "No comma can be added to the last field of Table! "));
                            break;
                        }
                        sqlExplainResults.add(checkCreateTable((Clickhouse20CreateTableStatement) item));
                    } else if (item instanceof SQLDropTableStatement) {
                        sqlExplainResults.add(checkDropTable((SQLDropTableStatement) item, initialSql));
                    } else {
                        sqlExplainResults.add(SqlExplainResult.success(type, current, explain.toString()));
                    }
                    continue;
                }
                preparedStatement =  conn.get().prepareStatement("explain " + current);
                results = preparedStatement.executeQuery();
                while (results.next()) {
                    explain.append(getTypeConvert().convertValue(results, "explain", "string") + "\r\n");
                }
                sqlExplainResults.add(SqlExplainResult.success(type, current, explain.toString()));
            }
        } catch (Exception e) {
            sqlExplainResults.add(SqlExplainResult.fail(current, LogUtil.getError(e)));
        } finally {
            close(preparedStatement, results);
            return sqlExplainResults;
        }
    }

    private SqlExplainResult checkCreateTable(Clickhouse20CreateTableStatement sqlStatement) {
        if (existTable(Table.build(sqlStatement.getTableName()))) {
            if (sqlStatement.isIfNotExists()) {
                return SqlExplainResult.success(sqlStatement.getClass().getSimpleName(), sqlStatement.toString(), null);
            } else {
                String schema = null == sqlStatement.getSchema() ? "" : sqlStatement.getSchema() + ".";
                return SqlExplainResult.fail(sqlStatement.toString(), "Table " + schema + sqlStatement.getTableName() + " already exists.");
            }
        } else {
            return SqlExplainResult.success(sqlStatement.getClass().getSimpleName(), sqlStatement.toString(), null);
        }
    }

    private SqlExplainResult checkDropTable(SQLDropTableStatement sqlStatement, String sql) {
        SQLExprTableSource sqlExprTableSource = sqlStatement.getTableSources().get(0);
        if (!existTable(Table.build(sqlExprTableSource.getTableName()))) {
            if (Pattern.compile("(?i)if exists").matcher(sql).find()) {
                return SqlExplainResult.success(sqlStatement.getClass().getSimpleName(), sqlStatement.toString(), null);
            } else {
                return SqlExplainResult.fail(sqlStatement.toString(), "Table " + sqlExprTableSource.getSchema() + "." + sqlExprTableSource.getTableName() + " not exists.");
            }
        } else {
            return SqlExplainResult.success(sqlStatement.getClass().getSimpleName(), sqlStatement.toString(), null);
        }
    }

    @Override
    public Map<String, String> getFlinkColumnTypeConversion() {
        return new HashMap<>();
    }
}
