package cn.jrack.action.core.aop;

import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.StrUtil;
import cn.jrack.action.core.annotations.ActionLog;
import cn.jrack.action.core.entity.ActionLogEntity;
import cn.jrack.action.core.enums.ActionTypeEnum;
import cn.jrack.action.core.service.ActionLogFrameworkService;
import cn.jrack.core.util.json.JsonUtil;
import cn.jrack.core.util.servlet.ServletUtil;

import cn.jrack.springboot.web.core.response.Result;
import cn.jrack.springboot.web.core.util.WebFrameworkUtils;
import com.google.common.collect.Maps;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.annotation.Annotation;
import java.lang.reflect.Array;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.IntStream;

import static cn.jrack.core.exception.enums.GlobalErrorCodeConstants.INTERNAL_SERVER_ERROR;
import static cn.jrack.core.exception.enums.GlobalErrorCodeConstants.SUCCESS;


/**
 * 拦截使用 @OperateLog 注解，如果满足条件，则生成操作日志。
 * 满足如下任一条件，则会进行记录：
 * 1. 使用 @ApiOperation + 非 @GetMapping
 * 2. 使用 @OperateLog 注解
 * <p>
 * 但是，如果声明 @OperateLog 注解时，将 enable 属性设置为 false 时，强制不记录。
 *
 * @author ZhaoYang
 */
@Aspect
@Slf4j
public class ActionLogAspect {

    /**
     * 用于记录操作内容的上下文
     */
    private static final ThreadLocal<String> CONTENT = new ThreadLocal<>();
    /**
     * 用于记录拓展字段的上下文
     */
    private static final ThreadLocal<Map<String, Object>> EXTS = new ThreadLocal<>();

    @Resource
    private ActionLogFrameworkService actionLogFrameworkService;

    @Around("@annotation(apiOperation)")
    public Object around(ProceedingJoinPoint joinPoint, ApiOperation apiOperation) throws Throwable {
        // 可能也添加了 @ApiOperation 注解
        ActionLog actionLog = getMethodAnnotation(joinPoint,
                ActionLog.class);
        return around0(joinPoint, actionLog, apiOperation);
    }

    @Around("!@annotation(io.swagger.annotations.ApiOperation) && @annotation(actionLog)")
    // 兼容处理，只添加 @OperateLog 注解的情况
    public Object around(ProceedingJoinPoint joinPoint,
                         ActionLog actionLog) throws Throwable {
        return around0(joinPoint, actionLog, null);
    }

    private Object around0(ProceedingJoinPoint joinPoint,
                           ActionLog actionLog,
                           ApiOperation apiOperation) throws Throwable {
        // 记录开始时间
        Date startTime = new Date();
        try {
            // 执行原有方法
            Object result = joinPoint.proceed();
            // 记录正常执行时的操作日志
            this.log(joinPoint, actionLog, apiOperation, startTime, result, null);
            return result;
        } catch (Throwable exception) {
            this.log(joinPoint, actionLog, apiOperation, startTime, null, exception);
            throw exception;
        } finally {
            clearThreadLocal();
        }
    }

    public static void setContent(String content) {
        CONTENT.set(content);
    }

    public static void addExt(String key, Object value) {
        if (EXTS.get() == null) {
            EXTS.set(new HashMap<>());
        }
        EXTS.get().put(key, value);
    }

    private static void clearThreadLocal() {
        CONTENT.remove();
        EXTS.remove();
    }

    private void log(ProceedingJoinPoint joinPoint,
                     ActionLog actionLog,
                     ApiOperation apiOperation,
                     Date startTime, Object result, Throwable exception) {
        try {
            // 判断不记录的情况
            if (!isLogEnable(joinPoint, actionLog)) {
                return;
            }
            // 真正记录操作日志
            this.log0(joinPoint, actionLog, apiOperation, startTime, result, exception);
        } catch (Throwable ex) {
            log.error("[log][记录操作日志时，发生异常，其中参数是 joinPoint({}) operateLog({}) apiOperation({}) result({}) exception({}) ]",
                    joinPoint, actionLog, apiOperation, result, exception, ex);
        }
    }

    private void log0(ProceedingJoinPoint joinPoint,
                      ActionLog actionLog,
                      ApiOperation apiOperation,
                      Date startTime, Object result, Throwable exception) {
        ActionLogEntity actionLogEntity = new ActionLogEntity();
        // 补全通用字段
        //operateLogObj.setTraceId(TracerUtils.getTraceId());
        actionLogEntity.setStartTime(startTime);
        String userId = "";
        try {
            userId = WebFrameworkUtils.getLoginUserId().toString();
        } catch (Exception ex)
        {

        }
        actionLogEntity.setUserId(userId);
        // 补充用户信息
        fillUserFields(actionLogEntity);
        // 补全模块信息
        fillModuleFields(actionLogEntity, joinPoint, actionLog, apiOperation);
        // 补全请求信息
        fillRequestFields(actionLogEntity);
        // 补全方法信息
        fillMethodFields(actionLogEntity, joinPoint, actionLog, startTime, result, exception);

        // 异步记录日志
        actionLogFrameworkService.createOperateLog(actionLogEntity);
    }

    private static void fillUserFields(ActionLogEntity actionLogEntity) {
//        operateLogObj.setUserId(WebFrameworkUtils.getLoginUserId());
//        operateLogObj.setUserType(WebFrameworkUtils.getLoginUserType());
    }

    private static void fillModuleFields(ActionLogEntity actionLogEntity,
                                         ProceedingJoinPoint joinPoint,
                                         ActionLog operateLog,
                                         ApiOperation apiOperation) {
        // module 属性
        if (operateLog != null) {
            actionLogEntity.setModule(operateLog.module());
        }
        if (StrUtil.isEmpty(actionLogEntity.getModule())) {
            Api api = getClassAnnotation(joinPoint, Api.class);
            if (api != null) {
                // 优先读取 @API 的 name 属性
                if (StrUtil.isNotEmpty(api.value())) {
                    actionLogEntity.setModule(api.value());
                }
                // 没有的话，读取 @API 的 tags 属性
                if (StrUtil.isEmpty(actionLogEntity.getModule()) && ArrayUtil.isNotEmpty(api.tags())) {
                    actionLogEntity.setModule(api.tags()[0]);
                }
            }
        }
        // name 属性
        if (operateLog != null) {
            actionLogEntity.setName(operateLog.name());
        }
        if (StrUtil.isEmpty(actionLogEntity.getName()) && apiOperation != null) {
            actionLogEntity.setName(apiOperation.value());
        }
        // type 属性
        if (operateLog != null && ArrayUtil.isNotEmpty(operateLog.type())) {
            actionLogEntity.setType(operateLog.type()[0].getType());
        }
        if (actionLogEntity.getType() == null) {
            RequestMethod requestMethod = obtainFirstMatchRequestMethod(obtainRequestMethod(joinPoint));
            ActionTypeEnum actionTypeEnum = convertOperateLogType(requestMethod);
            actionLogEntity.setType(actionTypeEnum != null ? actionTypeEnum.getType() : null);
        }
        // content 和 exts 属性
        actionLogEntity.setContent(CONTENT.get());
        actionLogEntity.setExts(EXTS.get());
    }

    private static void fillRequestFields(ActionLogEntity operateLogObj) {
        // 获得 Request 对象
        HttpServletRequest request = ServletUtil.getRequest();
        if (request == null) {
            return;
        }
        // 补全请求信息
        operateLogObj.setRequestMethod(request.getMethod());
        operateLogObj.setRequestUrl(request.getRequestURI());
        operateLogObj.setUserIp(ServletUtil.getClientIP(request));
        operateLogObj.setUserAgent(ServletUtil.getUserAgent(request));
    }

    private static void fillMethodFields(ActionLogEntity actionLogEntity,
                                         ProceedingJoinPoint joinPoint,
                                         ActionLog operateLog,
                                         Date startTime, Object result, Throwable exception) {
        MethodSignature methodSignature = (MethodSignature) joinPoint.getSignature();
        actionLogEntity.setJavaMethod(methodSignature.toString());
        if (operateLog == null || operateLog.logArgs()) {
            actionLogEntity.setJavaMethodArgs(obtainMethodArgs(joinPoint));
        }
        if (operateLog == null || operateLog.logResultData()) {
            actionLogEntity.setResultData(obtainResultData(result));
        }
        actionLogEntity.setDuration((int) (System.currentTimeMillis() - startTime.getTime()));
        // （正常）处理 resultCode 和 resultMsg 字段
        if (result instanceof Result) {
            Result<?> commonResult = (Result<?>) result;
            actionLogEntity.setResultCode(commonResult.getCode());
            actionLogEntity.setResultMsg(commonResult.getMsg());
        } else {
            actionLogEntity.setResultCode(SUCCESS.getCode());
        }
        // （异常）处理 resultCode 和 resultMsg 字段
        if (exception != null) {
            actionLogEntity.setResultCode(INTERNAL_SERVER_ERROR.getCode());
            actionLogEntity.setResultMsg(ExceptionUtil.getRootCauseMessage(exception));
        }
    }

    private static boolean isLogEnable(ProceedingJoinPoint joinPoint,
                                       ActionLog actionLog) {
        // 有 @OperateLog 注解的情况下
        if (actionLog != null) {
            return actionLog.enable();
        }
        // 没有 @ApiOperation 注解的情况下，只记录 POST、PUT、DELETE 的情况
        return obtainFirstLogRequestMethod(obtainRequestMethod(joinPoint)) != null;
    }

    private static RequestMethod obtainFirstLogRequestMethod(RequestMethod[] requestMethods) {
        if (ArrayUtil.isEmpty(requestMethods)) {
            return null;
        }
        return Arrays.stream(requestMethods).filter(requestMethod ->
                        requestMethod == RequestMethod.POST
                                || requestMethod == RequestMethod.PUT
                                || requestMethod == RequestMethod.DELETE)
                .findFirst().orElse(null);
    }

    private static RequestMethod obtainFirstMatchRequestMethod(RequestMethod[] requestMethods) {
        if (ArrayUtil.isEmpty(requestMethods)) {
            return null;
        }
        // 优先，匹配最优的 POST、PUT、DELETE
        RequestMethod result = obtainFirstLogRequestMethod(requestMethods);
        if (result != null) {
            return result;
        }
        // 然后，匹配次优的 GET
        result = Arrays.stream(requestMethods).filter(requestMethod -> requestMethod == RequestMethod.GET)
                .findFirst().orElse(null);
        if (result != null) {
            return result;
        }
        // 兜底，获得第一个
        return requestMethods[0];
    }

    private static ActionTypeEnum convertOperateLogType(RequestMethod requestMethod) {
        if (requestMethod == null) {
            return null;
        }
        switch (requestMethod) {
            case GET:
                return ActionTypeEnum.GET;
            case POST:
                return ActionTypeEnum.CREATE;
            case PUT:
                return ActionTypeEnum.UPDATE;
            case DELETE:
                return ActionTypeEnum.DELETE;
            default:
                return ActionTypeEnum.OTHER;
        }
    }

    private static RequestMethod[] obtainRequestMethod(ProceedingJoinPoint joinPoint) {
        RequestMapping requestMapping = AnnotationUtils.getAnnotation( // 使用 Spring 的工具类，可以处理 @RequestMapping 别名注解
                ((MethodSignature) joinPoint.getSignature()).getMethod(), RequestMapping.class);
        return requestMapping != null ? requestMapping.method() : new RequestMethod[]{};
    }

    @SuppressWarnings("SameParameterValue")
    private static <T extends Annotation> T getMethodAnnotation(ProceedingJoinPoint joinPoint, Class<T> annotationClass) {
        return ((MethodSignature) joinPoint.getSignature()).getMethod().getAnnotation(annotationClass);
    }

    @SuppressWarnings("SameParameterValue")
    private static <T extends Annotation> T getClassAnnotation(ProceedingJoinPoint joinPoint, Class<T> annotationClass) {
        return ((MethodSignature) joinPoint.getSignature()).getMethod().getDeclaringClass().getAnnotation(annotationClass);
    }

    private static String obtainMethodArgs(ProceedingJoinPoint joinPoint) {
        // TODO 提升：参数脱敏和忽略
        MethodSignature methodSignature = (MethodSignature) joinPoint.getSignature();
        String[] argNames = methodSignature.getParameterNames();
        Object[] argValues = joinPoint.getArgs();
        // 拼接参数
        Map<String, Object> args = Maps.newHashMapWithExpectedSize(argValues.length);
        for (int i = 0; i < argNames.length; i++) {
            String argName = argNames[i];
            Object argValue = argValues[i];
            // 被忽略时，标记为 ignore 字符串，避免和 null 混在一起
            args.put(argName, !isIgnoreArgs(argValue) ? argValue : "[ignore]");
        }
        return JsonUtil.toJsonString(args);
    }

    private static String obtainResultData(Object result) {
        // TODO 提升：结果脱敏和忽略
        if (result instanceof Result) {
            result = ((Result<?>) result).getData();
        }
        return JsonUtil.toJsonString(result);
    }

    private static boolean isIgnoreArgs(Object object) {
        Class<?> clazz = object.getClass();
        // 处理数组的情况
        if (clazz.isArray()) {
            return IntStream.range(0, Array.getLength(object))
                    .anyMatch(index -> isIgnoreArgs(Array.get(object, index)));
        }
        // 递归，处理数组、Collection、Map 的情况
        if (Collection.class.isAssignableFrom(clazz)) {
            return ((Collection<?>) object).stream()
                    .anyMatch((Predicate<Object>) ActionLogAspect::isIgnoreArgs);
        }
        if (Map.class.isAssignableFrom(clazz)) {
            return isIgnoreArgs(((Map<?, ?>) object).values());
        }
        // obj
        return object instanceof MultipartFile
                || object instanceof HttpServletRequest
                || object instanceof HttpServletResponse
                || object instanceof BindingResult;
    }

}
