package cn.jrack.excel.core.handler;

import cn.jrack.core.util.common.StringUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.write.builder.ExcelWriterBuilder;
import com.alibaba.excel.write.handler.WriteHandler;
import com.alibaba.excel.write.merge.OnceAbsoluteMergeStrategy;
import com.alibaba.excel.write.metadata.style.WriteCellStyle;
import com.alibaba.excel.write.metadata.style.WriteFont;
import com.alibaba.excel.write.style.HorizontalCellStyleStrategy;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;

import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.IndexedColors;

import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;


/**
 * Excel处理
 *
 * @author joy
 */
public class ExcelHandler {


    private List<List<String>> heads;
    //标题
    private String title;

    //excel数据
    private List<List<Object>> data;

    //sheet名称
    private String sheetName = "sheet";

    //文件名
    private String fileName = "EXCEL";

    private String filePath;

    private int headIndex = 0;

    private WriteCellStyle headStyle;

    private WriteCellStyle contentStyle;


    private ExcelWriterBuilder excelWriterBuilder;


    //合并单元格集合
    private List<WriteHandler> writeHandlers;

    public ExcelHandler() {
        this.heads = new ArrayList<>();
        this.writeHandlers = new ArrayList<>();
        this.data = new ArrayList<>();
        headStyle = new WriteCellStyle();
        contentStyle = new WriteCellStyle();
        excelWriterBuilder = new ExcelWriterBuilder();
    }

    public ExcelHandler title(String title) {
        this.title = title;
        return this;
    }

    public ExcelHandler addRow(List<Object> list) {

        this.data.add(list);
        return this;
    }

    public ExcelHandler addRow(Object... objects) {
        List<Object> list = new ArrayList<>();
        for (Object o : objects) {
            list.add(o);
        }
        return addRow(list);
    }

    public ExcelHandler addHeads(List<List<String>> heads) {
        this.heads = heads;
        return this;
    }

    public ExcelHandler addHead(List<String> headCell) {
        if (headCell.size() > headIndex) {
            headIndex = headCell.size();
        }
        this.heads.add(headCell);
        return this;
    }

    public ExcelHandler addHead(String... headName) {
        return addHead(buildHead(headName));
    }

    public ExcelHandler data(List<List<Object>> data) {
        if (this.data.size() > 0) {
            this.data.addAll(data);
        } else {
            this.data = data;
        }
        return this;
    }

    public ExcelHandler buildData(List list) {
        try {
            return data(convertToArrayList(list));
        } catch (IllegalAccessException ex) {

        }
        return this;
    }

    public void doWrite(String filePath) throws IOException {
        this.filePath = filePath;
        doWrite();
    }

    public void doWrite() throws IOException {
        checkAttribute();
        if (!StringUtil.isEmpty(title)) {
            for (List<String> l : this.heads) {
                l.add(0, title);
            }
        }
        File file = new File(filePath);
        excelWriterBuilder = EasyExcel.write(file);
        for (WriteHandler writeHandler :
                writeHandlers) {
            excelWriterBuilder.registerWriteHandler(writeHandler);
        }
        excelWriterBuilder.registerWriteHandler(new LongestMatchColumnWidthStyleStrategy());
        excelWriterBuilder.head(heads);
        excelWriterBuilder.sheet(sheetName).doWrite(this.data);
    }

    public void doExport(HttpServletResponse response) throws IOException {
        if (!StringUtil.isEmpty(title)) {
            for (List<String> l : this.heads) {
                l.add(0, title);
            }
        }
        // 这个策略是 头是头的样式 内容是内容的样式 其他的策略可以自己实现
        excelWriterBuilder = EasyExcel.write(response.getOutputStream());
        for (WriteHandler writeHandler :
                writeHandlers) {
            excelWriterBuilder.registerWriteHandler(writeHandler);
        }
        excelWriterBuilder.registerWriteHandler(new HorizontalCellStyleStrategy(headStyle, contentStyle));
        excelWriterBuilder.head(heads).autoCloseStream(false);
        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        String fileName = URLEncoder.encode(this.fileName, "UTF-8").replaceAll("\\+", "%20");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
        excelWriterBuilder.sheet(sheetName).doWrite(this.data);
    }

    //构建头部列
    public static List<String> buildHead(String... label) {
        List<String> list = new ArrayList<>();
        for (String l : label) {
            list.add(l);
        }
        return list;
    }


    //注册自定义单元样式
    public ExcelHandler registerWriteHandler(WriteHandler writeHandler) {
        writeHandlers.add(writeHandler);
//        excelWriterBuilder.registerWriteHandler(writeHandler);
        return this;
    }


    //合并指定行、列单元格
    public ExcelHandler mergeCell(int firstRowIndex, int lastRowIndex, int firstColumnIndex, int lastColumnIndex) {
        OnceAbsoluteMergeStrategy onceAbsoluteMergeStrategy = new OnceAbsoluteMergeStrategy(firstRowIndex, lastRowIndex, firstColumnIndex, lastColumnIndex);
        return registerWriteHandler(onceAbsoluteMergeStrategy);

    }


    //头部颜色控制
    public ExcelHandler headStyle(int fontHeightInPoints, IndexedColors fillForegroundColor, IndexedColors fontColor) {
        // 头的策略
        WriteCellStyle headWriteCellStyle = new WriteCellStyle();
        // 背景设置为红色
        headWriteCellStyle.setFillForegroundColor(fillForegroundColor.getIndex());
        WriteFont headWriteFont = new WriteFont();
        headWriteFont.setFontHeightInPoints((short) fontHeightInPoints);
        headWriteFont.setColor(fontColor.getIndex());
        headWriteCellStyle.setWriteFont(headWriteFont);

        return headStyle(headWriteCellStyle);
    }

    public ExcelHandler headStyle(WriteCellStyle writeCellStyle) {


        this.headStyle = writeCellStyle;

        return this;
    }

    public ExcelHandler border(Boolean enableBorder) {
        if (enableBorder) {
            WriteCellStyle contentWriteCellStyle = new WriteCellStyle();
            BorderStyle borderStyle = BorderStyle.THIN;
            contentWriteCellStyle.setBorderRight(borderStyle);
            contentWriteCellStyle.setBorderBottom(borderStyle);
            contentWriteCellStyle.setBorderTop(borderStyle);
            contentWriteCellStyle.setBorderLeft(borderStyle);
            return this.contentStyle(contentWriteCellStyle);
        }
        return this;
    }

    public ExcelHandler contentStyle(int fontHeightInPoints, IndexedColors fillForegroundColor, IndexedColors fontColor) {
        // 内容的策略
        WriteCellStyle contentWriteCellStyle = new WriteCellStyle();
        // 这里需要指定 FillPatternType 为FillPatternType.SOLID_FOREGROUND 不然无法显示背景颜色.头默认了 FillPatternType所以可以不指定
        contentWriteCellStyle.setFillPatternType(FillPatternType.SOLID_FOREGROUND);
        // 背景绿色
        contentWriteCellStyle.setFillForegroundColor(fillForegroundColor.getIndex());
        WriteFont contentWriteFont = new WriteFont();
        // 字体大小
        contentWriteFont.setFontHeightInPoints((short) fontHeightInPoints);
        contentWriteFont.setColor(fontColor.getIndex());
        contentWriteCellStyle.setWriteFont(contentWriteFont);

        return contentStyle(contentWriteCellStyle);
    }

    public ExcelHandler contentStyle(WriteCellStyle writeCellStyle) {
        this.contentStyle = writeCellStyle;
        return this;
    }


    //注册自定义单元样式
    public ExcelHandler openLongestMatchColumnWidthStyleStrategy() {
        registerWriteHandler(new LongestMatchColumnWidthStyleStrategy());
        return this;
    }


    public ExcelHandler fileName(String fileName) {
        this.fileName = fileName;
        return this;
    }

    public ExcelHandler sheetName(String sheetName) {
        this.sheetName = sheetName;
        return this;
    }

    public ExcelHandler filePath(String filePath) {
        this.filePath = filePath;
        return this;
    }

//    public ExcelHandler statistics(List<Object> list) {
//        this.statistics = list;
//        return this;
//    }

//    public ExcelHandler statistics(Object... objects) {
//        List<Object> list = new ArrayList<>();
//        for (Object o : objects) {
//            list.add(o);
//        }
//        return statistics(list);
//    }

    //检查属性
    private void checkAttribute() throws IOException {
        if (StringUtil.isEmpty(filePath)) {
            throw new IOException("filePath not set");
        }
    }

    //计算行
    private int computeRowSize() {
        int rn = headIndex + data.size();
        if (!StringUtil.isEmpty(title)) {
            rn++;
        }
        return rn;
    }

    //计算行索引
    private int computeRowIndex() {
        return computeRowSize() - 1;
    }


    public static <T> List<List<Object>> convertToArrayList(List<T> list) throws IllegalAccessException {
        List<List<Object>> listArrayList = new ArrayList<>();

        for (T t : list) {
            List<Object> objectList = new ArrayList<>();
            Object cs = t;
            Field[] fields = cs.getClass().getDeclaredFields();
            int i = 0;
            for (Field field : fields) {
                field.setAccessible(true);  //抑制java语言访问检查，反射访问private访问权限的属性值
                Object value = field.get(t); // 获取实体类中属性的值
                objectList.add(value);
            }
            listArrayList.add(objectList);
            i++;
        }

        return listArrayList;
    }


}
