package cn.jque.core.util;

import cn.jque.core.model.dto.DataCalendarDTO;
import com.google.common.collect.Lists;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.sql.Timestamp;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalAdjusters;
import java.time.temporal.WeekFields;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * 日期转换（Java8）
 * @author Liu.ShiTing
 * @version 1.0.0
 * @date 2023-01-10 09:16:50
 */
public class DateConvertUtils {

	//根据当前时间获取特定的日期或时间
	//主要思路是使用TemporalAdjusters工具类获取当前特定的日期，
	//然后再通过LocalDateTime.of将特定的日期与特定的时间组合成最终的LocalDateTime
	public static final Integer SPRING = 1;
	public static final Integer SUMMER = 2;
	public static final Integer AUTUMN = 3;
	public static final Integer WINTER = 4;

	public static final String SUNDAY = "星期日";
	public static final String MONDAY = "星期一";
	public static final String TUESDAY = "星期二";
	public static final String WEDNESDAY = "星期三";
	public static final String THURSDAY = "星期四";
	public static final String FRIDAY = "星期五";
	public static final String SATURDAY = "星期六";

	private static final String YEAR = "year";
	private static final String MONTH = "month";
	private static final String WEEK = "week";
	private static final String DAY = "day";
	private static final String HOUR = "hour";
	private static final String MINUTE = "minute";
	private static final String SECOND = "second";

	public static final String DATETIME_PATTERN = "yyyy-MM-dd HH:mm:ss";
	public static final String DATE_HOUR_MINUTE_PATTERN = "yyyy-MM-dd HH:mm";
	public static final String DATE_PATTERN = "yyyy-MM-dd";
	public static final String TIME_PATTERN = "HH:mm:ss";
	public static final String HOUR_MINUTE_PATTERN = "HH:mm";
	public static final String UNSIGNED_DATETIME_PATTERN = "yyyyMMddHHmmss";
	public static final String UNSIGNED_DATE_PATTERN = "yyyyMMdd";
	public static final String UNSIGNED_TIME_PATTERN = "HHmmss";


	//------------------------------ 年、季度、月、周、日获取的常用方法 start ------------------------------

	//------------------------------ 年相关的常用方法 ------------------------------

	/**
	 * 获取今年的第一天
	 * @return 2023-01-01
	 */
	public static LocalDate getCurrentYearFirstDay() {
		return LocalDate.now().with(TemporalAdjusters.firstDayOfYear());
	}

	/**
	 * 获取今年第一天的开始时间
	 * @return 2023-01-01 00:00:00
	 */
	public static LocalDateTime getCurrentYearFirstDayTime() {
		LocalDate firstDayOfYear = LocalDate.now().with(TemporalAdjusters.firstDayOfYear());
		return LocalDateTime.of(firstDayOfYear, LocalTime.MIN);
	}

	/**
	 * 获取下一年的第一天
	 * @return 2023-01-01
	 */
	public static LocalDate getNextYearFirstDay() {
		return LocalDate.now().with(TemporalAdjusters.firstDayOfNextYear());
	}

	/**
	 * 获取今年第一天的日期字符串
	 * @return String 格式：yyyy-MM-dd 00:00:00
	 */
	public static String getCurrentYearFirstDayStr() {
		return getSpecifyYearFirstDayStr(LocalDateTime.now());
	}

	/**
	 * 获取指定日期当年第一天的日期字符串
	 * @param localDateTime 指定日期时间
	 * @return String 格式：yyyy-MM-dd 00:00:00
	 */
	public static String getSpecifyYearFirstDayStr(LocalDateTime localDateTime) {
		return getSpecifyYearFirstDayStr(localDateTime, DATETIME_PATTERN);
	}

	/**
	 * 获取指定日期当年第一天的日期字符串（带日期格式化参数）
	 * @param localDateTime 指定日期时间
	 * @param pattern       日期时间格式
	 * @return String 格式：yyyy-MM-dd 00:00:00
	 */
	public static String getSpecifyYearFirstDayStr(LocalDateTime localDateTime, String pattern) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		return format(localDateTime.withDayOfYear(1).withHour(0).withMinute(0).withSecond(0), pattern);
	}

	/**
	 * 获取今年的最后一天
	 * @return 2023-12-31
	 */
	public static LocalDate getCurrentYearLastDay() {
		return LocalDate.now().with(TemporalAdjusters.lastDayOfYear());
	}

	/**
	 * 获取今年最后一天的结束时间
	 * @return 2023-12-31 23:59:59
	 */
	public static LocalDateTime getCurrentYearLastDayTime() {
		LocalDate lastDayOfYear = LocalDate.now().with(TemporalAdjusters.lastDayOfYear());
		return LocalDateTime.of(lastDayOfYear, LocalTime.MAX);
	}

	/**
	 * 获取今年最后一天的日期字符串
	 * @return String 格式：yyyy-MM-dd 23:59:59
	 */
	public static String getCurrentYearLastDayStr() {
		return getSpecifyYearLastDayStr(LocalDateTime.now());
	}

	/**
	 * 获取指定日期当年最后一天的日期字符串
	 * @param localDateTime 指定日期时间
	 * @return String 格式：yyyy-MM-dd 23:59:59
	 */
	public static String getSpecifyYearLastDayStr(LocalDateTime localDateTime) {
		return getSpecifyYearLastDayStr(localDateTime, DATETIME_PATTERN);
	}

	/**
	 * 获取指定日期当年最后一天的日期字符串（带日期格式化参数）
	 * @param localDateTime 指定日期时间
	 * @param pattern       日期时间格式
	 * @return String 格式：yyyy-MM-dd 23:59:59
	 */
	public static String getSpecifyYearLastDayStr(LocalDateTime localDateTime, String pattern) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		return format(localDateTime.with(TemporalAdjusters.lastDayOfYear()).withHour(23).withMinute(59).withSecond(59), pattern);
	}


	//------------------------------ 年月相关的常用方法 ------------------------------

	/**
	 * 获取当前年月
	 */
	public static YearMonth getCurrentYearMonth() {
		return YearMonth.now();
	}

	/**
	 * 获取本月有多少天
	 */
	public static int getCurrentMonthDays() {
		return getCurrentYearMonth().lengthOfMonth();
	}


	//------------------------------ 月相关的常用方法 ------------------------------

	/**
	 * 获取当前月的第一天
	 * @return 2023-01-01
	 */
	public static LocalDate getCurrentMonthFirstDay() {
		return LocalDate.now().with(TemporalAdjusters.firstDayOfMonth());
	}

	/**
	 * 获取下个月的第一天
	 * @return 2023-02-01
	 */
	public static LocalDate getNextMonthFirstDay() {
		return LocalDate.now().with(TemporalAdjusters.firstDayOfNextMonth());
	}

	/**
	 * 获取当前月第一天的开始时间
	 * @return 2023-01-01 00:00:00
	 */
	public static LocalDateTime getCurrentMonthFirstDayTime() {
		LocalDate firstDayOfMonth = LocalDate.now().with(TemporalAdjusters.firstDayOfMonth());
		return LocalDateTime.of(firstDayOfMonth, LocalTime.MIN);
	}

	/**
	 * 获取当前月第一天的日期字符串
	 * @return String 格式：yyyy-MM-dd 00:00:00
	 */
	public static String getCurrentMonthFirstDayStr() {
		return getSpecifyMonthFirstDayStr(LocalDateTime.now());
	}

	/**
	 * 获取指定日期当月第一天的日期字符串
	 * @param localDateTime 指定日期时间
	 * @return String 格式：yyyy-MM-dd 23:59:59
	 */
	public static String getSpecifyMonthFirstDayStr(LocalDateTime localDateTime) {
		return getSpecifyMonthFirstDayStr(localDateTime, DATETIME_PATTERN);
	}

	/**
	 * 获取指定日期当月第一天的日期字符串（带日期格式化参数）
	 * @param localDateTime 指定日期时间
	 * @return String 格式：yyyy-MM-dd 00:00:00
	 */
	public static String getSpecifyMonthFirstDayStr(LocalDateTime localDateTime, String pattern) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		Objects.requireNonNull(pattern, "pattern");
		return format(localDateTime.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0), pattern);
	}

	/**
	 * 获取当前月的最后一天
	 * @return 2023-01-31
	 */
	public static LocalDate getCurrentMonthLastDay() {
		return LocalDate.now().with(TemporalAdjusters.lastDayOfMonth());
	}

	/**
	 * 获取当前月最后一天的结束时间
	 * @return 2023-01-31 23:59:59
	 */
	public static LocalDateTime getCurrentMonthLastDayTime() {
		LocalDate lastDayOfMonth = LocalDate.now().with(TemporalAdjusters.lastDayOfMonth());
		return LocalDateTime.of(lastDayOfMonth, LocalTime.MAX);
	}

	/**
	 * 获取当前月最后一天的日期字符串
	 * @return String 格式：yyyy-MM-dd 23:59:59
	 */
	public static String getCurrentMonthLastDayStr() {
		return getSpecifyMonthLastDayStr(LocalDateTime.now());
	}

	/**
	 * 获取指定日期当月最后一天的日期字符串
	 * @param localDateTime 指定日期时间
	 * @return String 格式：yyyy-MM-dd 23:59:59
	 */
	public static String getSpecifyMonthLastDayStr(LocalDateTime localDateTime) {
		return getSpecifyMonthLastDayStr(localDateTime, DATETIME_PATTERN);
	}

	/**
	 * 获取指定日期当月最后一天的日期字符串（带日期格式化参数）
	 * @param localDateTime 指定日期时间
	 * @param pattern       日期时间格式
	 * @return String 格式：yyyy-MM-dd 23:59:59
	 */
	public static String getSpecifyMonthLastDayStr(LocalDateTime localDateTime, String pattern) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		Objects.requireNonNull(pattern, "pattern");
		return format(localDateTime.with(TemporalAdjusters.lastDayOfMonth()).withHour(23).withMinute(59).withSecond(59), pattern);
	}


	//------------------------------ 周相关的常用方法 ------------------------------

	/**
	 * 获取当前星期字符串.
	 * @return String 当前星期字符串，例如 星期二
	 */
	public static String getCurrentDayOfWeekStr() {
		return format(LocalDate.now(), "E");
	}

	/**
	 * 获取本周第一天的日期字符串
	 * @return String 格式：yyyy-MM-dd 00:00:00
	 */
	public static String getCurrentWeekFirstDayStr() {
		return getSpecifyWeekFirstDayStr(LocalDateTime.now());
	}

	/**
	 * 获取指定日期当周第一天的日期字符串（这里第一天为周一）
	 * @param localDateTime 指定日期时间
	 * @return String 格式：yyyy-MM-dd 00:00:00
	 */
	public static String getSpecifyWeekFirstDayStr(LocalDateTime localDateTime) {
		return getSpecifyWeekFirstDayStr(localDateTime, DATETIME_PATTERN);
	}

	/**
	 * 获取指定日期当周第一天的日期字符串（这里第一天为周一）
	 * @param localDateTime 指定日期时间
	 * @param pattern       日期时间格式
	 * @return 格式：yyyy-MM-dd 00:00:00
	 */
	public static String getSpecifyWeekFirstDayStr(LocalDateTime localDateTime, String pattern) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		Objects.requireNonNull(pattern, "pattern");
		return format(localDateTime.with(DayOfWeek.MONDAY).withHour(0).withMinute(0).withSecond(0), pattern);
	}

	/**
	 * 获取本周最后一天的日期字符串
	 * @return String 格式：yyyy-MM-dd 23:59:59
	 */
	public static String getCurrentWeekLastDayStr() {
		return getSpecifyWeekLastDayStr(LocalDateTime.now());
	}

	/**
	 * 获取指定日期当周最后一天的日期字符串（这里最后一天为周日）
	 * @param localDateTime 指定日期时间
	 * @return String 格式：yyyy-MM-dd 23:59:59
	 */
	public static String getSpecifyWeekLastDayStr(LocalDateTime localDateTime) {
		return getSpecifyWeekLastDayStr(localDateTime, DATETIME_PATTERN);
	}

	/**
	 * 获取指定日期当周最后一天的日期字符串（这里最后一天为周日）
	 * @param localDateTime 指定日期时间
	 * @param pattern       日期时间格式
	 * @return String 格式：yyyy-MM-dd 23:59:59
	 */
	public static String getSpecifyWeekLastDayStr(LocalDateTime localDateTime, String pattern) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		Objects.requireNonNull(pattern, "pattern");
		return format(localDateTime.with(DayOfWeek.SUNDAY).withHour(23).withMinute(59).withSecond(59), pattern);
	}

	/**
	 * 获取上一个周一
	 */
	public static LocalDateTime getPreviousMonday() {
		return LocalDateTime.now().with(TemporalAdjusters.previous(DayOfWeek.MONDAY));
	}

	/**
	 * 获取下一个周一
	 */
	public static LocalDateTime getNextMonday() {
		return LocalDateTime.now().with(TemporalAdjusters.next(DayOfWeek.MONDAY));
	}

	/**
	 * 获取上一个周五
	 */
	public static LocalDateTime getPreviousFriday() {
		return LocalDateTime.now().with(TemporalAdjusters.previousOrSame(DayOfWeek.FRIDAY));
	}

	/**
	 * 获取下一个周五
	 */
	public static LocalDateTime getNextFriday() {
		return LocalDateTime.now().with(TemporalAdjusters.nextOrSame(DayOfWeek.FRIDAY));
	}

	/**
	 * 获取本月第一个周五
	 */
	public static LocalDateTime getCurrentMonthFirstFriday() {
		return LocalDateTime.now().with(TemporalAdjusters.firstInMonth(DayOfWeek.FRIDAY));
	}

	/**
	 * 获取当前月最后一个周五
	 */
	public static LocalDateTime getCurrentMonthLastFriday() {
		return LocalDateTime.now().with(TemporalAdjusters.lastInMonth(DayOfWeek.FRIDAY));
	}

	/**
	 * 获取当前月的第几个周几
	 * @param index     第几个
	 * @param dayOfWeek 周（DayOfWeek.MONDAY、DayOfWeek.TUESDAY、DayOfWeek.WEDNESDAY、DayOfWeek.THURSDAY、DayOfWeek.FRIDAY、DayOfWeek.SATURDAY、DayOfWeek.SUNDAY）
	 * @return LocalDate 格式：yyyy-MM-dd
	 */
	public static LocalDate getCurrentMonthWeekLocalDate(int index, DayOfWeek dayOfWeek) {
		Objects.requireNonNull(dayOfWeek, "dayOfWeek");
		return LocalDate.now().with(TemporalAdjusters.dayOfWeekInMonth(index, dayOfWeek));
	}

	/**
	 * 获取几周后日期
	 * @param index 几周
	 * @return LocalDate 格式：yyyy-MM-dd
	 */
	public static LocalDate getLocalDateByIndex(int index) {
		return LocalDate.now().with(TemporalAdjusters.ofDateAdjuster(date -> date.plusWeeks(index)));
	}

	/**
	 * 获取指定日期是星期几
	 * @param localDate 日期
	 * @return String 星期几
	 */
	public static String getSpecifyDateWeekStr(LocalDate localDate) {
		Objects.requireNonNull(localDate, "localDate");
		String[] weekOfDays = {MONDAY, TUESDAY, WEDNESDAY, THURSDAY, FRIDAY, SATURDAY, SUNDAY};
		int dayOfWeek = localDate.getDayOfWeek().getValue() - 1;
		return weekOfDays[dayOfWeek];
	}

	/**
	 * 获取指定日期当周的日期时间字符串
	 * @param localDateTime 指定日期时间
	 * @param dayOfWeek     周（DayOfWeek.MONDAY、DayOfWeek.TUESDAY、DayOfWeek.WEDNESDAY、DayOfWeek.THURSDAY、DayOfWeek.FRIDAY、DayOfWeek.SATURDAY、DayOfWeek.SUNDAY）
	 * @param pattern       日期时间格式
	 * @return String 格式：yyyy-MM-dd 16:50:33
	 */
	public static String getSpecifyDayCurrentWeekDateStr(LocalDateTime localDateTime, DayOfWeek dayOfWeek, String pattern) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		Objects.requireNonNull(dayOfWeek, "dayOfWeek");
		Objects.requireNonNull(pattern, "pattern");
		WeekFields weekFields = WeekFields.of(dayOfWeek, 1);
		DayOfWeek firstDayOfWeek = weekFields.getFirstDayOfWeek();
		return format(localDateTime.with(localDateTime.with(firstDayOfWeek)), pattern);
	}


	//------------------------------ 天相关的常用方法 ------------------------------

	/**
	 * 获取今天的开始时间
	 * @return 2023-01-15 00:00:00
	 */
	public static LocalDateTime getCurrentDayFirstTime() {
		return LocalDateTime.of(LocalDate.now(), LocalTime.MIN);
	}

	/**
	 * 获取今天开始时间的日期字符串
	 * @return String 格式：yyyy-MM-dd 00:00:00
	 */
	public static String getCurrentDayFirstTimeStr() {
		return getSpecifyDayFirstTimeStr(LocalDateTime.now());
	}

	/**
	 * 获取指定日期开始时间的日期字符串（Specify date）
	 * @param localDateTime 指定日期时间
	 * @return String 格式：yyyy-MM-dd 00:00:00
	 */
	public static String getSpecifyDayFirstTimeStr(LocalDateTime localDateTime) {
		return getSpecifyDayFirstTimeStr(localDateTime, DATETIME_PATTERN);
	}

	/**
	 * 获取指定日期开始时间的日期字符串（带日期格式化参数）
	 * @param localDateTime 指定日期时间
	 * @param pattern       日期时间格式
	 * @return String 格式：yyyy-MM-dd 00:00:00
	 */
	public static String getSpecifyDayFirstTimeStr(LocalDateTime localDateTime, String pattern) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		Objects.requireNonNull(pattern, "pattern");
		return format(localDateTime.withHour(0).withMinute(0).withSecond(0), pattern);
	}

	/**
	 * 获取今天的结束时间
	 * @return 2023-01-15 23:59:59
	 */
	public static LocalDateTime getCurrentDayLastTime() {
		return LocalDateTime.of(LocalDate.now(), LocalTime.MAX);
	}

	/**
	 * 获取今天结束时间的日期字符串
	 * @return String 格式：yyyy-MM-dd 23:59:59
	 */
	public static String getCurrentDayLastTimeStr() {
		return getSpecifyDayLastTimeStr(LocalDateTime.now());
	}

	/**
	 * 获取指定日期结束时间的日期字符串
	 * @param localDateTime 指定日期时间
	 * @return String 格式：yyyy-MM-dd 23:59:59
	 */
	public static String getSpecifyDayLastTimeStr(LocalDateTime localDateTime) {
		return getSpecifyDayLastTimeStr(localDateTime, DATETIME_PATTERN);
	}

	/**
	 * 获取指定日期结束时间的日期字符串（带日期格式化参数）
	 * @param localDateTime 指定日期时间
	 * @param pattern       日期时间格式
	 * @return String 格式：yyyy-MM-dd 23:59:59
	 */
	public static String getSpecifyDayLastTimeStr(LocalDateTime localDateTime, String pattern) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		Objects.requireNonNull(pattern, "pattern");
		return format(localDateTime.withHour(23).withMinute(59).withSecond(59), pattern);
	}

	/**
	 * 获取当前日期和时间字符串
	 * @return String 日期时间字符串，例如 2023-01-15 09:51:53
	 */
	public static String getCurrentLocalDateTimeStr() {
		return format(LocalDateTime.now(), DATETIME_PATTERN);
	}

	/**
	 * 获取当前日期字符串
	 * @return String 日期字符串，例如2023-01-15
	 */
	public static String getCurrentLocalDateStr() {
		return format(LocalDate.now(), DATE_PATTERN);
	}

	//------------------------------ 时相关的常用方法 ------------------------------

	/**
	 * 获取当前时间字符串.
	 * @return String 时间字符串，例如 09:51:53
	 */
	public static String getCurrentLocalTimeStr() {
		return format(LocalTime.now(), TIME_PATTERN);
	}

	//------------------------------ 年、季度、月、周、日获取的常用方法 end ------------------------------


	//------------------------------ 转LocalDateTime start ------------------------------

	/**
	 * LocalDate转LocalDateTime
	 */
	public static LocalDateTime toLocalDateTime(LocalDate localDate) {
		Objects.requireNonNull(localDate, "localDate");
		return localDate.atTime(LocalTime.now());
	}

	/**
	 * LocalDate转LocalDateTime
	 */
	public static LocalDateTime toLocalDateTime(LocalDate localDate, int hour, int minute, int second) {
		Objects.requireNonNull(localDate, "localDate");
		return localDate.atTime(hour, minute, second);
	}

	/**
	 * LocalDate转LocalDateTime
	 * @param localDate
	 * @return 2023-01-14 00:00:00
	 */
	public static LocalDateTime toLocalDateFirstTime(LocalDate localDate) {
		Objects.requireNonNull(localDate, "localDate");
		return localDate.atStartOfDay();
	}

	/**
	 * LocalDate转LocalDateTime
	 * @param localDate
	 * @return 2023-01-21 23:59:59
	 */
	public static LocalDateTime toLocalDateLastTime(LocalDate localDate) {
		Objects.requireNonNull(localDate, "localDate");
		return LocalDateTime.of(localDate, LocalTime.MAX);
	}

	/**
	 * LocalTime转LocalDateTime
	 * 以当天的日期+LocalTime组成新的LocalDateTime
	 * @param localTime LocalTime
	 * @return LocalDateTime
	 */
	public static LocalDateTime toLocalDateTime(LocalTime localTime) {
		Objects.requireNonNull(localTime, "localTime");
		return LocalDate.now().atTime(localTime);
	}

	/**
	 * Date转LocalDateTime
	 */
	public static LocalDateTime toLocalDateTime(Date date) {
		Objects.requireNonNull(date, "date");
		return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
	}

	/**
	 * ZonedDateTime转LocalDateTime
	 * 注意时间对应的时区和默认时区差异
	 * @param zonedDateTime ZonedDateTime
	 * @return LocalDateTime
	 */
	public static LocalDateTime toLocalDateTime(ZonedDateTime zonedDateTime) {
		Objects.requireNonNull(zonedDateTime, "zonedDateTime");
		return zonedDateTime.toLocalDateTime();
	}

	/**
	 * Instant转LocalDateTime
	 * @param instant Instant
	 * @return LocalDateTime
	 */
	public static LocalDateTime toLocalDateTime(Instant instant) {
		Objects.requireNonNull(instant, "instant");
		return LocalDateTime.ofInstant(instant, ZoneId.systemDefault());
	}


	/**
	 * 时间戳epochMilli毫秒转LocalDateTime
	 * @param epochMilli 时间戳
	 * @return LocalDateTime
	 */
	public static LocalDateTime toLocalDateTime(long epochMilli) {
		return LocalDateTime.ofInstant(Instant.ofEpochMilli(epochMilli), ZoneId.systemDefault());
	}

	/**
	 * Timestamp转LocalDateTime
	 * @param timestamp Timestamp
	 * @return LocalDateTime
	 */
	public static LocalDateTime toLocalDateTime(Timestamp timestamp) {
		Objects.requireNonNull(timestamp, "timestamp");
		return timestamp.toLocalDateTime();
	}

	/**
	 * 日期时间字符串转LocalDateTime
	 * @param localDateTimeStr 日期时间字符串
	 * @return LocalDateTime 日期时间
	 */
	public static LocalDateTime toLocalDateTime(String localDateTimeStr) {
		Objects.requireNonNull(localDateTimeStr, "localDateTimeStr");
		return LocalDateTime.parse(localDateTimeStr, DateTimeFormatter.ofPattern(DATETIME_PATTERN));
	}

	/**
	 * 日期时间字符串转LocalDateTime
	 * @param localDateTimeStr 日期时间字符串
	 * @param pattern          日期时间格式
	 * @return LocalDateTime 日期时间
	 */
	public static LocalDateTime toLocalDateTime(String localDateTimeStr, String pattern) {
		Objects.requireNonNull(localDateTimeStr, "localDateTimeStr");
		Objects.requireNonNull(pattern, "pattern");
		return LocalDateTime.parse(localDateTimeStr, DateTimeFormatter.ofPattern(pattern));
	}

	//------------------------------ 转LocalDateTime end ------------------------------


	//------------------------------ 转LocalDate start ------------------------------

	/**
	 * LocalDateTime转LocalDate
	 * @param localDateTime
	 * @return
	 */
	public static LocalDate toLocalDate(LocalDateTime localDateTime) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		return localDateTime.toLocalDate();
	}

	/**
	 * YearMonth转LocalDate
	 * 注意dayOfMonth范围：1到31之间，最大值根据月份确定特殊情况，如2月闰年29，非闰年28
	 * 如果要转换为当月最后一天，可以使用下面方法：toLocalDateEndOfMonth(YearMonth)
	 * @param yearMonth  YearMonth
	 * @param dayOfMonth 天
	 * @return LocalDate
	 */
	public static LocalDate toLocalDate(YearMonth yearMonth, int dayOfMonth) {
		Objects.requireNonNull(yearMonth, "YearMonth");
		return yearMonth.atDay(dayOfMonth);
	}

	/**
	 * YearMonth转LocalDate（转换为当月第一天）
	 * @param yearMonth YearMonth
	 * @return LocalDatev
	 */
	public static LocalDate toLocalDateStartOfMonth(YearMonth yearMonth) {
		return toLocalDate(yearMonth, 1);
	}

	/**
	 * YearMonth转LocalDate（转换为当月最后一天）
	 * @param yearMonth YearMonth
	 * @return LocalDate
	 */
	public static LocalDate toLocalDateEndOfMonth(YearMonth yearMonth) {
		Objects.requireNonNull(yearMonth, "YearMonth");
		return yearMonth.atEndOfMonth();
	}

	/**
	 * Date转LocalDate
	 */
	public static LocalDate toLocalDate(Date date) {
		Objects.requireNonNull(date, "date");
		return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
	}

	/**
	 * ZonedDateTime转LocalDate
	 * 注意时间对应的时区和默认时区差异
	 * @param zonedDateTime ZonedDateTime
	 * @return LocalDate
	 */
	public static LocalDate toLocalDate(ZonedDateTime zonedDateTime) {
		Objects.requireNonNull(zonedDateTime, "zonedDateTime");
		return zonedDateTime.toLocalDate();
	}

	/**
	 * Instant转LocalDate
	 * @param instant Instant
	 * @return LocalDate
	 */
	public static LocalDate toLocalDate(Instant instant) {
		return toLocalDateTime(instant).toLocalDate();
	}

	/**
	 * 时间戳epochMilli毫秒转LocalDate
	 * @param epochMilli 时间戳
	 * @return LocalDate
	 */
	public static LocalDate toLocalDate(long epochMilli) {
		return toLocalDateTime(epochMilli).toLocalDate();
	}

	/**
	 * 日期字符串转LocalDate
	 * @param localDateStr 日期字符串
	 * @return LocalDate 日期
	 */
	public static LocalDate toLocalDate(String localDateStr) {
		Objects.requireNonNull(localDateStr, "localDateStr");
		return LocalDate.parse(localDateStr, DateTimeFormatter.ofPattern(DATE_PATTERN));
	}

	/**
	 * 日期字符串转LocalDate
	 * @param localDateStr 日期字符串
	 * @param pattern      日期格式
	 * @return LocalDate 日期
	 */
	public static LocalDate toLocalDate(String localDateStr, String pattern) {
		Objects.requireNonNull(localDateStr, "localDateStr");
		Objects.requireNonNull(pattern, "pattern");
		return LocalDate.parse(localDateStr, DateTimeFormatter.ofPattern(pattern));
	}
	//------------------------------ 转LocalDate end ------------------------------


	//------------------------------ 转LocalTime start ------------------------------

	/**
	 * LocalDateTime转LocalTime
	 * @param localDateTime
	 * @return
	 */
	public static LocalTime toLocalTime(LocalDateTime localDateTime) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		return localDateTime.toLocalTime();
	}

	/**
	 * Date转LocalTime
	 */
	public static LocalTime toLocalTime(Date date) {
		Objects.requireNonNull(date, "date");
		return date.toInstant().atZone(ZoneId.systemDefault()).toLocalTime();
	}

	/**
	 * ZonedDateTime转LocalTime
	 * 注意时间对应的时区和默认时区差异
	 * @param zonedDateTime ZonedDateTime
	 * @return LocalTime
	 */
	public static LocalTime toLocalTime(ZonedDateTime zonedDateTime) {
		Objects.requireNonNull(zonedDateTime, "zonedDateTime");
		return zonedDateTime.toLocalTime();
	}

	/**
	 * Instant转LocalTime
	 * @param instant Instant
	 * @return LocalTime
	 */
	public static LocalTime toLocalTime(Instant instant) {
		return toLocalDateTime(instant).toLocalTime();
	}

	/**
	 * 时间戳的毫秒转LocalTime
	 * @param epochMilli 时间戳的毫秒
	 * @return LocalTime
	 */
	public static LocalTime toLocalTime(long epochMilli) {
		return toLocalDateTime(epochMilli).toLocalTime();
	}

	/**
	 * 时间部分的毫秒转LocalTime
	 * @param timeMilli 时间部分的毫秒
	 * @return LocalTime
	 */
	public static LocalTime toLocalTime(int timeMilli) {
		return LocalTime.ofNanoOfDay((long) timeMilli * 1000_000);
	}

	/**
	 * 日期字符串转LocalTime
	 * @param localTimeStr 时间字符串
	 * @return LocalTime 时间
	 */
	public static LocalTime toLocalTime(String localTimeStr) {
		Objects.requireNonNull(localTimeStr, "localTimeStr");
		return LocalTime.parse(localTimeStr, DateTimeFormatter.ofPattern(TIME_PATTERN));
	}

	/**
	 * 日期字符串转LocalTime
	 * @param localTimeStr 时间字符串
	 * @param pattern      时间格式
	 * @return LocalTime 时间
	 */
	public static LocalTime toLocalTime(String localTimeStr, String pattern) {
		Objects.requireNonNull(localTimeStr, "localTimeStr");
		Objects.requireNonNull(pattern, "pattern");
		return LocalTime.parse(localTimeStr, DateTimeFormatter.ofPattern(pattern));
	}


	//------------------------------ 转LocalTime end ------------------------------


	//------------------------------ 转YearMonth start ------------------------------

	/**
	 * LocalDateTime转YearMonth
	 * @param localDateTime LocalDateTime
	 * @return YearMonth
	 */
	public static YearMonth toYearMonth(LocalDateTime localDateTime) {
		LocalDate localDate = toLocalDate(localDateTime);
		return YearMonth.of(localDate.getYear(), localDate.getMonthValue());
	}

	/**
	 * LocalDate转YearMonth
	 * @param localDate LocalDate
	 * @return YearMonth
	 */
	public static YearMonth toYearMonth(LocalDate localDate) {
		Objects.requireNonNull(localDate, "localDate");
		return YearMonth.of(localDate.getYear(), localDate.getMonthValue());
	}

	/**
	 * Date转YearMonth
	 * @param date Date
	 * @return YearMonth
	 */
	public static YearMonth toYearMonth(Date date) {
		LocalDate localDate = toLocalDate(date);
		return YearMonth.of(localDate.getYear(), localDate.getMonthValue());
	}

	/**
	 * ZonedDateTime转YearMonth
	 * @param zonedDateTime ZonedDateTime
	 * @return YearMonth
	 */
	public static YearMonth toYearMonth(ZonedDateTime zonedDateTime) {
		LocalDate localDate = toLocalDate(zonedDateTime);
		return YearMonth.of(localDate.getYear(), localDate.getMonthValue());
	}

	/**
	 * Instant转YearMonth
	 * @param instant Instant
	 * @return YearMonth
	 */
	public static YearMonth toYearMonth(Instant instant) {
		LocalDate localDate = toLocalDate(instant);
		return YearMonth.of(localDate.getYear(), localDate.getMonthValue());
	}


	//------------------------------ 转YearMonth end ------------------------------


	//------------------------------ 转Date start ------------------------------

	/**
	 * LocalDateTime转Date
	 */
	public static Date toDate(LocalDateTime localDateTime) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		return Date.from(localDateTime.atZone(ZoneId.systemDefault()).toInstant());
	}

	/**
	 * LocalDate转Date
	 */
	public static Date toDate(LocalDate localDate) {
		Objects.requireNonNull(localDate, "localDate");
		return Date.from(localDate.atStartOfDay().atZone(ZoneId.systemDefault()).toInstant());
	}

	/**
	 * LocalTime转Date
	 */
	public static Date toDate(LocalDate localDate, LocalTime localTime) {
		Objects.requireNonNull(localDate, "localDate");
		Objects.requireNonNull(localTime, "localTime");
		LocalDateTime localDateTime = LocalDateTime.of(localDate, localTime);
		return Date.from(localDateTime.atZone(ZoneId.systemDefault()).toInstant());
	}

	/**
	 * LocalTime转Date
	 * 以当天的日期+LocalTime组成新的LocalDateTime转换为Date
	 * @param localTime LocalTime
	 * @return Date
	 */
	public static Date toDate(LocalTime localTime) {
		Objects.requireNonNull(localTime, "localTime");
		return Date.from(LocalDate.now().atTime(localTime).atZone(ZoneId.systemDefault()).toInstant());
	}

	/**
	 * YearMonth转Date
	 * 注意dayOfMonth范围：1到31之间，最大值根据月份确定特殊情况，如2月闰年29，非闰年28
	 * 如果要转换为当月最后一天，可以使用下面方法：yearMonthToDateEndOfMonth(YearMonth)
	 * @param yearMonth  YearMonth
	 * @param dayOfMonth 天
	 * @return Date
	 */
	public static Date toDate(YearMonth yearMonth, int dayOfMonth) {
		return toDate(yearMonth.atDay(dayOfMonth));
	}

	/**
	 * YearMonth转Date，转换为当月第一天
	 * @param yearMonth YearMonth
	 * @return Date
	 */
	public static Date toDateStartOfMonth(YearMonth yearMonth) {
		return toDate(yearMonth, 1);
	}

	/**
	 * YearMonth转Date，转换为当月最后一天
	 * @param yearMonth YearMonth
	 * @return Date
	 */
	public static Date toDateEndOfMonth(YearMonth yearMonth) {
		return toDate(yearMonth.atEndOfMonth());
	}

	/**
	 * 将LocalDate转java.sql.Date
	 */
	public static java.sql.Date toSqlDate(LocalDate date) {
		Objects.requireNonNull(date, "date");
		//LocalDate ---> java.util.Date
		Date tmp = toDate(date);
		// java.util.Date --->java.sql.Date
		return new java.sql.Date(tmp.getTime());
	}

	/**
	 * ZonedDateTime转Date
	 * 注意时间对应的时区和默认时区差异
	 * @param zonedDateTime ZonedDateTime
	 * @return Date
	 */
	public static Date toDate(ZonedDateTime zonedDateTime) {
		Objects.requireNonNull(zonedDateTime, "zonedDateTime");
		return Date.from(zonedDateTime.toInstant());
	}

	/**
	 * Instant转Date
	 * @param instant Instant
	 * @return Date
	 */
	public static Date toDate(Instant instant) {
		Objects.requireNonNull(instant, "instant");
		return Date.from(instant);
	}

	/**
	 * 时间戳epochMilli毫秒转Date
	 * @param epochMilli 时间戳
	 * @return Date
	 */
	public static Date toDate(long epochMilli) {
		return new Date(epochMilli);
	}


	//------------------------------ 转Date end ------------------------------


	//------------------------------ 转ZonedDateTime start ------------------------------

	/**
	 * LocalDateTime转ZonedDateTime，时区为系统默认时区
	 * @param localDateTime LocalDateTime
	 * @return ZonedDateTime
	 */
	public static ZonedDateTime toZonedDateTime(LocalDateTime localDateTime) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		return localDateTime.atZone(ZoneId.systemDefault());
	}

	/**
	 * LocalDateTime转ZonedDateTime，时区为zoneId对应时区<br>
	 * 注意，需要保证localDateTime和zoneId是对应的，不然会出现错误<br>
	 * 比如，localDateTime是巴黎时区下的对象，zoneId也应该是巴黎时区id<br>
	 * @param localDateTime LocalDateTime
	 * @param zoneId        时区id
	 * @return ZonedDateTime
	 */
	public static ZonedDateTime toZonedDateTime(LocalDateTime localDateTime, String zoneId) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		Objects.requireNonNull(zoneId, "zoneId");
		return localDateTime.atZone(ZoneId.of(zoneId));
	}

	/**
	 * LocalDate转ZonedDateTime，时区为系统默认时区
	 * @param localDate LocalDate
	 * @return ZonedDateTime such as 2020-02-19T00:00+08:00[Asia/Shanghai]
	 */
	public static ZonedDateTime toZonedDateTime(LocalDate localDate) {
		Objects.requireNonNull(localDate, "localDate");
		return localDate.atStartOfDay().atZone(ZoneId.systemDefault());
	}

	/**
	 * LocalTime转ZonedDateTime
	 * 以当天的日期+LocalTime组成新的ZonedDateTime，时区为系统默认时区
	 * @param localTime LocalTime
	 * @return ZonedDateTime
	 */
	public static ZonedDateTime toZonedDateTime(LocalTime localTime) {
		Objects.requireNonNull(localTime, "localTime");
		return LocalDate.now().atTime(localTime).atZone(ZoneId.systemDefault());
	}

	/**
	 * Date转ZonedDateTime，时区为系统默认时区
	 * @param date Date
	 * @return ZonedDateTime
	 */
	public static ZonedDateTime toZonedDateTime(Date date) {
		Objects.requireNonNull(date, "date");
		return Instant.ofEpochMilli(date.getTime()).atZone(ZoneId.systemDefault());
	}

	/**
	 * Date转ZonedDateTime，可以直接转换为对应的时区
	 * @param date   Date 没有时区区分
	 * @param zoneId 目标时区
	 * @return ZonedDateTime
	 */
	public static ZonedDateTime toZonedDateTime(Date date, String zoneId) {
		return toZonedDateTime(date, ZoneId.of(zoneId));
	}

	/**
	 * Date转ZonedDateTime
	 * @param date Date
	 * @param zone 目标时区
	 * @return ZonedDateTime
	 */
	public static ZonedDateTime toZonedDateTime(Date date, ZoneId zone) {
		Objects.requireNonNull(date, "date");
		Objects.requireNonNull(zone, "zone");
		return Instant.ofEpochMilli(date.getTime()).atZone(zone);
	}

	/**
	 * Instant转ZonedDateTime，时区为系统默认时区
	 * @param instant Instant
	 * @return ZonedDateTime
	 */
	public static ZonedDateTime toZonedDateTime(Instant instant) {
		Objects.requireNonNull(instant, "instant");
		return LocalDateTime.ofInstant(instant, ZoneId.systemDefault()).atZone(ZoneId.systemDefault());
	}

	/**
	 * 时间戳epochMilli毫秒转ZonedDateTime，时区为系统默认时区
	 * @param epochMilli 时间戳
	 * @return ZonedDateTime
	 */
	public static ZonedDateTime toZonedDateTime(long epochMilli) {
		return LocalDateTime.ofInstant(Instant.ofEpochMilli(epochMilli), ZoneId.systemDefault()).atZone(ZoneId.systemDefault());
	}
	//------------------------------ 转ZonedDateTime end ------------------------------


	//------------------------------ 转Instant start ------------------------------

	/**
	 * LocalDateTime转Instant
	 * @param localDateTime LocalDateTime
	 * @return Instant
	 */
	public static Instant toInstant(LocalDateTime localDateTime) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		return localDateTime.atZone(ZoneId.systemDefault()).toInstant();
	}

	/**
	 * LocalDate转Instant
	 * @param localDate LocalDate
	 * @return Instant
	 */
	public static Instant toInstant(LocalDate localDate) {
		return toLocalDateTime(localDate).atZone(ZoneId.systemDefault()).toInstant();
	}

	/**
	 * LocalTime转Instant
	 * 以当天的日期+LocalTime组成新的LocalDateTime转换为Instant
	 * @param localTime LocalTime
	 * @return Instant
	 */
	public static Instant toInstant(LocalTime localTime) {
		return toLocalDateTime(localTime).atZone(ZoneId.systemDefault()).toInstant();
	}

	/**
	 * Date转Instant
	 * @param date Date
	 * @return Instant
	 */
	public static Instant toInstant(Date date) {
		Objects.requireNonNull(date, "date");
		return date.toInstant();
	}

	/**
	 * ZonedDateTime转Instant
	 * 注意，zonedDateTime时区必须和当前系统时区一致，不然会出现问题
	 * @param zonedDateTime ZonedDateTime
	 * @return Instant
	 */
	public static Instant toInstant(ZonedDateTime zonedDateTime) {
		Objects.requireNonNull(zonedDateTime, "zonedDateTime");
		return zonedDateTime.toInstant();
	}

	/**
	 * 时间戳epochMilli毫秒转Instant
	 * @param epochMilli 时间戳
	 * @return Instant
	 */
	public static Instant toInstant(long epochMilli) {
		return Instant.ofEpochMilli(epochMilli);
	}

	/**
	 * Timestamp转Instant
	 * @param timestamp Timestamp
	 * @return Instant
	 */
	public static Instant toInstant(Timestamp timestamp) {
		Objects.requireNonNull(timestamp, "timestamp");
		return timestamp.toInstant();
	}

	//------------------------------ 转Instant end ------------------------------


	//------------------------------ 转时间戳 start ------------------------------

	/**
	 * LocalDateTime转时间戳
	 * 从1970-01-01T00:00:00Z开始的毫秒值
	 * @param localDateTime LocalDateTime
	 * @return 时间戳
	 */
	public static long toEpochMilli(LocalDateTime localDateTime) {
		return toInstant(localDateTime).toEpochMilli();
	}

	/**
	 * LocalDate转时间戳
	 * 从1970-01-01T00:00:00Z开始的毫秒值
	 * @param localDate LocalDate
	 * @return 时间戳
	 */
	public static long toEpochMilli(LocalDate localDate) {
		return toInstant(localDate).toEpochMilli();
	}

	/**
	 * LocalTime转时间戳
	 * 从1970-01-01T00:00:00Z开始的毫秒值
	 * 转换时会加上当天日期部分组成localDateTime再转换
	 * @param localTime localTime
	 * @return 时间戳
	 */
	public static long toEpochMilli(LocalTime localTime) {
		return toInstant(localTime).toEpochMilli();
	}

	/**
	 * Date转时间戳
	 * 从1970-01-01T00:00:00Z开始的毫秒值
	 * @param date Date
	 * @return 时间戳
	 */
	public static long toEpochMilli(Date date) {
		Objects.requireNonNull(date, "date");
		return date.getTime();
	}

	/**
	 * ZonedDateTime转时间戳，注意，zonedDateTime时区必须和当前系统时区一致，不然会出现问题
	 * 从1970-01-01T00:00:00Z开始的毫秒值
	 * @param zonedDateTime ZonedDateTime
	 * @return 时间戳
	 */
	public static long toEpochMilli(ZonedDateTime zonedDateTime) {
		Objects.requireNonNull(zonedDateTime, "zonedDateTime");
		return zonedDateTime.toInstant().toEpochMilli();
	}

	/**
	 * Instant转时间戳
	 * 从1970-01-01T00:00:00Z开始的毫秒值
	 * @param instant Instant
	 * @return 时间戳
	 */
	public static long toEpochMilli(Instant instant) {
		Objects.requireNonNull(instant, "instant");
		return instant.toEpochMilli();
	}

	/**
	 * Timestamp转时间戳
	 * 从1970-01-01T00:00:00Z开始的毫秒值
	 * @param timestamp Timestamp
	 * @return 时间戳
	 */
	public static long toEpochMilli(Timestamp timestamp) {
		Objects.requireNonNull(timestamp, "timestamp");
		return timestamp.getTime();
	}

	//------------------------------ 转时间戳 end ------------------------------


	//------------------------------ 转Timestamp start ------------------------------

	/**
	 * LocalDateTime转Timestamp
	 * @param localDateTime LocalDateTime
	 * @return Timestamp
	 */
	public static Timestamp toTimestamp(LocalDateTime localDateTime) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		return Timestamp.valueOf(localDateTime);
	}

	/**
	 * Date转Timestamp
	 * @param date Date
	 * @return Timestamp
	 */
	public static Timestamp toTimestamp(Date date) {
		Objects.requireNonNull(date, "date");
		return new Timestamp(date.getTime());
	}

	/**
	 * ZonedDateTime转Timestamp
	 * @param zonedDateTime ZonedDateTime
	 * @return Timestamp
	 */
	public static Timestamp toTimestamp(ZonedDateTime zonedDateTime) {
		return toTimestamp(toLocalDateTime(zonedDateTime));
	}

	/**
	 * Instant转Timestamp
	 * @param instant Instant
	 * @return Timestamp
	 */
	public static Timestamp toTimestamp(Instant instant) {
		Objects.requireNonNull(instant, "instant");
		return Timestamp.from(instant);
	}

	/**
	 * 时间戳epochMilli转Timestamp
	 * @param epochMilli 时间戳
	 * @return Timestamp
	 */
	public static Timestamp toTimestamp(long epochMilli) {
		return new Timestamp(epochMilli);
	}

	//------------------------------ 转Timestamp end ------------------------------


	//------------------------------ 转other start ------------------------------

	/**
	 * localTime转时间部分的毫秒
	 * @param localTime localTime
	 * @return 时间部分的毫秒
	 */
	public static int toTimeMilli(LocalTime localTime) {
		Objects.requireNonNull(localTime, "localTime");
		return (int) (localTime.toNanoOfDay() / 1000_000);
	}

	//------------------------------ 转other end ------------------------------


	//------------------------------ 转字符串 ------------------------------

	/**
	 * 获取日期时间字符串
	 * @param temporal 需要转化的日期时间
	 * @return String 日期时间字符串，例如 2023-01-20 15:56:53
	 */
	public static String format(TemporalAccessor temporal) {
		Objects.requireNonNull(temporal, "temporal");
		return DateTimeFormatter.ofPattern(DATETIME_PATTERN).format(temporal);
	}

	/**
	 * 获取日期时间字符串
	 * @param temporal 需要转化的日期时间
	 * @param pattern  时间格式
	 * @return String 日期时间字符串，例如 2023-01-20 15:56:53
	 */
	public static String format(TemporalAccessor temporal, String pattern) {
		Objects.requireNonNull(temporal, "temporal");
		Objects.requireNonNull(pattern, "pattern");
		return DateTimeFormatter.ofPattern(pattern).format(temporal);
	}

	/**
	 * 获取偏移今天的时间
	 * @param offSet  负数表示前几天，正数表示后几天
	 * @param pattern 时间格式化类型
	 * @return 格式化后的结果
	 */
	public static String getOffSetLocalDateTimeStr(Integer offSet, String pattern) {
		Objects.requireNonNull(offSet, "offSet");
		Objects.requireNonNull(pattern, "pattern");
		return LocalDateTime.now().plusDays(offSet).format(DateTimeFormatter.ofPattern(pattern));
	}


	//------------------------------ 特殊的一些方法 ------------------------------

	/**
	 * 获取和当前日期存在的日期差
	 * @param localDate
	 * @return
	 */
	public static Period getDateDuration(LocalDate localDate) {
		Objects.requireNonNull(localDate, "localDate");
		Period period = Period.between(localDate, LocalDate.now());
		/*System.out.println(period.getYears());
		System.out.println(period.getMonths());
		System.out.println(period.getDays());*/
		return period;
	}

	/**
	 * 获取和当前时间存在的时间差
	 * @param localTime
	 * @return
	 */
	public static Duration getTimeDuration(LocalTime localTime) {
		Objects.requireNonNull(localTime, "localTime");
		Duration duration = Duration.between(LocalTime.now(), localTime);
		/*System.out.println(duration.toDays());
		System.out.println(duration.toHours());
		System.out.println(duration.toMinutes());
		System.out.println(duration.toMillis());*/
		return duration;
	}

	/**
	 * 获取日历表
	 * @param index -1：上月，0：当月，1：下月
	 * @return 结果集
	 */
	public static List<DataCalendarDTO> getCalendarTable(Integer index) {

		//获取当前月的第一天
		LocalDate currentMonthFirstDay = getFirstDayOfMonth(index);
		//获取当前月的最后一天
		LocalDate currentMonthEndDay = getLastDayOfMonth(index);

		List<DataCalendarDTO> localDateList = Lists.newArrayList();
		long length = currentMonthEndDay.toEpochDay() - currentMonthFirstDay.toEpochDay();
		for (long i = length; i >= 0; i--) {
			DataCalendarDTO dataCalendarDTO = new DataCalendarDTO();
			//当前日期
			LocalDate localDate = currentMonthEndDay.minusDays(i);
			dataCalendarDTO.setDate(localDate.format(DateTimeFormatter.ofPattern(DATE_PATTERN)));
			//周几
			dataCalendarDTO.setWeek(localDate.getDayOfWeek().getValue());
			localDateList.add(dataCalendarDTO);
		}
		return localDateList;
	}

	/**
	 * 获取某个月的第一天
	 * @param index -1：上月，0：当月，1：下月
	 * @return 几号
	 */
	public static LocalDate getFirstDayOfMonth(Integer index) {
		Objects.requireNonNull(index, "index");
		return LocalDate.now().plusMonths(index).with(TemporalAdjusters.firstDayOfMonth());
	}

	/**
	 * 获取某个月的最后一天
	 * @param index -1：上月，0：当月，1：下月
	 * @return 几号
	 */
	public static LocalDate getLastDayOfMonth(Integer index) {
		Objects.requireNonNull(index, "index");
		return LocalDate.now().plusMonths(index).with(TemporalAdjusters.lastDayOfMonth());
	}

	/**
	 * @param start 开始日期
	 * @param end   结束日期
	 * @return 开始与结束之间的所有日期，包括起止
	 */
	public static List<LocalDate> getMiddleAllDate(LocalDate start, LocalDate end) {
		Objects.requireNonNull(start, "start");
		Objects.requireNonNull(end, "begin");
		List<LocalDate> localDateList = Lists.newArrayList();
		long length = end.toEpochDay() - start.toEpochDay();
		for (long i = length; i >= 0; i--) {
			localDateList.add(end.minusDays(i));
		}
		return localDateList;
	}

	/**
	 * 切割日期（按照周期切割成小段日期段）
	 * @param startDate 开始日期（yyyy-MM-dd）
	 * @param endDate   结束日期（yyyy-MM-dd）
	 * @param period    周期（天，周，月，年）
	 * @return 切割之后的日期集合
	 * <li>startDate="2019-02-28",endDate="2019-03-05",period="day"</li>
	 * <li>结果为：[2019-02-28, 2019-03-01, 2019-03-02, 2019-03-03, 2019-03-04, 2019-03-05]</li>
	 * <li>startDate="2019-02-28",endDate="2019-03-25",period="week"</li>
	 * <li>结果为：[2019-02-28,2019-03-06, 2019-03-07,2019-03-13, 2019-03-14,2019-03-20, 2019-03-21,2019-03-25]</li>
	 * <li>startDate="2019-02-28",endDate="2019-05-25",period="month"</li>
	 * <li>结果为：[2019-02-28,2019-02-28, 2019-03-01,2019-03-31, 2019-04-01,2019-04-30, 2019-05-01,2019-05-25]</li>
	 * <li>startDate="2019-02-28",endDate="2020-05-25",period="year"</li>
	 * <li>结果为：[2019-02-28,2019-12-31, 2020-01-01,2020-05-25]</li>
	 */
	public static List<String> listDateStrs(String startDate, String endDate, String period) {
		Objects.requireNonNull(startDate, "startDate");
		Objects.requireNonNull(endDate, "endDate");
		List<String> result = Lists.newArrayList();
		DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern(DATE_PATTERN);
		LocalDate end = LocalDate.parse(endDate, dateTimeFormatter);
		LocalDate start = LocalDate.parse(startDate, dateTimeFormatter);
		LocalDate tmp = start;
		switch (period) {
			case DAY:
				while (start.isBefore(end) || start.isEqual(end)) {
					result.add(start.toString());
					start = start.plusDays(1);
				}
				break;
			case WEEK:
				while (tmp.isBefore(end) || tmp.isEqual(end)) {
					if (tmp.plusDays(6).isAfter(end)) {
						result.add(tmp + "," + end);
					} else {
						result.add(tmp + "," + tmp.plusDays(6));
					}
					tmp = tmp.plusDays(7);
				}
				break;
			case MONTH:
				while (tmp.isBefore(end) || tmp.isEqual(end)) {
					LocalDate lastDayOfMonth = tmp.with(TemporalAdjusters.lastDayOfMonth());
					if (lastDayOfMonth.isAfter(end)) {
						result.add(tmp + "," + end);
					} else {
						result.add(tmp + "," + lastDayOfMonth);
					}
					tmp = lastDayOfMonth.plusDays(1);
				}
				break;
			case YEAR:
				while (tmp.isBefore(end) || tmp.isEqual(end)) {
					LocalDate lastDayOfYear = tmp.with(TemporalAdjusters.lastDayOfYear());
					if (lastDayOfYear.isAfter(end)) {
						result.add(tmp + "," + end);
					} else {
						result.add(tmp + "," + lastDayOfYear);
					}
					tmp = lastDayOfYear.plusDays(1);
				}
				break;
			default:
				break;
		}
		return result;
	}

	/**
	 * 获取指定日期时间加上指定数量日期时间单位之后的日期时间.
	 * @param localDateTime 日期时间
	 * @param num           数量
	 * @param chronoUnit    日期时间单位
	 * @return LocalDateTime 新的日期时间
	 */
	public static LocalDateTime plus(LocalDateTime localDateTime, int num, ChronoUnit chronoUnit) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		Objects.requireNonNull(chronoUnit, "chronoUnit");
		return localDateTime.plus(num, chronoUnit);
	}

	/**
	 * 获取指定日期时间减去指定数量日期时间单位之后的日期时间.
	 * @param localDateTime 日期时间
	 * @param num           数量
	 * @param chronoUnit    日期时间单位
	 * @return LocalDateTime 新的日期时间
	 */
	public static LocalDateTime minus(LocalDateTime localDateTime, int num, ChronoUnit chronoUnit) {
		Objects.requireNonNull(localDateTime, "localDateTime");
		Objects.requireNonNull(chronoUnit, "chronoUnit");
		return localDateTime.minus(num, chronoUnit);
	}

	/**
	 * 根据ChronoUnit计算两个日期时间之间相隔日期时间
	 * @param start      开始日期时间
	 * @param end        结束日期时间
	 * @param chronoUnit 日期时间单位
	 * @return long 相隔日期时间
	 */
	public static long getChronoUnitBetween(LocalDateTime start, LocalDateTime end, ChronoUnit chronoUnit) {
		Objects.requireNonNull(start, "start");
		Objects.requireNonNull(end, "end");
		Objects.requireNonNull(chronoUnit, "chronoUnit");
		return Math.abs(start.until(end, chronoUnit));
	}

	/**
	 * 根据ChronoUnit计算两个日期之间相隔年数或月数或天数
	 * @param start      开始日期
	 * @param end        结束日期
	 * @param chronoUnit 日期时间单位,(ChronoUnit.YEARS,ChronoUnit.MONTHS,ChronoUnit.WEEKS,ChronoUnit.DAYS)
	 * @return long 相隔年数或月数或天数
	 */
	public static long getChronoUnitBetween(LocalDate start, LocalDate end, ChronoUnit chronoUnit) {
		Objects.requireNonNull(start, "start");
		Objects.requireNonNull(end, "end");
		Objects.requireNonNull(chronoUnit, "chronoUnit");
		return Math.abs(start.until(end, chronoUnit));
	}

	/**
	 * 时间截断
	 * @param localTime 时间 格式：11:55:13
	 * @return LocalTime 格式：11:55
	 */
	public static LocalTime truncatedTo(LocalTime localTime) {
		Objects.requireNonNull(localTime, "localTime");
		return localTime.truncatedTo(ChronoUnit.MINUTES);
	}

	//------------------------------ 单位转换 ------------------------------

	/**
	 * 单位转换
	 * @param sourceDuration 数量
	 * @param sourceUnit     原单位
	 * @param targetUnit     新单位
	 * @return 返回结果
	 */
	public static long unitConversion(long sourceDuration, TimeUnit sourceUnit, TimeUnit targetUnit) {
		return targetUnit.convert(sourceDuration, sourceUnit);
	}

	/**
	 * 单位转换，精确计算
	 * @param sourceDuration 数量
	 * @param sourceUnit     原单位
	 * @param targetUnit     新单位
	 * @param scale          小数位数
	 * @param roundingMode   舍入模式
	 * @return 返回结果
	 */
	public static BigDecimal unitConversionPrecise(long sourceDuration, TimeUnit sourceUnit, TimeUnit targetUnit, int scale, RoundingMode roundingMode) {
		return BigDecimal.valueOf(sourceDuration)
				.multiply(BigDecimal.valueOf(sourceUnit.toNanos(1))).divide(BigDecimal.valueOf(targetUnit.toNanos(1)), scale, roundingMode);
	}

	/**
	 * 单位转换，精确计算，返回字符串，去掉尾部多余0
	 * @param sourceDuration 数量
	 * @param sourceUnit     原单位
	 * @param targetUnit     新单位
	 * @param scale          小数位数
	 * @param roundingMode   舍入模式
	 * @return 返回结果
	 */
	public static String unitConversionPreciseString(long sourceDuration, TimeUnit sourceUnit, TimeUnit targetUnit, int scale, RoundingMode roundingMode) {
		return unitConversionPrecise(sourceDuration, sourceUnit, targetUnit, scale, roundingMode).stripTrailingZeros().toPlainString();
	}

	/**
	 * 单位转换，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数
	 * @param sourceDuration 数量
	 * @param sourceUnit     原单位
	 * @param targetUnit     新单位
	 * @return 返回结果，保留1位小数，ROUND_DOWN 舍去多余小数
	 */
	public static BigDecimal unitConversionPrecise(long sourceDuration, TimeUnit sourceUnit, TimeUnit targetUnit) {
		return unitConversionPrecise(sourceDuration, sourceUnit, targetUnit, 1, RoundingMode.DOWN);
	}

	/**
	 * 单位转换，天转小时
	 * @param num 待转换值
	 * @return 小时数
	 */
	public static long dayToHour(long num) {
		return unitConversion(num, TimeUnit.DAYS, TimeUnit.HOURS);
	}

	/**
	 * 单位转换，天转分钟
	 * @param num 待转换值
	 * @return 分钟数
	 */
	public static long dayToMinute(long num) {
		return unitConversion(num, TimeUnit.DAYS, TimeUnit.MINUTES);
	}

	/**
	 * 单位转换，天转秒
	 * @param num 待转换值
	 * @return 秒数
	 */
	public static long dayToSecond(long num) {
		return unitConversion(num, TimeUnit.DAYS, TimeUnit.SECONDS);
	}

	/**
	 * 单位转换，天转毫秒
	 * @param num 待转换值
	 * @return 毫秒数
	 */
	public static long dayToMillisecond(long num) {
		return unitConversion(num, TimeUnit.DAYS, TimeUnit.MILLISECONDS);
	}

	/**
	 * 单位转换，小时转分钟
	 * @param num 待转换值
	 * @return 分钟数
	 */
	public static long hourToMinute(long num) {
		return unitConversion(num, TimeUnit.HOURS, TimeUnit.MINUTES);
	}

	/**
	 * 单位转换，小时转秒
	 * @param num 待转换值
	 * @return 秒数
	 */
	public static long hourToSecond(long num) {
		return unitConversion(num, TimeUnit.HOURS, TimeUnit.SECONDS);
	}

	/**
	 * 单位转换，小时转毫秒
	 * @param num 待转换值
	 * @return 毫秒数
	 */
	public static long hourToMillisecond(long num) {
		return unitConversion(num, TimeUnit.HOURS, TimeUnit.MILLISECONDS);
	}

	/**
	 * 单位转换，小时转天
	 * @param num 待转换值
	 * @return 天数
	 */
	public static long hourToDay(long num) {
		return unitConversion(num, TimeUnit.HOURS, TimeUnit.DAYS);
	}

	/**
	 * 小时转天，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数
	 * @param num 待转换值
	 * @return 天数，保留1位小数，ROUND_DOWN 舍去多余小数
	 */
	public static BigDecimal hourToDayPrecise(long num) {
		return unitConversionPrecise(num, TimeUnit.HOURS, TimeUnit.DAYS);
	}

	/**
	 * 小时转天，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数 ，返回字符串，去掉尾部多余0
	 * @param num 待转换值
	 * @return 天数，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 */
	public static String hourToDayPreciseString(long num) {
		return hourToDayPrecise(num).stripTrailingZeros().toPlainString();
	}

	/**
	 * 单位转换，分钟转秒
	 * @param num 待转换值
	 * @return 秒数
	 */
	public static long minuteToSecond(long num) {
		return unitConversion(num, TimeUnit.MINUTES, TimeUnit.SECONDS);
	}

	/**
	 * 单位转换，分钟转毫秒
	 * @param num 待转换值
	 * @return 毫秒数
	 */
	public static long minuteToMillisecond(long num) {
		return unitConversion(num, TimeUnit.MINUTES, TimeUnit.MILLISECONDS);
	}

	/**
	 * 单位转换，分钟转小时
	 * @param num 待转换值
	 * @return 小时数
	 */
	public static long minuteToHour(long num) {
		return unitConversion(num, TimeUnit.MINUTES, TimeUnit.HOURS);
	}

	/**
	 * 分钟转小时，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数
	 * @param num 待转换值
	 * @return 小时数，保留1位小数，ROUND_DOWN 舍去多余小数
	 */
	public static BigDecimal minuteToHourPrecise(long num) {
		return unitConversionPrecise(num, TimeUnit.MINUTES, TimeUnit.HOURS);
	}

	/**
	 * 分钟转小时，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 * @param num 待转换值
	 * @return 小时数，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 */
	public static String minuteToHourPreciseString(long num) {
		return minuteToHourPrecise(num).stripTrailingZeros().toPlainString();
	}

	/**
	 * 单位转换，分钟转天
	 * @param num 待转换值
	 * @return 天数
	 */
	public static long minuteToDay(long num) {
		return unitConversion(num, TimeUnit.MINUTES, TimeUnit.DAYS);
	}

	/**
	 * 分钟转天，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数
	 * @param num 待转换值
	 * @return 天数，保留1位小数，ROUND_DOWN 舍去多余小数
	 */
	public static BigDecimal minuteToDayPrecise(long num) {
		return unitConversionPrecise(num, TimeUnit.MINUTES, TimeUnit.DAYS);
	}

	/**
	 * 分钟转天，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 * @param num 待转换值
	 * @return 天数，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 */
	public static String minuteToDayPreciseString(long num) {
		return minuteToDayPrecise(num).stripTrailingZeros().toPlainString();
	}

	/**
	 * 单位转换，秒转毫秒
	 * @param num 待转换值
	 * @return 毫秒数
	 */
	public static long secondToMillisecond(long num) {
		return unitConversion(num, TimeUnit.SECONDS, TimeUnit.MILLISECONDS);
	}

	/**
	 * 单位转换，秒转分钟
	 * @param num 待转换值
	 * @return 分钟数
	 */
	public static long secondToMinute(long num) {
		return unitConversion(num, TimeUnit.SECONDS, TimeUnit.MINUTES);
	}

	/**
	 * 秒转分钟，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数
	 * @param num 待转换值
	 * @return 分钟数，保留1位小数，ROUND_DOWN 舍去多余小数
	 */
	public static BigDecimal secondToMinutePrecise(long num) {
		return unitConversionPrecise(num, TimeUnit.SECONDS, TimeUnit.MINUTES);
	}

	/**
	 * 秒转分钟，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 * @param num 待转换值
	 * @return 分钟数，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 */
	public static String secondToMinutePreciseString(long num) {
		return secondToMinutePrecise(num).stripTrailingZeros().toPlainString();
	}

	/**
	 * 单位转换，秒转小时
	 * @param num 待转换值
	 * @return 小时数
	 */
	public static long secondToHour(long num) {
		return unitConversion(num, TimeUnit.SECONDS, TimeUnit.HOURS);
	}

	/**
	 * 秒转小时，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数
	 * @param num 待转换值
	 * @return 小时数，保留1位小数，ROUND_DOWN 舍去多余小数
	 */
	public static BigDecimal secondToHourPrecise(long num) {
		return unitConversionPrecise(num, TimeUnit.SECONDS, TimeUnit.HOURS);
	}

	/**
	 * 秒转小时，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 * @param num 待转换值
	 * @return 小时数，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 */
	public static String secondToHourPreciseString(long num) {
		return secondToHourPrecise(num).stripTrailingZeros().toPlainString();
	}

	/**
	 * 单位转换，秒转天
	 * @param num 待转换值
	 * @return 天数
	 */
	public static long secondToDay(long num) {
		return unitConversion(num, TimeUnit.SECONDS, TimeUnit.DAYS);
	}

	/**
	 * 秒转天，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数
	 * @param num 待转换值
	 * @return 天数，保留1位小数，ROUND_DOWN 舍去多余小数
	 */
	public static BigDecimal secondToDayPrecise(long num) {
		return unitConversionPrecise(num, TimeUnit.SECONDS, TimeUnit.DAYS);
	}

	/**
	 * 秒转天，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 * @param num 待转换值
	 * @return 天数，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 */
	public static String secondToDayPreciseString(long num) {
		return secondToDayPrecise(num).stripTrailingZeros().toPlainString();
	}

	/**
	 * 单位转换，毫秒转秒
	 * @param num 待转换值
	 * @return 秒数
	 */
	public static long millisecondToSecond(long num) {
		return unitConversion(num, TimeUnit.MILLISECONDS, TimeUnit.SECONDS);
	}

	/**
	 * 毫秒转秒，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数
	 * @param num 待转换值
	 * @return 秒数，保留1位小数，ROUND_DOWN 舍去多余小数
	 */
	public static BigDecimal millisecondToSecondPrecise(long num) {
		return unitConversionPrecise(num, TimeUnit.MILLISECONDS, TimeUnit.SECONDS);
	}

	/**
	 * 毫秒转秒，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 * @param num 待转换值
	 * @return 秒数，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 */
	public static String millisecondToSecondPreciseString(long num) {
		return millisecondToSecondPrecise(num).stripTrailingZeros().toPlainString();
	}

	/**
	 * 毫秒转秒，精确计算，保留3位小数，ROUND_DOWN 舍去多余小数
	 * @param num 待转换值
	 * @return 秒数，保留3位小数，ROUND_DOWN 舍去多余小数
	 */
	public static BigDecimal millisecondToSecondPreciseThree(long num) {
		return unitConversionPrecise(num, TimeUnit.MILLISECONDS, TimeUnit.SECONDS, 3, RoundingMode.DOWN);
	}

	/**
	 * 毫秒转秒，精确计算，保留3位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 * @param num 待转换值
	 * @return 秒数，保留3位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0 比如 0.11
	 */
	public static String millisecondToSecondPreciseThreeString(long num) {
		return millisecondToSecondPreciseThree(num).stripTrailingZeros().toPlainString();
	}

	/**
	 * 毫秒转秒，精确计算，保留3位小数，ROUND_DOWN 舍去多余小数，返回原始字符串
	 * @param num 待转换值
	 * @return 秒数，保留3位小数，ROUND_DOWN 舍去多余小数，返回原始字符串 比如 0.110
	 */
	public static String millisecondToSecondPreciseThreePlainString(long num) {
		return millisecondToSecondPreciseThree(num).toPlainString();
	}

	/**
	 * 单位转换，毫秒转分钟
	 * @param num 待转换值
	 * @return 分钟数
	 */
	public static long millisecondToMinute(long num) {
		return unitConversion(num, TimeUnit.MILLISECONDS, TimeUnit.MINUTES);
	}

	/**
	 * 毫秒转分钟，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数
	 * @param num 待转换值
	 * @return 分钟数，保留1位小数，ROUND_DOWN 舍去多余小数
	 */
	public static BigDecimal millisecondToMinutePrecise(long num) {
		return unitConversionPrecise(num, TimeUnit.MILLISECONDS, TimeUnit.MINUTES);
	}

	/**
	 * 毫秒转分钟，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数
	 * @param num 待转换值
	 * @return 分钟数，保留1位小数，ROUND_DOWN 舍去多余小数
	 */
	public static String millisecondToMinutePreciseString(long num) {
		return millisecondToMinutePrecise(num).stripTrailingZeros().toPlainString();
	}

	/**
	 * 单位转换，毫秒转小时
	 * @param num 待转换值
	 * @return 小时数
	 */
	public static long millisecondToHour(long num) {
		return unitConversion(num, TimeUnit.MILLISECONDS, TimeUnit.HOURS);
	}

	/**
	 * 毫秒转小时，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数
	 * @param num 待转换值
	 * @return 小时数，保留1位小数，ROUND_DOWN 舍去多余小数
	 */
	public static BigDecimal millisecondToHourPrecise(long num) {
		return unitConversionPrecise(num, TimeUnit.MILLISECONDS, TimeUnit.HOURS);
	}

	/**
	 * 毫秒转小时，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 * @param num 待转换值
	 * @return 小时数，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 */
	public static String millisecondToHourPreciseString(long num) {
		return millisecondToHourPrecise(num).stripTrailingZeros().toPlainString();
	}

	/**
	 * 单位转换，毫秒转天
	 * @param num 待转换值
	 * @return 天数
	 */
	public static long millisecondToDay(long num) {
		return unitConversion(num, TimeUnit.MILLISECONDS, TimeUnit.DAYS);
	}

	/**
	 * 毫秒转天，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数
	 * @param num 待转换值
	 * @return 天数，保留1位小数，ROUND_DOWN 舍去多余小数
	 */
	public static BigDecimal millisecondToDayPrecise(long num) {
		return unitConversionPrecise(num, TimeUnit.MILLISECONDS, TimeUnit.DAYS);
	}

	/**
	 * 毫秒转天，精确计算，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 * @param num 待转换值
	 * @return 天数，保留1位小数，ROUND_DOWN 舍去多余小数，返回字符串，去掉尾部多余0
	 */
	public static String millisecondToDayPreciseString(long num) {
		return millisecondToDayPrecise(num).stripTrailingZeros().toPlainString();
	}

}
