package cn.jque.core.util;

import cn.jque.core.model.ChromelessEnum;
import io.webfolder.cdp.Launcher;
import io.webfolder.cdp.session.Session;
import io.webfolder.cdp.session.SessionFactory;
import lombok.extern.slf4j.Slf4j;

import java.io.File;
import java.io.IOException;

import static java.nio.file.Files.write;
import static java.util.Arrays.asList;

@Slf4j
public class Chromeless2PdfUtils {

	private static final String FILE_PREFIX = "file://";
	private static Launcher launcher;
	private static SessionFactory factory;

	static {
		try {
			//google浏览器初始化
			launcher = new Launcher();
			System.out.println(String.format("当前系统的chrome命令：%s", launcher.findChrome()));
			factory = launcher.launch(asList("--headless", "--disable-gpu", "--no-sandbox"));
			System.out.println(String.format("Chromeless2Pdf 初始化完成, 端口：%d", factory.getPort()));
		} catch (Exception e) {
			log.info(String.format("初始化失败：%s", e.getMessage()), e);
		}
	}

	/**
	 * HTML文件转PDF
	 * @param src        HTML文件
	 * @param dest       PDF文件
	 * @param pageLayout 页面布局
	 */
	public static void transToPdf(File src, File dest, String pageLayout) throws IOException {
		ChromelessEnum layout = ChromelessEnum.valueOf(pageLayout);
		String context;

		//增加出错重试机制，增强功能性
		try {
			context = factory.createBrowserContext();
		} catch (Exception e) {
			retryInitFactory();
			context = factory.createBrowserContext();
		}

		try (Session session = factory.create(context)) {
			session.navigate(FILE_PREFIX + src.getAbsolutePath());
			session.waitDocumentReady();
			byte[] content = session
					.getCommand()
					.getPage()
					.printToPDF(layout.isLandscape(), false, true, 1D,
							layout.getPaperWidth(), layout.getPaperHeight(),
							0D, 0D, 0D, 0D,
							null, null, null, null, null);
			write(dest.toPath(), content);
		}
		factory.disposeBrowserContext(context);
	}

	/**
	 * 避免由于启动没有初始化google插件的问题，进行重试操作，重试3次
	 */
	private static void retryInitFactory() {
		int retryTime = 0;
		int RETRY_TIME = 3;
		while (retryTime < RETRY_TIME) {
			//首先关闭所有连接，重新初始化
			if (factory != null) {
				factory.close();
			}
			if (launcher != null) {
				launcher.kill();
			}
			try {
				//google浏览器初始化
				launcher = new Launcher();
				System.out.println(String.format("当前系统的chrome命令：%s", launcher.findChrome()));
				factory = launcher.launch(asList("--headless", "--disable-gpu"));
				System.out.println(String.format("Chromeless2Pdf 初始化完成, 端口：%d", factory.getPort()));

				retryTime++;

			} catch (Exception e) {
				log.error(String.format("重试：%s", e.getMessage()), e);

			}
			if (factory != null) {
				break;
			}
		}
	}

	public void destroy() {
		log.info("Chromeless2Pdf Destroying...");
		//google浏览器关闭
		factory.close();
		launcher.kill();
	}
}
