package cn.jque.core.util;

import org.springframework.beans.BeanUtils;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.cglib.core.Converter;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Supplier;

/**
 * 对象转换工具
 * @author Liu.ShiTing
 * @version 1.0.0
 * @date 2022-07-05 14:42:36
 */
public class BeanConvertUtils extends BeanUtils {


	/**
	 * cglib实现对象转换（无返回值）
	 * @param source 源对象
	 * @param target 目标对象
	 * @param <S>    源对象类型
	 * @param <T>    目标对象类型
	 */
	public static <S, T> void convertTo(S source, T target) {
		BeanCopier beanCopier = BeanCopier.create(source.getClass(), target.getClass(), false);
		beanCopier.copy(source, target, null);
	}

	/**
	 * cglib实现对象转换（无返回值）
	 * @param source 源对象
	 * @param target 目标对象
	 * @param converter 不为空，表示会开启Convert，可自定义Convert类，实现Converter接口（支持同名不同类型的属性拷贝）
	 *                  https://blog.csdn.net/weixin_40170876/article/details/81475611
	 * @param <S>    源对象类型
	 * @param <T>    目标对象类型
	 */
	public static <S, T> void convertToo(S source, T target, Converter converter) {
		BeanCopier beanCopier;
		if (converter == null) {
			beanCopier = BeanCopier.create(source.getClass(), target.getClass(), false);
		} else {
			beanCopier = BeanCopier.create(source.getClass(), target.getClass(), true);
		}
		beanCopier.copy(source, target, converter);
	}

	/**
	 * 对象转换（有返回值）
	 * @param source         源对象
	 * @param targetSupplier 目标对象供应方
	 * @param <S>            源对象类型
	 * @param <T>            目标对象类型
	 * @return 目标对象
	 */
	public static <S, T> T convertTo(S source, Supplier<T> targetSupplier) {
		return convertTo(source, targetSupplier, null);
	}

	/**
	 * 转换对象（有返回值，带回调）
	 * @param source         源对象
	 * @param targetSupplier 目标对象供应方
	 * @param callBack       回调方法
	 * @param <S>            源对象类型
	 * @param <T>            目标对象类型
	 * @return 目标对象
	 */
	public static <S, T> T convertTo(S source, Supplier<T> targetSupplier, ConvertCallBack<S, T> callBack) {
		if (null == source || null == targetSupplier) {
			return null;
		}

		T target = targetSupplier.get();
		copyProperties(source, target);
		if (callBack != null) {
			callBack.callBack(source, target);
		}
		return target;
	}

	public static <S, T> List<T> convertListTo(List<S> sources, Supplier<T> targetSupplier) {
		return convertListTo(sources, targetSupplier, null);
	}

	/**
	 * List转换（有返回值，带回调）
	 * @param sources        源对象list
	 * @param targetSupplier 目标对象供应方
	 * @param callBack       回调方法
	 * @param <S>            源对象类型
	 * @param <T>            目标对象类型
	 * @return 目标对象list
	 */
	public static <S, T> List<T> convertListTo(List<S> sources, Supplier<T> targetSupplier, ConvertCallBack<S, T> callBack) {
		if (null == sources || null == targetSupplier) {
			return null;
		}

		List<T> list = new ArrayList<>(sources.size());
		for (S source : sources) {
			T target = targetSupplier.get();
			copyProperties(source, target);
			if (callBack != null) {
				callBack.callBack(source, target);
			}
			list.add(target);
		}
		return list;
	}

	/**
	 * 回调接口
	 * @param <S> 源对象类型
	 * @param <T> 目标对象类型
	 */
	@FunctionalInterface
	public interface ConvertCallBack<S, T> {
		void callBack(S t, T s);
	}

}
