package cn.jque.common.base;

import cn.jque.common.enums.ResponseEnum;
import lombok.Data;

import java.io.Serializable;

/**
 * @author Liu.ShiTing
 * @version 1.0.0
 * @date 2022-09-13 19:14:07
 */
@Data
public class Response<T> implements Serializable {

	private static final long serialVersionUID = 8236647872062931279L;

	/**
	 * 状态码
	 */
	private int code;

	/**
	 * 提示信息
	 */
	private String message;

	/**
	 * 返回数据
	 */
	private T data;

	/**
	 * 返回数据元素的个数
	 */
	private Integer total;


	/**
	 * 成功操作，携带默认消息
	 * @param <T>
	 * @return
	 */
	public static <T> Response<T> success() {
		return success(ResponseEnum.SUCCESS.getMessage());
	}

	/**
	 * 成功操作，携带成功状态码、自定义消息
	 * @param message
	 * @param <T>
	 * @return
	 */
	public static <T> Response<T> success(String message) {
		return success(message, null);
	}

	/**
	 * 成功操作，携带成功状态码、默认消息、数据
	 * @param data
	 * @param <T>
	 * @return
	 */
	public static <T> Response<T> success(T data) {
		return success(ResponseEnum.SUCCESS.getMessage(), data);
	}

	/**
	 * 成功操作，携带成功状态码、自定义消息、数据和数据集中元素的个数
	 * @param message
	 * @param data
	 * @param total
	 * @param <T>
	 * @return
	 */
	public static <T> Response<T> success(String message, T data, Integer total) {
		return success(ResponseEnum.SUCCESS.getCode(), message, data, total);
	}

	/**
	 * 成功操作，携带成功状态码、自定义消息、数据
	 * @param message
	 * @param data
	 * @param <T>
	 * @return
	 */
	public static <T> Response<T> success(String message, T data) {
		return success(ResponseEnum.SUCCESS.getCode(), message, data, null);
	}

	/**
	 * 成功操作，携带自定义成功状态码、自定义消息
	 * @param code
	 * @param message
	 * @param <T>
	 * @return
	 */
	public static <T> Response<T> success(int code, String message) {
		return success(code, message, null, null);
	}

	/**
	 * 成功操作，携带成功状态码、自定义消息、数据和数据集中元素的个数
	 * @param code
	 * @param message
	 * @param data
	 * @param total
	 * @param <T>
	 * @return
	 */
	public static <T> Response<T> success(int code, String message, T data, Integer total) {
		Response<T> result = new Response<>();
		result.setCode(code);
		result.setMessage(message);
		result.setData(data);
		result.setTotal(total);
		return result;
	}

	/**
	 * 失败操作，携带默认数据
	 * @param <T>
	 * @return
	 */
	public static <T> Response<T> failure() {
		return failure(ResponseEnum.FAILURE.getMessage());
	}

	/**
	 * 失败操作，携带失败状态码、自定义消息
	 * @param message
	 * @param <T>
	 * @return
	 */
	public static <T> Response<T> failure(String message) {
		return failure(message, null);
	}

	/**
	 * 失败操作，携带失败状态码、自定义消息、数据
	 * @param message
	 * @param data
	 * @param <T>
	 * @return
	 */
	public static <T> Response<T> failure(String message, T data) {
		return failure(ResponseEnum.FAILURE.getCode(), message, data);
	}

	/**
	 * 失败操作，携带自定义状态码、消息、数据
	 * @param code
	 * @param message
	 * @param data
	 * @param <T>
	 * @return
	 */
	public static <T> Response<T> failure(int code, String message, T data) {
		Response<T> result = new Response<>();
		result.setCode(code);
		result.setMessage(message);
		result.setData(data);
		return result;
	}

	/**
	 * 失败操作，携带自定义状态码、消息
	 * @param code
	 * @param message
	 * @param <T>
	 * @return
	 */
	public static <T> Response<T> failure(int code, String message) {
		return failure(code, message, null);
	}

}
