package cn.jiangzeyin.job;

import cn.jiangzeyin.entity.IQuartzInfo;
import cn.jiangzeyin.system.JobDataUtil;
import cn.jiangzeyin.system.JobPropertiesInfo;
import cn.jiangzeyin.system.SystemJobManager;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import org.quartz.*;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 调度类
 *
 * @author jiangzeyin
 */
// 异常后保存JobData
@PersistJobDataAfterExecution
// 不允许并发执行
@DisallowConcurrentExecution
public abstract class BaseJob implements Job {

    /**
     * 调度类型
     */
    public static final class Type {
        public static final int Spider = 1; // 采集
        public static final int Scheduled = 0; // 调度
        public static final int Guard = -1; // 监听
    }

    /**
     * 运行状态
     */
    private static final ConcurrentHashMap<String, JobRunResultStatus> jobStatus = new ConcurrentHashMap<>();

    /**
     * 当前调度对象
     */
    protected JobExecutionContext context;
    private HashMap<String, String> jobData = new HashMap<>();
    private Class entityClass;

    /**
     * @param context context
     * @throws JobExecutionException 异常
     * @author jiangzeyin
     */
    @Override
    public void execute(JobExecutionContext context) throws JobExecutionException {
        // TODO Auto-generated method stub
        this.execute(context, JobDataUtil.getInstance().getJobDefaultClass());
    }

    /**
     * @param context     context
     * @param entityClass 运行对应实体
     * @throws JobExecutionException 异常
     */
    public void execute(JobExecutionContext context, Class entityClass) throws JobExecutionException {
        if (entityClass == null)
            throw new IllegalArgumentException("entityClass is null");
        this.context = context;
        this.entityClass = entityClass;
        JSONArray jsonArray = JobDataUtil.getInstance().getJobRunData(entityClass, getJobId());
        for (int i = jsonArray == null ? -1 : jsonArray.size() - 1; i > -1; i--) {
            JSONObject jsonObject = jsonArray.getJSONObject(i);
            jobData.put(jsonObject.getString(SystemJobManager.RunData.getKeyName()), jsonObject.getString(SystemJobManager.RunData.getValueName()));
        }
        putStatus(false);
    }

    /**
     * @param name  name
     * @param value value
     */
    public synchronized void saveRunDataValue(String name, Object value) {
        JSONArray jsonArray = JobDataUtil.getInstance().getJobRunData(entityClass, getJobId());
        if (jsonArray == null) {
            jsonArray = new JSONArray();
        }
        boolean isFind = false;
        for (int i = 0; i < jsonArray.size(); i++) {
            JSONObject jsonObject = jsonArray.getJSONObject(i);
            String jsonName = jsonObject.getString(SystemJobManager.RunData.getKeyName());
            if (name.equals(jsonName)) {
                jsonObject.put(SystemJobManager.RunData.getValueName(), value);
                isFind = true;
            }
        }
        if (!isFind) {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put(SystemJobManager.RunData.getKeyName(), name);
            jsonObject.put(SystemJobManager.RunData.getValueName(), value);
            jsonArray.add(jsonObject);
        }
        //updateData(jsonArray.toString());
        JobDataUtil.getInstance().update(entityClass, getJobId(), jsonArray.toString());
    }

    /**
     * @param jsonObject json
     * @author jiangzeyin
     */
    public synchronized void saveRunDataValue(JSONObject jsonObject) {
        if (jsonObject == null)
            return;
        JSONArray jsonArray = JobDataUtil.getInstance().getJobRunData(entityClass, getJobId());
        if (jsonArray == null) {
            jsonArray = new JSONArray();
        }
        Set<Map.Entry<String, Object>> entries = jsonObject.entrySet();
        for (Map.Entry<String, Object> item : entries) {
            if (item == null)
                continue;
            //System.out.println(item);
            String keyName = item.getKey();
            boolean isFind = false;
            if (keyName != null)
                for (int i = 0; i < jsonArray.size(); i++) {
                    JSONObject tempJson = jsonArray.getJSONObject(i);
                    String jsonName = tempJson.getString(SystemJobManager.RunData.getKeyName());
                    if (keyName.equals(jsonName)) {
                        tempJson.put(SystemJobManager.RunData.getValueName(), item.getValue());
                        isFind = true;
                    }
                }
            if (!isFind) {
                JSONObject tempJson = new JSONObject();
                tempJson.put(SystemJobManager.RunData.getKeyName(), keyName);
                tempJson.put(SystemJobManager.RunData.getValueName(), item.getValue());
                jsonArray.add(tempJson);
            }
        }
        JobDataUtil.getInstance().update(entityClass, getJobId(), jsonArray.toString());
        //updateData(jsonArray.toString());
    }

    /**
     * @return json
     * @author jiangzeyin
     */
    public JSONObject getRunDataValueToJSONObject() {
        JSONArray jsonArray = JobDataUtil.getInstance().getJobRunData(entityClass, getJobId());
        return IQuartzInfo.convertJSONObject(jsonArray);
    }

    /**
     * 运行结束
     *
     * @author jiangzeyin
     */
    protected synchronized void putStatus() {
        putStatus(true);
    }


    /**
     * @param isEnd isend
     * @author jiangzeyin
     */
    private synchronized void putStatus(boolean isEnd) {
        String name = getName();
        JobRunResultStatus job = jobStatus.get(name);
        if (job == null) {
            job = new JobRunResultStatus();
            job.setName(name);
            jobStatus.put(name, job);
        }
        if (isEnd) {
            job.setRunCount(job.getRunCount() + 1);
            job.setEndTime(System.currentTimeMillis());
        } else {
            job.setContext(context);
            job.setEndTime(null);
        }
    }

    /**
     * 获取调度运行状态
     *
     * @return list
     * @author jiangzeyin
     */
    public static List<JobRunResultStatus> getJobStatus() {
        List<JobRunResultStatus> list = new ArrayList<>();
        list.addAll(BaseJob.jobStatus.values());
        return list;
    }

    public String getString(String name, String default_) {
        String value = jobData.get(name);
        if (value == null)
            return default_;
        return value;
    }

    /**
     * 获取调度运行传入数据
     *
     * @param key      key
     * @param default_ 默认子
     * @return int
     * @author jiangzeyin
     */
    protected int getInt(String key, int default_) {
        String value = jobData.get(key);
        if (value == null)
            return default_;
        if (value.isEmpty())
            return default_;
        return Integer.parseInt(value);
    }

    /**
     * 获取当前调度名称
     *
     * @return name
     * @author jiangzeyin
     */
    protected String getName() {
        if (context == null)
            return "未知名称";
        String name = context.getJobDetail().getJobDataMap().getString("name");
        int type = getJobType();// context.getJobDetail().getJobDataMap().getInt("type");
        String systemTag = SystemJobManager.getValue(JobPropertiesInfo.TAG, "");// UtilSystemCache.getInstance().getSystemTag();
        if (type == Type.Spider) {
            name = String.format("%s(采集)_%s", name, systemTag);
        } else if (type == Type.Scheduled) {
            name = String.format("%s(调度)_%s", name, systemTag);
        } else if (type == Type.Guard) {
            name = String.format("%s(守护)_%s", name, systemTag);
        } else {
            name = String.format("%s(未知)_%s", name, systemTag);
        }
        return name;
    }

    protected int getJobId() {
        return context.getJobDetail().getJobDataMap().getIntValue("id");
    }

    protected int getJobType() {
        return 0;
    }
}
