package jexx.poi.read.impl;

import jexx.bean.BeanUtil;
import jexx.poi.SheetReader;
import jexx.poi.RowMapper;
import jexx.poi.cell.ICell;
import jexx.poi.cell.IMergedCell;
import jexx.poi.header.Headers;
import jexx.poi.header.IDataHeader;
import jexx.poi.meta.node.INode;
import jexx.poi.read.SheetCellReader;
import jexx.poi.row.RowMap;
import jexx.poi.row.RowActionPredicate;
import jexx.poi.row.RowScanAction;
import jexx.util.Assert;
import jexx.util.CollectionUtil;
import jexx.util.StringUtil;
import org.apache.poi.ss.usermodel.Sheet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * sheet reader
 *
 * @author jeff
 */
public class SheetReaderImpl extends SheetCellReader implements SheetReader {

    private static final Logger LOG = LoggerFactory.getLogger(SheetReaderImpl.class);

    private static final int FIRST_ROW_NUM = 1;

    /** 当前行号游标, 从1开始 */
    protected int currentRowNum = FIRST_ROW_NUM;
    /**
     * 跳过空行, 空行为当前行的单元格为空, 只对批量读取数据时生效
     */
    private boolean skipEmptyRow = false;
    /**
     * 是否支持在读取单元格数据时去除两端空格;
     * <b>读取时候去除两边空格会导致meta读取不匹配</b>
     */
    @Deprecated
    protected boolean supportTrimData = false;

    public SheetReaderImpl(Sheet sheet) {
        super(sheet);
    }

    @Override
    public SheetReaderImpl setSkipEmptyRow(boolean skipEmptyRow){
        this.skipEmptyRow = skipEmptyRow;
        return this;
    }

    @Override
    public SheetReaderImpl setSupportTrimData(boolean supportTrimData) {
        this.supportTrimData = supportTrimData;
        return this;
    }

    @Override
    public SheetReaderImpl passCurrentRow(){
        skipRows(1);
        return this;
    }


    @Override
    public SheetReaderImpl skipRows(int skipRowSize){
        currentRowNum += skipRowSize;
        return this;
    }

    @Override
    public SheetReaderImpl jumpToRowNum(final int rowNum){
        Assert.isTrue(rowNum > 0, "rowNum is not illegal");
        currentRowNum = rowNum;
        return this;
    }

    @Override
    public SheetReaderImpl reset(){
        currentRowNum = FIRST_ROW_NUM;
        return this;
    }

    //------------------------------------------------------------readRow

    @Override
    public <T> T readRow(final int rowNum, final int startColumnNum, final int endColumnNum, final RowMapper<T> rowMapper){
        Assert.isTrue(rowNum >= 1 && rowNum <= lastRowNumOfSheet, "Sheet[{}]'s last row num is {}", sheet.getSheetName(), lastRowNumOfSheet);
        Assert.isTrue(endColumnNum >= startColumnNum, "endColumnNum[{}] must be equal or greater than startColumnNum[{}]", startColumnNum, endColumnNum);
        this.currentRowNum = rowNum;

        int minColumnNum = Math.max(1, startColumnNum);

        int lastCellNum = getLastCellNumAtOneRow(rowNum);
        if(lastCellNum == -1){
            return null;
        }
        int maxColumnNum = Math.min(lastCellNum, endColumnNum);

        List<IMergedCell> cells = readCellsAtOneRow(rowNum, minColumnNum, maxColumnNum);
        return rowMapper.mapRow(cells, rowNum);
    }

    @Override
    public <T> T readRow(final int rowNum, final RowMapper<T> rowMapper){
        return readRow(rowNum, 1, getLastCellNumOfCurrentRow(), rowMapper);
    }

    @Override
    public <T> T readRow(final RowMapper<T> rowMapper){
        return readRow(this.currentRowNum, rowMapper);
    }

    @Override
    public <T> List<T> readRows(final int lastRowNum, final RowMapper<T> rowMapper, boolean includeNull){
        List<T> list = new ArrayList<>();
        if(this.currentRowNum > lastRowNum){
            return list;
        }

        for(int rowNum = currentRowNum; rowNum <= lastRowNum; rowNum++){
            T t = readRow(rowNum, rowMapper);
            if(t != null || includeNull){
                list.add(t);
            }
        }
        return list;
    }

    @Override
    public <T> List<T> readRows(final RowMapper<T> rowMapper, boolean includeNull){
        return readRows(lastRowNumOfSheet, rowMapper, includeNull);
    }

    @Override
    public <T> List<T> readRows(final RowMapper<T> rowMapper){
       return readRows(rowMapper, false);
    }

    //------------------------------------------------------------readListRow

    @Override
    public List<Object> readListRow(final int readRowNum, final int startColumnNum, final int endColumnNum){
        Assert.isTrue(startColumnNum > 0 && endColumnNum >= startColumnNum, "rowNum={},startColumnNum={},endColumnNum={}", currentRowNum, startColumnNum, endColumnNum);
        return readRow(readRowNum, startColumnNum, endColumnNum, (rowData, rowNum)-> rowData.stream().map(ICell::getValue).collect(Collectors.toList()));
    }

    @Override
    public List<Object> readListRow(int startColumnNum){
        return readListRow(this.currentRowNum, startColumnNum, getLastCellNumOfCurrentRow());
    }

    @Override
    public List<Object> readListRow(){
        return readListRow(1);
    }

    @Override
    public List<List<Object>> readListRows(final int readRowNum, final int startColumnNum, final int endColumnNum){
        List<List<Object>> list = new ArrayList<>();
        List<Object> data;
        for(int i = readRowNum; i <= lastRowNumOfSheet; i++){
            data = readListRow(i, startColumnNum, endColumnNum);
            if(data != null){
                list.add(data);
            }
        }
        return list;
    }

    @Override
    public List<List<Object>> readListRows(final int startColumnNum, final int endColumnNum){
        return readListRows(this.currentRowNum, startColumnNum, endColumnNum);
    }

    @Override
    public List<List<Object>> readListRows(final int startColumnNum){
        int lastCellNum = getLastCellNumOfCurrentRow();
        return readListRows(startColumnNum, lastCellNum);
    }

    @Override
    public List<List<Object>> readListRows(){
        return readListRows(1);
    }


    //------------------------------------------------------------readMapRow

    @Override
    public Map<String, Object> readMapRow(final int readRowNum, final Headers headers){
        Assert.isTrue(headers != null && CollectionUtil.isNotEmpty(headers.getHeaders()), "headers not empty");
        int minHeaderNum = 0;
        int maxHeaderNum = 0;
        for(IDataHeader dataHeader : headers.getDataHeaders()){
            if(minHeaderNum == 0){
                minHeaderNum = dataHeader.getStartColumnNum();
                maxHeaderNum = dataHeader.getStartColumnNum();
            }
            else{
                minHeaderNum = Math.min(minHeaderNum, dataHeader.getStartColumnNum());
                maxHeaderNum = Math.max(maxHeaderNum, dataHeader.getStartColumnNum());
            }
        }
        return readRow(readRowNum, minHeaderNum, maxHeaderNum, (cells, rowNum)-> {
            RowMap labelMap = new RowMap();
            for(IMergedCell cell : cells){
                IDataHeader header = headers.getDataHeaderByColumnNum(cell.getFirstColumnNum());
                if(header == null){
                    continue;
                }
                String key = header.getKey();
                Object value = cell.getValue();
                if(supportTrimData){
                    if(value instanceof String){
                        value = StringUtil.trim(value.toString());
                    }
                }
                if(header.getUnwrapLabelFunction() != null){
                    value = header.getUnwrapLabelFunction().unwrap(value);
                }
                labelMap.putIfAbsent(key, value);
            }

            Map<String, Object> dataMap = new HashMap<>(16);
            for(IDataHeader dataHeader : headers.getDataHeaders()){
                INode node = dataHeader.getNodeByLabel(labelMap);
                if(node != null){
                    dataMap.put(dataHeader.getKey(), node.getValue());
                }
            }
            return dataMap;
        });
    }

    @Override
    public Map<String, Object> readMapRow(final Headers headers){
        return readMapRow(this.currentRowNum, headers);
    }

    @Override
    public List<Map<String, Object>> readMapRows(final int rowNum, final Headers headers){
        if(rowNum > lastRowNumOfSheet){
            return new ArrayList<>();
        }

        List<Map<String, Object>> maps = new ArrayList<>();
        for(int i = rowNum; i <= lastRowNumOfSheet; i++){
            Map<String, Object> map = readMapRow(i, headers);
            if(skipEmptyRow && checkMapEmpty(map)){
                continue;
            }
            maps.add(map);
        }
        return maps;
    }

    @Override
    public List<Map<String, Object>> readMapRows(final Headers headers){
        return readMapRows(this.currentRowNum, headers);
    }

    //-----------------------------------------bean

    @Override
    public <T> T readBeanRow(int rowNum, Class<T> clazz, final Headers headers){
        Map<String, Object> map = readMapRow(rowNum, headers);
        return BeanUtil.toBean(clazz, map);
    }

    @Override
    public <T> T readBeanRow(final Class<T> clazz, final Headers headers){
        return readBeanRow(currentRowNum, clazz, headers);
    }

    @Override
    public <T> List<T> readBeanRows(final int readRowNum, final Class<T> clazz, final Headers headers, RowActionPredicate<T> skipRow){
        List<T> list = new ArrayList<>();
        Map<String, Object> map;
        for(int i = readRowNum; i <= lastRowNumOfSheet; i++){
            map = readMapRow(i, headers);
            if(skipEmptyRow && checkMapEmpty(map)){
                continue;
            }

            if(map != null){
                T t = BeanUtil.toBean(clazz, map);
                if(t != null){
                    if(skipRow == null){
                        list.add(t);
                    }
                    else{
                        RowScanAction action = skipRow.action(t, currentRowNum);
                        if(RowScanAction.STOP == action){
                            break;
                        }
                        else if(RowScanAction.KEEP_AND_STOP == action){
                            list.add(t);
                            break;
                        }
                        else{
                            list.add(t);
                        }
                    }
                }
            }
        }
        return list;
    }

    @Override
    public <T> List<T> readBeanRows(final int rowNum, final Class<T> clazz, final Headers headers){
        return readBeanRows(rowNum, clazz, headers, null);
    }

    @Override
    public <T> List<T> readBeanRows(final Class<T> clazz, final Headers headers, RowActionPredicate<T> skipRow){
        return readBeanRows(this.currentRowNum, clazz, headers, skipRow);
    }

    @Override
    public <T> List<T> readBeanRows(final Class<T> clazz, final Headers headers){
        return readBeanRows(clazz, headers, null);
    }

    @Override
    public boolean validateHeaders(int validateRowNum, Headers headers){
        if(CollectionUtil.isEmpty(headers.getDataHeaders())){
            return false;
        }

        List<IMergedCell> cells = readCellsAtOneRow(validateRowNum);
        if(CollectionUtil.isEmpty(cells)){
            return false;
        }

        Map<Integer, Object> map = cells.stream().filter(s -> s.getValue() != null).collect(Collectors.toMap(ICell::getFirstColumnNum, ICell::getValue));

        for(IDataHeader header : headers.getDataHeaders()){
            String mapValue = (String)map.get(header.getStartColumnNum());
            if(!header.getValue().equals(mapValue)){
                LOG.warn("validate header[{}] failed, now is \"{}\"!", header.getValue(), mapValue);
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean validateHeaders(Headers headers){
        return validateHeaders(this.currentRowNum, headers);
    }


    /**
     * 获取当前行的最后列号
     */
    protected int getLastCellNumOfCurrentRow(){
        return getLastCellNumAtOneRow(this.currentRowNum);
    }

    private boolean checkMapEmpty(Map<String, Object> map){
        if(map == null){
            return true;
        }
        Object value;
        for (Map.Entry<String, Object> m : map.entrySet()){
            value = m.getValue();
            if(value != null){
                if(value instanceof String && StringUtil.isEmpty((String)value)){
                    continue;
                }
                return false;
            }
        }
        return true;
    }

}
