package jexx.poi;

import jexx.poi.header.Headers;
import jexx.poi.row.RowActionPredicate;

import java.util.List;
import java.util.Map;

/**
 * sheet读取器
 * @author jeff
 * @since 2019/10/8
 */
public interface SheetReader {

    /**
     * 设置跳过空行; 这会验证每条数据是否为空
     * @param skipEmptyRow 是否跳过空行
     * @return {@link SheetReader}
     */
    SheetReader setSkipEmptyRow(boolean skipEmptyRow);

    /**
     * 设置支持读取数据时去除两边空格，默认支持
     * @param supportTrimData 是否支持读取数据时去除两边空格
     */
    SheetReader setSupportTrimData(boolean supportTrimData);

    /**
     * 跳过当前行
     * @return {@link SheetReader}
     */
    SheetReader passCurrentRow();

    /**
     * 跳过指定几行
     * @param skipRowSize 几行
     * @return {@link SheetReader}
     */
    SheetReader skipRows(int skipRowSize);

    /**
     * 直接跳至某一行
     * @param rowNum 行号,从1开始
     * @return {@link SheetReader}
     */
    SheetReader jumpToRowNum(final int rowNum);

    /**
     * 重新设置为初始行
     * @return {@link SheetReader}
     */
    SheetReader reset();

    /**
     * 读取指定行的数据
     * @param rowNum 指定行
     * @param startColumnNum 开始列号
     * @param endColumnNum 结束列号
     * @param rowMapper {@link RowMapper}
     * @param <T> 转换泛型
     * @return 转换对象
     */
    <T> T readRow(final int rowNum, final int startColumnNum, final int endColumnNum, final RowMapper<T> rowMapper);


    /**
     * 读取指定行的数据
     * @param rowNum 指定行
     * @param rowMapper {@link RowMapper}
     * @param <T> 转换泛型
     * @return 转换对象
     */
    <T> T readRow(final int rowNum, final RowMapper<T> rowMapper);

    /**
     * 读取下一行数据
     * @param rowMapper {@link RowMapper}
     * @param <T> 转换泛型
     * @return 转换对象
     */
    <T> T readRow(final RowMapper<T> rowMapper);

    /**
     * 通过行转换类,把当前行到指定一行转换成想要的集合数据,空
     * @param lastRowNum 读取结束行, 从1开始
     * @param rowMapper {@link RowMapper}
     * @param includeNull 行转换器得到的结果为空，是否包含
     * @param <T> 泛型
     * @return 对象集合
     */
    <T>  List<T> readRows(final int lastRowNum, final RowMapper<T> rowMapper, boolean includeNull);

    /**
     * 通过行转换类,把当前行到最后一行转换成想要的集合数据,空
     * @param rowMapper {@link RowMapper}
     * @param includeNull 行转换器得到的结果为空，是否包含
     * @param <T> 泛型
     * @return 对象集合
     */
    <T> List<T> readRows(final RowMapper<T> rowMapper, boolean includeNull);

    /**
     * 通过行转换类,把当前行到最后一行转换成想要的集合数据,不包含空数据
     * @param rowMapper {@link RowMapper}
     * @param <T> 泛型
     * @return 对象集合
     */
    <T> List<T> readRows(final RowMapper<T> rowMapper);

    /**
     * 读取指定范围列的数据
     * @param readRowNum 指定行号, base 1
     * @param startColumnNum 开始列号,从1开始
     * @param endColumnNum 结束列号,从1开始
     * @return 一行的范围数据
     */
    List<Object> readListRow(final int readRowNum, final int startColumnNum, final int endColumnNum);

    /**
     * 从某一列开始读取数据
     * @param startColumnNum 开始列号,从1开始
     * @return 一行的范围数据
     */
    List<Object> readListRow(int startColumnNum);

    /**
     * 从第一列开始读取数据
     * @return 一行的范围数据
     */
    List<Object> readListRow();

    /**
     * 指定行开始读取行数据
     * @param readRowNum 指定行
     * @param startColumnNum 开始列号
     * @param endColumnNum 结束列号
     * @return 行数据集合
     */
    List<List<Object>> readListRows(final int readRowNum, final int startColumnNum, final int endColumnNum);

    List<List<Object>> readListRows(final int startColumnNum, final int endColumnNum);

    List<List<Object>> readListRows(final int startColumnNum);

    List<List<Object>> readListRows();

    Map<String, Object> readMapRow(final int readRowNum, final Headers headers);

    /**
     * 读取一行指定范围的数据.
     * @param headers headers
     * @return 一行数据
     */
    Map<String, Object> readMapRow(final Headers headers);

    List<Map<String, Object>> readMapRows(final int rowNum, final Headers headers);

    List<Map<String, Object>> readMapRows(final Headers headers);

    <T> T readBeanRow(int rowNum, Class<T> clazz, final Headers headers);

    <T> T readBeanRow(final Class<T> clazz, final Headers headers);

    /**
     * 读取行数据转换为对象
     * @param readRowNum 指定行读取
     * @param clazz clazz
     * @param headers {@link Headers}
     * @param skipRow 是否跳过当前行, 为null则不进行跳行判断
     * @param <T> 泛型
     * @return 行的集合
     */
    <T> List<T> readBeanRows(final int readRowNum, final Class<T> clazz, final Headers headers, RowActionPredicate<T> skipRow);

    <T> List<T> readBeanRows(final int rowNum, final Class<T> clazz, final Headers headers);

    /**
     * 读取行数据转换为对象
     * @param clazz 转换对象
     * @param headers {@link Headers}
     * @param skipRow 跳行判断
     * @param <T> 泛型
     * @return 对象集合
     */
    <T> List<T> readBeanRows(final Class<T> clazz, final Headers headers, RowActionPredicate<T> skipRow);


    /**
     * 读取行数据转换为bean对象
     * @param clazz 转换对象
     * @param headers {@link Headers}
     * @param <T> 泛型
     * @return 对象集合
     */
    <T> List<T> readBeanRows(final Class<T> clazz, final Headers headers);

    /**
     * 验证指定行 header是否匹配
     * @param validateRowNum 验证行
     * @param headers headers
     * @return 是否匹配
     */
    boolean validateHeaders(int validateRowNum, Headers headers);

    /**
     * 验证当前行是否匹配header
     * @param headers {@link Headers}
     * @return 是否匹配
     */
    boolean validateHeaders(Headers headers);

}
