package jexx.log;

import jexx.lang.Caller;
import jexx.log.dialect.console.ConsoleLogFactory;
import jexx.log.dialect.jdk.JdkLogFactory;
import jexx.log.dialect.slf4j.Slf4jLogFactory;
import jexx.util.ClassLoaderUtil;
import jexx.log.dialect.log4j2.Log4j2LogFactory;

import java.net.URL;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * log factory
 *
 * @author jeff
 */
public abstract class LogFactory {

	private String logFrameworkName;
	private Map<Object, Log> logCache;
	
	private static volatile LogFactory currentLogFactory;
	private static final Object lock = new Object();
	
	public LogFactory(String logFrameworkName) {
		this.logFrameworkName = logFrameworkName;
		logCache = new ConcurrentHashMap<>();
	}
	
	public static LogFactory getCurrentLogFactory() {
		if (null == currentLogFactory) {
			synchronized (lock) {
				if (null == currentLogFactory) {
					currentLogFactory = detectLogFactory();
				}
			}
		}
		return currentLogFactory;
	}
	
	public static Log get(Class<?> clazz) {
		return getCurrentLogFactory().getLog(clazz);
	}

    /**
     * 获取调用者的日志类, 自动发现当前类
     * @return {@link Log}
     */
	public static Log get(){
	    return get(Caller.getCallerCaller());
    }
	
	public Log getLog(String name){
		Log log = logCache.get(name);
		if(null == log){
			log = createLog(name);
			logCache.put(name, log);
		}
		return log;
	}
	
	public Log getLog(Class<?> clazz){
		Log log = logCache.get(clazz);
		if(null == log){
			log = createLog(clazz);
			logCache.put(clazz, log);
		}
		return log;
	}
	
	public abstract Log createLog(String name);
	
	public abstract Log createLog(Class<?> clazz);
	
	private static LogFactory detectLogFactory() {
		LogFactory logFactory = null;
		try{
			logFactory = new Slf4jLogFactory();
			logFactory.getLog(LogFactory.class).debug("Use [{}] Logger As Default.", logFactory.logFrameworkName);
		}
		catch(NoClassDefFoundError e1){
			try{
				logFactory = new Log4j2LogFactory();
				logFactory.getLog(LogFactory.class).debug("Use [{}] Logger As Default.", logFactory.logFrameworkName);
			}
			catch(NoClassDefFoundError e2){
                final URL url = ClassLoaderUtil.getClassLoader().getResource("logging.properties");
                logFactory = (null != url) ? new JdkLogFactory() : new ConsoleLogFactory();
			}
		}
		return logFactory;
	}
	
	protected void checkLogExist(Object logClassName) {
		// 不做任何操作
	}
	
}
