package jexx.bean;

import jexx.util.ObjectUtil;

import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.Enumeration;

/**
 * 属性描述器相关工具类
 */
public class PropertyDescriptorUtil {

    /**
     * 拷贝属性描述
     * @param source 源属性描述
     * @param target 目标属性描述
     */
    public static void copyNonMethodProperties(PropertyDescriptor source, PropertyDescriptor target) {
        target.setExpert(source.isExpert());
        target.setHidden(source.isHidden());
        target.setPreferred(source.isPreferred());
        target.setName(source.getName());
        target.setShortDescription(source.getShortDescription());
        target.setDisplayName(source.getDisplayName());

        // Copy all attributes (emulating behavior of private FeatureDescriptor#addTable)
        Enumeration<String> keys = source.attributeNames();
        while (keys.hasMoreElements()) {
            String key = keys.nextElement();
            target.setValue(key, source.getValue(key));
        }

        // See java.beans.PropertyDescriptor#PropertyDescriptor(PropertyDescriptor)
        target.setPropertyEditorClass(source.getPropertyEditorClass());
        target.setBound(source.isBound());
        target.setConstrained(source.isConstrained());
    }


    /**
     * 通过读方法和写方法获取属性类型
     * @param readMethod 读方法
     * @param writeMethod 写方法
     * @return 属性类型
     */
    public static Class<?> findPropertyType(Method readMethod, Method writeMethod)
            throws IntrospectionException {

        Class<?> propertyType = null;

        if (readMethod != null) {
            Class<?>[] params = readMethod.getParameterTypes();
            if (params.length != 0) {
                throw new IntrospectionException("Bad read method arg count: " + readMethod);
            }
            propertyType = readMethod.getReturnType();
            if (propertyType == Void.TYPE) {
                throw new IntrospectionException("Read method returns void: " + readMethod);
            }
        }

        if (writeMethod != null) {
            Class<?>[] params = writeMethod.getParameterTypes();
            if (params.length != 1) {
                throw new IntrospectionException("Bad write method arg count: " + writeMethod);
            }
            if (propertyType != null) {
                if (propertyType.isAssignableFrom(params[0])) {
                    // Write method's property type potentially more specific
                    propertyType = params[0];
                }
                else if (params[0].isAssignableFrom(propertyType)) {
                    // Proceed with read method's property type
                }
                else {
                    throw new IntrospectionException(
                            "Type mismatch between read and write methods: " + readMethod + " - " + writeMethod);
                }
            }
            else {
                propertyType = params[0];
            }
        }

        return propertyType;
    }

    /**
     * 找到索引属性的类型
     * @param name 属性名称
     * @param propertyType 属性类型
     * @param indexedReadMethod 索引读方法
     * @param indexedWriteMethod 索引写方法
     * @return 索引属性类型
     */
    public static Class<?> findIndexedPropertyType(String name, Class<?> propertyType, Method indexedReadMethod, Method indexedWriteMethod) throws IntrospectionException {
        Class<?> indexedPropertyType = null;

        if (indexedReadMethod != null) {
            Class<?>[] params = indexedReadMethod.getParameterTypes();
            if (params.length != 1) {
                throw new IntrospectionException("Bad indexed read method arg count: " + indexedReadMethod);
            }
            if (params[0] != Integer.TYPE) {
                throw new IntrospectionException("Non int index to indexed read method: " + indexedReadMethod);
            }
            indexedPropertyType = indexedReadMethod.getReturnType();
            if (indexedPropertyType == Void.TYPE) {
                throw new IntrospectionException("Indexed read method returns void: " + indexedReadMethod);
            }
        }

        if (indexedWriteMethod != null) {
            Class<?>[] params = indexedWriteMethod.getParameterTypes();
            if (params.length != 2) {
                throw new IntrospectionException("Bad indexed write method arg count: " + indexedWriteMethod);
            }
            if (params[0] != Integer.TYPE) {
                throw new IntrospectionException("Non int index to indexed write method: " + indexedWriteMethod);
            }
            if (indexedPropertyType != null) {
                if (indexedPropertyType.isAssignableFrom(params[1])) {
                    // Write method's property type potentially more specific
                    indexedPropertyType = params[1];
                }
                else if (params[1].isAssignableFrom(indexedPropertyType)) {
                    // Proceed with read method's property type
                }
                else {
                    throw new IntrospectionException("Type mismatch between indexed read and write methods: " +
                            indexedReadMethod + " - " + indexedWriteMethod);
                }
            }
            else {
                indexedPropertyType = params[1];
            }
        }

        if (propertyType != null && (!propertyType.isArray() ||
                propertyType.getComponentType() != indexedPropertyType)) {
            throw new IntrospectionException("Type mismatch between indexed and non-indexed methods: " +
                    indexedReadMethod + " - " + indexedWriteMethod);
        }

        return indexedPropertyType;
    }

    public static boolean equals(PropertyDescriptor pd, PropertyDescriptor otherPd) {
        return (ObjectUtil.nullSafeEquals(pd.getReadMethod(), otherPd.getReadMethod()) &&
                ObjectUtil.nullSafeEquals(pd.getWriteMethod(), otherPd.getWriteMethod()) &&
                ObjectUtil.nullSafeEquals(pd.getPropertyType(), otherPd.getPropertyType()) &&
                ObjectUtil.nullSafeEquals(pd.getPropertyEditorClass(), otherPd.getPropertyEditorClass()) &&
                pd.isBound() == otherPd.isBound() && pd.isConstrained() == otherPd.isConstrained());
    }


}
