package jexx.bean;

import jexx.convert.Convert;
import jexx.log.Log;
import jexx.log.LogFactory;
import jexx.util.MapUtil;
import jexx.util.ReflectUtil;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Map;

/**
 * 参考spring的BeanWrapperImpl
 */
public class DyBeanImpl extends AbstractBeanPropertyAccessor implements DyBean {

    private static final Log LOG = LogFactory.get(DyBeanImpl.class);

    protected CachedIntrospectionResults cachedIntrospectionResults;

    @Override
    protected AbstractBeanPropertyAccessor newNestedPropertyAccessor(Object object, String nestedPath) {
        DyBeanImpl dyBean = new DyBeanImpl();
        dyBean.setWrappedInstance(object, nestedPath, getWrappedInstance());
        dyBean.setAllowCollectionAutoGrow(this.allowCollectionAutoGrow);
        dyBean.setAllowCreateHoldValueIfNull(this.allowCreateHoldValueIfNull);
        return dyBean;
    }

    protected CachedIntrospectionResults getCachedIntrospectionResults() {
        if (this.cachedIntrospectionResults == null) {
            this.cachedIntrospectionResults = CachedIntrospectionResults.forClass(getWrappedClass());
        }
        return this.cachedIntrospectionResults;
    }

    @Override
    public boolean isReadableProperty(String propertyName) {
        PropertyDescriptor propertyDescriptor = getCachedIntrospectionResults().getPropertyDescriptor(propertyName);
        if(propertyDescriptor == null){
            LOG.debug("propertyName \"{}\" not found!", propertyName);
        }
        return propertyDescriptor != null && propertyDescriptor.getReadMethod() != null;
    }

    @Override
    public boolean isWritableProperty(String propertyName) {
        PropertyDescriptor propertyDescriptor = getCachedIntrospectionResults().getPropertyDescriptor(propertyName);
        if(propertyDescriptor == null){
            LOG.debug("propertyName \"{}\" not found!", propertyName);
        }
        return propertyDescriptor != null && propertyDescriptor.getWriteMethod() != null;
    }

    @Override
    public PropertyDescriptor[] getPropertyDescriptors() {
        return getCachedIntrospectionResults().getPropertyDescriptors();
    }

    @Override
    public PropertyDescriptor getPropertyDescriptor(String propertyName) {
        return getCachedIntrospectionResults().getPropertyDescriptor(propertyName);
    }

    @Override
    protected PropertyHandle getPropertyHandle(String propertyName) {
        PropertyDescriptor propertyDescriptor = getCachedIntrospectionResults().getPropertyDescriptor(propertyName);
        if(propertyDescriptor == null){
            return null;
        }
        return new BeanPropertyHandle(propertyDescriptor);
    }

    protected class BeanPropertyHandle extends PropertyHandle{

        private PropertyDescriptor propertyDescriptor;

        public BeanPropertyHandle(PropertyDescriptor propertyDescriptor) {
            super(propertyDescriptor.getName(), propertyDescriptor.getReadMethod() != null, propertyDescriptor.getWriteMethod() != null);
            this.propertyDescriptor = propertyDescriptor;
        }

        @Override
        public Object getValue() throws Exception {
            final Method readMethod = this.propertyDescriptor.getReadMethod();
            ReflectUtil.makeAccessible(readMethod);
            Object wrappedObject = getWrappedInstance();
            return readMethod.invoke(wrappedObject, (Object[]) null);
        }

        @Override
        public Class<?> getPropertyType() {
            return propertyDescriptor.getPropertyType();
        }

        @Override
        public Class<?> getCollectionType(int nestingLevel) {
            return getNestedType(nestingLevel);
        }

        @Override
        public Class<?> getMapKeyType(int nestingLevel) {
            return getNestedType(nestingLevel, MapUtil.toMap(nestingLevel, 0));
        }

        @Override
        public Class<?> getMapValueType(int nestingLevel) {
            return getNestedType(nestingLevel, MapUtil.toMap(nestingLevel, 1));
        }

        @Override
        public void setValue(Object value) throws Exception {
            final Method writeMethod = this.propertyDescriptor.getWriteMethod();
            Class<?>[] parameterTypes = writeMethod.getParameterTypes();
            ReflectUtil.makeAccessible(writeMethod);
            writeMethod.invoke(getWrappedInstance(), Convert.convert(parameterTypes[0], value));
        }

        @Override
        protected Class<?> getNestedType(int nestingLevel, Map<Integer, Integer> typeIndexesPerLevel) {
            final Method readMethod = this.propertyDescriptor.getReadMethod();
            Type type = readMethod.getGenericReturnType();
            Class clazz = type.getClass();
            for(int i = 1; i <= nestingLevel; i++){
                if(clazz.isArray()){
                    type = clazz.getComponentType();
                }
                else{
                    if(type instanceof ParameterizedType){
                        Type[]  types = ((ParameterizedType) type).getActualTypeArguments();
                        Integer index = (typeIndexesPerLevel != null ? typeIndexesPerLevel.get(i) : null);
                        index = (index == null ? types.length - 1 : index);
                        type = types[index];
                    }
                }
            }
            return resolveClass(type);
        }
    }


}
