package cn.jasonone.him.util;

import com.corundumstudio.socketio.AckRequest;
import com.corundumstudio.socketio.AuthorizationListener;
import com.corundumstudio.socketio.Configuration;
import com.corundumstudio.socketio.HandshakeData;
import com.corundumstudio.socketio.SocketIOClient;
import com.corundumstudio.socketio.SocketIOServer;
import com.corundumstudio.socketio.Transport;
import com.corundumstudio.socketio.listener.ConnectListener;
import com.corundumstudio.socketio.listener.DataListener;
import com.corundumstudio.socketio.listener.DisconnectListener;

import cn.jasonone.him.HimConst;
import cn.jasonone.him.HimProperties;
import cn.jasonone.him.model.HimUserInfo;
import cn.jasonone.him.model.Message;
import cn.jasonone.him.service.HimService;
import lombok.extern.slf4j.Slf4j;

/**
 * 服务器工具类
 * 
 * @author Jason
 *
 */
@Slf4j
public class ServerUtil {
	private static SocketIOServer server;
	private static HimService himService;

	/**
	 * 创建服务器对象
	 * 
	 * @param himProperties 配置对象
	 * @param service    业务对象
	 * @return 服务器对象
	 */
	public static SocketIOServer createServer(HimProperties himProperties, HimService service) {
		if (himProperties == null) {
			himProperties = new HimProperties();
		}
		if (service == null) {
			himService = new HimService() {
			};
		} else {
			himService = service;
		}
		Configuration configuration = new Configuration();
		configuration.setPort(himProperties.getPort());
		configuration.setTransports(Transport.POLLING, Transport.WEBSOCKET);
		configuration.setOrigin(himProperties.getOrigin());
		configuration.setHostname(himProperties.getHostname());
		configuration.setAuthorizationListener(new AuthorizationListener() {
			@Override
			public boolean isAuthorized(HandshakeData data) {
				return himService.isAuthorization(data);
			}
		});
		configuration.setPingTimeout(himProperties.getHeartbeatTimeout());
		configuration.setPingInterval(himProperties.getHeartbeatInterval());
		configuration.setOrigin(himProperties.getOrigin());
		SocketIOServer server = new SocketIOServer(configuration);
		log.info("正在启动Him服务器,服务端口为:{}", himProperties.getPort());
		// 绑定客户端上线事件
		server.addConnectListener(new ConnectListener() {
			@Override
			public void onConnect(SocketIOClient client) {
				HimUserInfo user = himService.connection(client);
				if (user != null) {
					user.setSessionId(client.getSessionId());
					client.set(HimConst.SESSION_USER, user);
				}
			}
		});
		// 绑定客户端离线事件
		server.addDisconnectListener(new DisconnectListener() {
			@Override
			public void onDisconnect(SocketIOClient client) {
				HimUserInfo user = client.get(HimConst.SESSION_USER);
				if (user != null) {
					himService.disconnection(client, user);
				}
			}
		});
		//好友消息
		server.addEventListener(HimConst.MESSAGE_FRIEND, Message.class, new DataListener<Message>() {
			@Override
			public void onData(SocketIOClient client, Message data, AckRequest ackSender) throws Exception {
				log.debug("接收消息[{}]:{}", HimConst.MESSAGE_FRIEND, data);
				himService.messageFriend(client, ackSender, data);
			}
		});
		//群消息
		server.addEventListener(HimConst.MESSAGE_GROUP, Message.class, new DataListener<Message>() {
			@Override
			public void onData(SocketIOClient client, Message data, AckRequest ackSender) throws Exception {
				log.debug("接收消息[{}]:{}", HimConst.MESSAGE_GROUP, data);
				himService.messageGroup(client, ackSender, data);
			}
		});
		//系统消息
		server.addEventListener(HimConst.MESSAGE_SYSTEM, Message.class, new DataListener<Message>() {
			@Override
			public void onData(SocketIOClient client, Message data, AckRequest ackSender) throws Exception {
				log.debug("接收消息[{}]:{}", HimConst.MESSAGE_SYSTEM, data);
				himService.messageSystem(client, ackSender, data);
			}
		});
		return server;
	}
}
