package cn.ironblog;

import cn.ironblog.common.CommonRequest;
import cn.ironblog.common.DownloadManager;
import cn.ironblog.flv.FlvDownloadCallBack;
import cn.ironblog.flv.FlvDownloadManager;
import cn.ironblog.flv.FlvFileManager;
import cn.ironblog.m3u8.JoinVideoThread;
import cn.ironblog.m3u8.M3U8DownloadCallBack;
import cn.ironblog.m3u8.M3U8DownloadManager;
import cn.ironblog.util.Utils;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;


import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.Proxy;
import java.net.URL;
import java.util.*;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

/**
 * @author WangNing
 **/

public class VideoDownloader {

    /**
     * 线程池
     */
    private ExecutorService threadPool;

    /**
     * Video 网址
     */
    private String videoUrl;

    /**
     * 下载客户端
     */
    private OkHttpClient client;

    /**
     * 异步回调
     */
    private VideoCallBack callBack;

    /**
     * 当前状态
     */
    private volatile VideoState currentState = VideoState.DOWNLOAD_READY;

    /**
     * 视频地址
     */
    private String videoPath ;

    /**
     * Video名称
     */
    private String videoName;

    private DownloadManager downloadManager;

    private final AtomicLong downloadLength = new AtomicLong();

    private JoinVideoThread joinTread = null;

    private Future<?> downloadingFuture = null;

    private final Proxy proxy ;

    public VideoDownloader(String url, String videoPath, String videoName, VideoCallBack callBack, Proxy proxy, ExecutorService threadPool){
        this.proxy = proxy;
        File file = new File(videoPath);
        if(!file.exists()){
            boolean success = file.mkdirs();
            if(!success){
                currentState = VideoState.DOWNLOAD_FAIL;
                callBack.downloadError("文件夹创建失败");
                return;
            }
        }
        this.videoUrl = url;
        this.client = Utils.getClient(proxy, downloadLength);
        this.videoPath = videoPath;
        this.videoName = videoName;
        this.threadPool = threadPool;
        this.callBack = callBack;

    }

    public Future<?> download(){
        if(currentState == VideoState.DOWNLOAD_READY){
            if(videoUrl.endsWith(".m3u8")){
                downloadingFuture =  threadPool.submit(this::downloadM3U8);
                return downloadingFuture;
            }else {

                downloadingFuture = threadPool.submit(this::downloadFlv);
                return downloadingFuture;

            }
        }

        return null;

    }

    /**
     * 下载M3U8类型的链接
     */
    private void downloadM3U8(){

        URL videoURL = null;

        try{
            videoURL = new URL(videoUrl);
        }catch (MalformedURLException e) {
            currentState = VideoState.DOWNLOAD_FAIL;
            callBack.downloadError("网址格式错误");
            return;
        }

        Request request = new Request.Builder()
                .url(videoUrl)
                .header("referer", "https://"+videoURL.getHost())
                .header("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/86.0.4240.75 Safari/537.36 Edg/86.0.622.38")
                .build();

        List<String> videoList = new ArrayList<>();

        // 下载并解析 M3U8地址
        try(Response response = client.newCall(request).execute()){

            if(response.body() == null){

                currentState = VideoState.DOWNLOAD_FAIL;
                callBack.downloadError("M3U8文件下载失败");
                return;
            }
            String m3u8 = response.body().string();
            videoList = Utils.getVideoList(request.url().url(), m3u8);

        }catch (IOException ignored){
        }

        if(videoList.isEmpty()){
            currentState = VideoState.DOWNLOAD_FAIL;
            callBack.downloadError("下载M3U8文件失败");
            return;
        }

        // 生成下载管理器，管理下载。实现断点续传
        Map<String, String> headers = new HashMap<>();
        headers.put("referer", "https://"+videoURL.getHost());
        headers.put("Connection","close");
        headers.put("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/86.0.4240.75 Safari/537.36 Edg/86.0.622.38");

        if(downloadManager == null){
            downloadManager = new M3U8DownloadManager(videoList, headers);
        }


        // 记录下载完成的线程
        Vector<Integer> completedVector = new Vector<>();

        // 合并线程
        joinTread = new JoinVideoThread((M3U8DownloadManager) downloadManager, videoPath, videoList.size(), videoName);

        Future<?> joinFuture = threadPool.submit(joinTread);

        // 分发下载线程，启动合并线程
        AtomicInteger downloadedNum = new AtomicInteger(1);

        // 分发请求
        for(CommonRequest newRequest: downloadManager.getRequest()){
            if(newRequest.getOneExtraData("index") instanceof Integer){
                int index = (Integer) newRequest.getOneExtraData("index");
                M3U8DownloadCallBack callBack = new M3U8DownloadCallBack(videoPath, index,
                        downloadedNum, (M3U8DownloadManager) downloadManager);

                client.newCall(newRequest.getRequest()).enqueue(callBack);

            }

        }

        // 重置状态为下载中
        currentState = VideoState.DOWNLOADING;

        while (client.dispatcher().queuedCallsCount()>0 || client.dispatcher().runningCallsCount() > 0){

            try {

                if(currentState == VideoState.DOWNLOAD_PAUSE){
                    return;
                }
                long currentDownload = downloadLength.get();
                double currentPercent = ((double)downloadedNum.get() / videoList.size() )* 100;

                callBack.process(currentPercent, currentDownload);

                Thread.sleep(1000);

            } catch (InterruptedException e) {
                break;
            }

        }

        try {
            joinFuture.get();
        } catch (InterruptedException|ExecutionException e) {
            currentState = VideoState.DOWNLOAD_SUCCESS;
        }

        client.dispatcher().executorService().shutdown();
        if(currentState == VideoState.DOWNLOADING){
            callBack.downloadSuccess();
        }


    }

    /**
     * 下载flv类型的文件
     */
    private void downloadFlv(){

        Request request = new Request.Builder().url(videoUrl)
                .header("RANGE", "bytes=" + "0-")
                .build();


        int splitSize = 20;

        long length = 0;

        try(Response response = client.newCall(request).execute()){

            length = response.body().contentLength();


        }catch (IOException|NullPointerException e){
            currentState = VideoState.DOWNLOAD_FAIL;
            callBack.downloadError("下载文件错误");
            return;
        }

        if(downloadManager == null){
            downloadManager = new FlvDownloadManager(videoUrl, length, 20);
        }

        for(CommonRequest newRequest:downloadManager.getRequest()){

            FlvFileManager flvFileManager = new FlvFileManager(videoPath, videoName, length);
            long start = (long)newRequest.getOneExtraData("start");
            long end = (long)newRequest.getOneExtraData("end");
            client.newCall(newRequest.getRequest()).enqueue(new FlvDownloadCallBack(flvFileManager, client, start, end,
                    (FlvDownloadManager)downloadManager));

        }

        currentState = VideoState.DOWNLOADING;

        while (client.dispatcher().queuedCallsCount()>0 || client.dispatcher().runningCallsCount() > 0){

            if(currentState == VideoState.DOWNLOAD_PAUSE){
                return;
            }

            long currentDownload = downloadLength.get();
            callBack.process(currentDownload / (double)length * 100, currentDownload);
            try {
                Thread.sleep(1000);
            }catch (InterruptedException ignored){

            }
        }

        currentState = VideoState.DOWNLOAD_SUCCESS;
        callBack.downloadSuccess();
        client.dispatcher().executorService().shutdown();
    }

    public static enum VideoState{
        /**
         * 下载未开始
         */
        DOWNLOAD_READY,

        /**
         * 下载中
         */
        DOWNLOADING,

        /**
         * 下载完成
         */
        DOWNLOAD_SUCCESS,

        /**
         * 下载失败
         */
        DOWNLOAD_FAIL,

        /**
         * 下载暂停
         */
        DOWNLOAD_PAUSE;

    }

    public synchronized VideoState getCurrentState() {
        return currentState;
    }


    public void stopDownload(){

    }

    public void pauseDownload(){

        long pauseLength = downloadLength.get();

        if(currentState == VideoState.DOWNLOADING){
            currentState = VideoState.DOWNLOAD_PAUSE;
        }

        System.out.println("清空请求");
        // 清空准备下载的请求，不中断正在请求的连接
        client.dispatcher().cancelAll();
        client.dispatcher().executorService().shutdownNow();

        if(downloadingFuture!=null){
            System.out.println("关闭下载线程");
            downloadingFuture.cancel(true);
        }

        if(joinTread!=null){

            System.out.println("！！！！！打断合并线程");
            joinTread.setStop(true);
        }

        downloadLength.set(pauseLength);


    }

    public Future<?> startDownload(){
        client = Utils.getClient(proxy, downloadLength);
        if(currentState == VideoState.DOWNLOAD_PAUSE){
            currentState = VideoState.DOWNLOAD_READY;
            return download();
        }
        return null;

    }

}
