package cn.ipokerface.weixin.mp.proxy;

import cn.ipokerface.weixin.exception.WeixinException;
import cn.ipokerface.weixin.model.Token;
import cn.ipokerface.weixin.model.media.MpArticle;
import cn.ipokerface.weixin.model.media.MpNewsMessageTuple;
import cn.ipokerface.weixin.model.media.MultiMessageTuple;
import cn.ipokerface.weixin.mp.request.WeixinMPlatformApis;
import cn.ipokerface.weixin.request.ApiResult;
import cn.ipokerface.weixin.request.WeixinResponse;
import cn.ipokerface.weixin.token.TokenManager;
import cn.ipokerface.weixin.utils.StringUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by       PokerFace
 * Create Date      2019-12-28.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class MessageProxy extends MPlatformProxy {

    public MessageProxy(TokenManager tokenManager){
        super(tokenManager);
    }



    /**
     * 上传图文消息,一个图文消息支持1到10条图文
     * 具备微信支付权限的公众号，在使用高级群发接口上传、群发图文消息类型时，可使用标签加入外链
     *
     * @param articles
     *            图片消息
     * @return 媒体ID
     * @throws WeixinException
     * @see <a href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140549&token=&lang=zh_CN">上传图文素材</a>
     * @see MpArticle
     */
    public String uploadArticle(List<MpArticle> articles) throws WeixinException {
        String article_upload_uri = WeixinMPlatformApis.article_upload_uri;
        Token token = tokenManager.getCache();
        JSONObject obj = new JSONObject();
        obj.put("articles", articles);
        WeixinResponse response = weixinRequestClient.post(String.format(article_upload_uri, token.getAccessToken()),
                obj.toJSONString());

        return response.getAsJson().getString("media_id");
    }

    /**
     * 分组群发
     * <p>
     * 在返回成功时,意味着群发任务提交成功,并不意味着此时群发已经结束,所以,仍有可能在后续的发送过程中出现异常情况导致用户未收到消息,
     * 如消息有时会进行审核、服务器不稳定等,此外,群发任务一般需要较长的时间才能全部发送完毕
     * </p>
     *
     * @param tuple
     *            消息元件
     * @param isToAll
     *            用于设定是否向全部用户发送，值为true或false，选择true该消息群发给所有用户，
     *            选择false可根据group_id发送给指定群组的用户
     * @param groupId
     *            分组ID
     * @return 第一个元素为消息发送任务的ID,第二个元素为消息的数据ID，该字段只有在群发图文消息时，才会出现,可以用于在图文分析数据接口中
     * @throws WeixinException
     * @see cn.ipokerface.weixin.model.media.TextMessageTuple
     * @see cn.ipokerface.weixin.model.media.ImageMessageTuple
     * @see cn.ipokerface.weixin.model.media.VoiceMessageTuple
     * @see cn.ipokerface.weixin.model.media.MpVideoMessageTuple
     * @see MpNewsMessageTuple
     * @see cn.ipokerface.weixin.model.media.CardMessageTuple
     * @see MultiMessageTuple
     * @see <a href=
     *      "https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140549&token=&lang=zh_CN">根据分组群发</a>
     */
    @Deprecated
    public String[] messageByGroupId(MultiMessageTuple tuple, boolean isToAll, int groupId) throws WeixinException {
        if (tuple instanceof MpNewsMessageTuple) {
            MpNewsMessageTuple _news = (MpNewsMessageTuple) tuple;
            List<MpArticle> _articles = _news.getArticles();
            if (StringUtil.isBlank(_news.getMediaId())) {
                if (_articles.isEmpty()) {
                    throw new WeixinException("mass fail:mediaId or articles is required");
                }
                tuple = new MpNewsMessageTuple(uploadArticle(_articles));
            }
        }
        String msgtype = tuple.getMessageType();
        JSONObject obj = new JSONObject();
        JSONObject item = new JSONObject();
        item.put("is_to_all", isToAll);
        if (!isToAll) {
            item.put("group_id", groupId);
        }
        obj.put("filter", item);
        obj.put(msgtype, JSON.toJSON(tuple));
        obj.put("msgtype", msgtype);
        String mass_group_uri = WeixinMPlatformApis.message_group_uri;
        Token token = tokenManager.getCache();
        WeixinResponse response = weixinRequestClient.post(String.format(mass_group_uri, token.getAccessToken()),
                obj.toJSONString());

        obj = response.getAsJson();
        return new String[] { obj.getString("msg_id"), obj.getString("msg_data_id") };
    }

    /**
     * 分组ID群发图文消息
     *
     * @param articles
     *            图文列表
     * @param groupId
     *            分组ID
     * @return 第一个元素为消息发送任务的ID,第二个元素为消息的数据ID，该字段只有在群发图文消息时，才会出现。
     * @see <a href=
     *      "https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140549&token=&lang=zh_CN">根据分组群发</a>

     * @see MpArticle
     * @throws WeixinException
     */
    @Deprecated
    public String[] massArticleByGroupId(List<MpArticle> articles, int groupId) throws WeixinException {
        String mediaId = uploadArticle(articles);
        return messageByGroupId(new MpNewsMessageTuple(mediaId), false, groupId);
    }

    /**
     * 群发消息
     * <p>
     * 在返回成功时,意味着群发任务提交成功,并不意味着此时群发已经结束,所以,仍有可能在后续的发送过程中出现异常情况导致用户未收到消息,
     * 如消息有时会进行审核、服务器不稳定等,此外,群发任务一般需要较长的时间才能全部发送完毕
     * </p>
     *
     * @param tuple
     *            消息元件
     * @param filter
     *            过滤条件
     * @return 第一个元素为消息发送任务的ID,第二个元素为消息的数据ID，该字段只有在群发图文消息时，才会出现,可以用于在图文分析数据接口中
     * @throws WeixinException
     * @see
     * @see cn.ipokerface.weixin.model.media.TextMessageTuple
     * @see cn.ipokerface.weixin.model.media.ImageMessageTuple
     * @see cn.ipokerface.weixin.model.media.VoiceMessageTuple
     * @see cn.ipokerface.weixin.model.media.MpVideoMessageTuple
     * @see MpNewsMessageTuple
     * @see cn.ipokerface.weixin.model.media.CardMessageTuple
     * @see MultiMessageTuple
     */
    private String[] message(MultiMessageTuple tuple, JSONObject filter) throws WeixinException {
        if (tuple instanceof MpNewsMessageTuple) {
            MpNewsMessageTuple _news = (MpNewsMessageTuple) tuple;
            List<MpArticle> _articles = _news.getArticles();
            if (StringUtil.isBlank(_news.getMediaId())) {
                if (_articles.isEmpty()) {
                    throw new WeixinException("mass fail:mediaId or articles is required");
                }
                tuple = new MpNewsMessageTuple(uploadArticle(_articles));
            }
            if (!filter.containsKey("send_ignore_reprint")) {
                filter.put("send_ignore_reprint", 0);
            }
        }
        String msgtype = tuple.getMessageType();
        JSONObject obj = new JSONObject();
        obj.putAll(filter);
        obj.put(msgtype, JSON.toJSON(tuple));
        obj.put("msgtype", msgtype);
        String mass_group_uri = WeixinMPlatformApis.message_group_uri;
        Token token = tokenManager.getCache();
        WeixinResponse response = weixinRequestClient.post(String.format(mass_group_uri, token.getAccessToken()),
                obj.toJSONString());

        obj = response.getAsJson();
        return new String[] { obj.getString("msg_id"), obj.getString("msg_data_id") };
    }

    /**
     * 群发消息给所有粉丝
     *
     * @param tuple
     *            消息元件
     * @return 第一个元素为消息发送任务的ID,第二个元素为消息的数据ID，该字段只有在群发图文消息时，才会出现,可以用于在图文分析数据接口中
     * @throws WeixinException
     * @see <a href=
     *      "https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140549&token=&lang=zh_CN">根据标签群发</a>
     */
    public String[] massToAll(MultiMessageTuple tuple) throws WeixinException {
        String filter = String.format("{\"filter\":{\"is_to_all\":true}}");
        return message(tuple, JSON.parseObject(filter));
    }

    /**
     * 标签群发消息
     *
     * @param tuple
     *            消息元件
     * @param tagId
     *            标签ID
     * @return 第一个元素为消息发送任务的ID,第二个元素为消息的数据ID，该字段只有在群发图文消息时，才会出现,可以用于在图文分析数据接口中
     * @throws WeixinException
     * @see cn.ipokerface.weixin.mp.model.tag.Tag
     * @see #message(MultiMessageTuple, JSONObject)
     * @see <a href=
     *      "https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140549&token=&lang=zh_CN">根据标签群发</a>
     */
    public String[] messageByTagId(MultiMessageTuple tuple, int tagId) throws WeixinException {
        String filter = String.format("{\"filter\":{\"is_to_all\":false,\"tag_id\":%d}}", tagId);
        return message(tuple, JSON.parseObject(filter));
    }

    /**
     * 标签群发图文消息
     *
     * @param articles
     *            图文列表
     * @param tagId
     *            标签ID
     * @param ignoreReprint
     *            图文消息被判定为转载时，是否继续群发
     * @return 第一个元素为消息发送任务的ID,第二个元素为消息的数据ID，该字段只有在群发图文消息时，才会出现。
     * @see <a href=
     *      "https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140549&token=&lang=zh_CN">根据标签群发</a>
     *
     * @see MpArticle
     * @throws WeixinException
     */
    public String[] massArticleByTagId(List<MpArticle> articles, int tagId, boolean ignoreReprint)
            throws WeixinException {
        String mediaId = uploadArticle(articles);
        String text = String.format("{\"filter\":{\"is_to_all\":false,\"tag_id\":%d}}", tagId);
        JSONObject filter = JSON.parseObject(text);
        filter.put("send_ignore_reprint", ignoreReprint ? 1 : 0);
        return message(new MpNewsMessageTuple(mediaId), filter);
    }

    /**
     * openId群发消息
     *
     * @param tuple
     *            消息元件
     * @param openIds
     *            openId列表
     * @return 第一个元素为消息发送任务的ID,第二个元素为消息的数据ID，该字段只有在群发图文消息时，才会出现,可以用于在图文分析数据接口中
     * @throws WeixinException
     * @see <a href=
     *      "https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140549&token=&lang=zh_CN">根据openid群发</a>
     *
     * @see #message(MultiMessageTuple, JSONObject)
     */
    public String[] messageByOpenIds(MultiMessageTuple tuple, String... openIds) throws WeixinException {
        JSONObject filter = new JSONObject();
        filter.put("touser", openIds);
        return message(tuple, filter);
    }

    /**
     * openid群发图文消息
     *
     * @param articles
     *            图文列表
     * @param ignoreReprint
     *            图文消息被判定为转载时，是否继续群发
     * @param openIds
     *            openId列表
     * @return 第一个元素为消息发送任务的ID,第二个元素为消息的数据ID，该字段只有在群发图文消息时，才会出现,可以用于在图文分析数据接口中.
     * @see <a href=
     *      "https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140549&token=&lang=zh_CN">根据openid群发</a>
     *
     * @see MpArticle
     * @throws WeixinException
     */
    public String[] messageArticleByOpenIds(List<MpArticle> articles, boolean ignoreReprint, String... openIds)
            throws WeixinException {
        String mediaId = uploadArticle(articles);
        JSONObject filter = new JSONObject();
        filter.put("touser", openIds);
        filter.put("send_ignore_reprint", ignoreReprint ? 1 : 0);
        return message(new MpNewsMessageTuple(mediaId), filter);
    }

    /**
     * 删除群发消息
     *
     * @param msgid
     *            发送出去的消息ID
     * @throws WeixinException
     * @see <a href=
     *      "https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140549&token=&lang=zh_CN">删除群发</a>
     * @see #deleteMessageNews(String, int)
     */
    public ApiResult deleteMessageNews(String msgid) throws WeixinException {
        return deleteMessageNews(msgid, 0);
    }

    /**
     * 删除群发消息
     * <p>
     * 请注意,只有已经发送成功的消息才能删除删除消息只是将消息的图文详情页失效,已经收到的用户,还是能在其本地看到消息卡片
     * </p>
     *
     * @param msgid
     *            发送出去的消息ID
     * @param articleIndex
     *            要删除的文章在图文消息中的位置，第一篇编号为1，该字段不填或填0会删除全部文章
     * @throws WeixinException
     * @see <a href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140549&token=&lang=zh_CN">删除群发</a>
     *
     */
    public ApiResult deleteMessageNews(String msgid, int articleIndex) throws WeixinException {
        JSONObject obj = new JSONObject();
        obj.put("msgid", msgid);
        if (articleIndex > 0)
            obj.put("article_idx", articleIndex);
        String mass_delete_uri = WeixinMPlatformApis.message_delete_uri;
        Token token = tokenManager.getCache();
        WeixinResponse response = weixinRequestClient.post(String.format(mass_delete_uri, token.getAccessToken()),
                obj.toJSONString());

        return response.getAsResult();
    }

    /**
     * 预览群发消息
     * 开发者可通过该接口发送消息给指定用户，在手机端查看消息的样式和排版
     *
     * @param toUser
     *            接收用户的openID
     * @param toWxName
     *            接收用户的微信号 towxname和touser同时赋值时，以towxname优先
     * @param tuple
     *            消息元件
     * @return 处理结果
     * @throws WeixinException
     * @see MultiMessageTuple
     * @see <a href=
     *      "https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140549&token=&lang=zh_CN">预览群发消息</a>
     */
    public ApiResult previewMessageNews(String toUser, String toWxName, MultiMessageTuple tuple) throws WeixinException {
        String msgtype = tuple.getMessageType();
        JSONObject obj = new JSONObject();
        obj.put("touser", toUser);
        obj.put("towxname", toWxName);
        obj.put(msgtype, JSON.toJSON(tuple));
        obj.put("msgtype", msgtype);
        String mass_preview_uri = WeixinMPlatformApis.message_preview_uri;
        Token token = tokenManager.getCache();
        WeixinResponse response = weixinRequestClient.post(String.format(mass_preview_uri, token.getAccessToken()),
                obj.toJSONString());

        return response.getAsResult();
    }

    /**
     * 查询群发发送状态
     *
     * @param msgId
     *            消息ID
     * @return 消息发送状态,如sendsuccess:发送成功、sendfail:发送失败
     * @throws WeixinException
     * @see <a href=
     *      "https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140549&token=&lang=zh_CN">查询群发状态</a>
     */
    public String getMessageNewStatus(String msgId) throws WeixinException {
        JSONObject obj = new JSONObject();
        obj.put("msg_id", msgId);
        String mass_get_uri = WeixinMPlatformApis.message_get_uri;
        Token token = tokenManager.getCache();
        WeixinResponse response = weixinRequestClient.post(String.format(mass_get_uri, token.getAccessToken()),
                obj.toJSONString());

        String status = response.getAsJson().getString("msg_status");
        String desc = massStatusMap.get(status);
        return String.format("%s:%s", status, desc);
    }

    private final static Map<String, String> massStatusMap;
    static {
        massStatusMap = new HashMap<String, String>();
        massStatusMap.put("sendsuccess", "发送成功");
        massStatusMap.put("send_success", "发送成功");
        massStatusMap.put("success", "发送成功");
        massStatusMap.put("send success", "发送成功");
        massStatusMap.put("sendfail", "发送失败");
        massStatusMap.put("send_fail", "发送失败");
        massStatusMap.put("fail", "发送失败");
        massStatusMap.put("send fail", "发送失败");
        massStatusMap.put("err(10001)", "涉嫌广告");
        massStatusMap.put("err(20001)", "涉嫌政治");
        massStatusMap.put("err(20004)", "涉嫌社会");
        massStatusMap.put("err(20006)", "涉嫌违法犯罪");
        massStatusMap.put("err(20008)", "涉嫌欺诈");
        massStatusMap.put("err(20013)", "涉嫌版权");
        massStatusMap.put("err(22000)", "涉嫌互推(互相宣传)");
        massStatusMap.put("err(21000)", "涉嫌其他");
    }

}
