package cn.ipokerface.weixin.mp.proxy;

import cn.ipokerface.weixin.exception.WeixinException;
import cn.ipokerface.weixin.model.Token;
import cn.ipokerface.weixin.model.page.PageData;
import cn.ipokerface.weixin.model.page.Pageable;
import cn.ipokerface.weixin.mp.model.article.ArticleComment;
import cn.ipokerface.weixin.mp.request.WeixinMPlatformApis;
import cn.ipokerface.weixin.request.ApiResult;
import cn.ipokerface.weixin.request.WeixinResponse;
import cn.ipokerface.weixin.token.TokenManager;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by       PokerFace
 * Create Date      2019-12-28.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class ArticleProxy extends MediaProxy {

    public ArticleProxy(TokenManager tokenManager) {
        super(tokenManager);
    }


    /**
     * 打开/关闭已群发文章评论
     *
     *
     * @param open
     *            true为打开，false为关闭
     * @param msgid
     *            群发返回的msg_data_id
     * @param index
     *            多图文时，用来指定第几篇图文，从0开始，不带默认操作该msg_data_id的第一篇图文
     * @return 操作结果
     * @throws WeixinException
     */
    public ApiResult openComment(boolean open, String msgid, int index) throws WeixinException {
        String news_comment = open ? WeixinMPlatformApis.news_comment_open : WeixinMPlatformApis.news_comment_close;
        Token token = tokenManager.getCache();
        JSONObject obj = new JSONObject();
        obj.put("msg_data_id", msgid);
        obj.put("index", index);
        WeixinResponse response = weixinRequestClient.post(String.format(news_comment, token.getAccessToken()),
                obj.toJSONString());

        return response.getAsResult();
    }

    /**
     * 获取评论列表
     *
     * @see ArticleComment
     * @see cn.ipokerface.weixin.mp.model.article.ArticleComment.ArticleCommentType
     *
     * @param page
     *            分页信息
     * @param commentType
     *            评论类型 为空获取全部类型
     * @param msgid
     *            群发返回的msg_data_id
     * @param index
     *            多图文时，用来指定第几篇图文，从0开始，不带默认操作该msg_data_id的第一篇图文
     * @return 分页数据
     * @throws WeixinException
     */
    public PageData<ArticleComment> listArticleComments(Pageable page, ArticleComment.ArticleCommentType commentType, String msgid,
                                                        int index) throws WeixinException {
        String news_comment_list = WeixinMPlatformApis.news_comment_list;
        Token token = tokenManager.getCache();
        JSONObject obj = new JSONObject();
        obj.put("msg_data_id", "msgid");
        obj.put("index", index);
        obj.put("begin", page.getOffset());
        obj.put("count", Math.max(50, page.getPageSize())); // 获取数目（>=50会被拒绝）
        if (commentType != null) {
            obj.put("type", commentType.ordinal() + 1);
        } else {
            obj.put("type", 0);
        }
        WeixinResponse response = weixinRequestClient.post(String.format(news_comment_list, token.getAccessToken()),
                obj.toJSONString());

        JSONObject result = response.getAsJson();
        int total = result.getIntValue("total");
        List<ArticleComment> content = JSON.parseArray(result.getString("comment"), ArticleComment.class);
        return new PageData<ArticleComment>(page, total, content);
    }

    /**
     * 获取评论列表
     * @see #listArticleComments(Pageable, ArticleComment.ArticleCommentType, String, int)
     *
     * @param commentType
     *            评论类型 为空获取全部类型
     * @param msgid
     *            群发返回的msg_data_id
     * @param index
     *            多图文时，用来指定第几篇图文，从0开始，不带默认操作该msg_data_id的第一篇图文
     * @return 分页数据
     *
     * @throws WeixinException
     */
    public List<ArticleComment> listAllArticleComments(ArticleComment.ArticleCommentType commentType, String msgid, int index)
            throws WeixinException {
        List<ArticleComment> comments = new ArrayList<ArticleComment>();
        PageData<ArticleComment> page = null;
        Pageable pageable = null;
        for (pageable = new Pageable(1, 50);; pageable = pageable.next()) {
            page = listArticleComments(pageable, commentType, msgid, index);
            if (!page.hasContent()) {
                break;
            }
            comments.addAll(page.getContent());
        }
        return comments;
    }

    /**
     * 评论标记/取消精选
     *
     * @param markelect
     *            true为标记，false为取消
     * @param msgid
     *            群发返回的msg_data_id
     * @param index
     *            多图文时，用来指定第几篇图文，从0开始，不带默认操作该msg_data_id的第一篇图文
     * @param commentId
     *            用户评论ID
     * @return 操作结果
     * @see #listArticleComments(Pageable,ArticleComment.ArticleCommentType, String, int)
     * @throws WeixinException
     */
    public ApiResult markelectComment(boolean markelect, String msgid, int index, String commentId)
            throws WeixinException {
        String news_comment = markelect ? WeixinMPlatformApis.news_comment_markelect
                : WeixinMPlatformApis.news_comment_unmarkelect;
        Token token = tokenManager.getCache();
        JSONObject obj = new JSONObject();
        obj.put("msg_data_id", "msgid");
        obj.put("index", index);
        obj.put("user_comment_id", commentId);
        WeixinResponse response = weixinRequestClient.post(String.format(news_comment, token.getAccessToken()),
                obj.toJSONString());

        return response.getAsResult();
    }

    /**
     * 删除评论
     *
     * @param msgid
     *            群发返回的msg_data_id
     * @param index
     *            多图文时，用来指定第几篇图文，从0开始，不带默认操作该msg_data_id的第一篇图文
     * @param commentId
     *            用户评论ID
     * @return 操作结果
     * @see #listArticleComments(Pageable, ArticleComment.ArticleCommentType, String, int)
     * @throws WeixinException
     */
    public ApiResult deleteComment(String msgid, int index, String commentId) throws WeixinException {
        String news_comment_delete = WeixinMPlatformApis.news_comment_delete;
        Token token = tokenManager.getCache();
        JSONObject obj = new JSONObject();
        obj.put("msg_data_id", "msgid");
        obj.put("index", index);
        obj.put("user_comment_id", commentId);
        WeixinResponse response = weixinRequestClient.post(String.format(news_comment_delete, token.getAccessToken()),
                obj.toJSONString());

        return response.getAsResult();
    }

    /**
     * 回复评论
     *
     * @param msgid
     *            群发返回的msg_data_id
     * @param index
     *            多图文时，用来指定第几篇图文，从0开始，不带默认操作该msg_data_id的第一篇图文
     * @param commentId
     *            用户评论ID
     * @param content
     *            回复内容
     * @return 操作结果
     * @see #listArticleComments(Pageable, ArticleComment.ArticleCommentType, String, int)
     * @throws WeixinException
     */
    public ApiResult replyComment(String msgid, int index, String commentId, String content) throws WeixinException {
        String news_comment_reply = WeixinMPlatformApis.news_comment_reply_add;
        Token token = tokenManager.getCache();
        JSONObject obj = new JSONObject();
        obj.put("msg_data_id", "msgid");
        obj.put("index", index);
        obj.put("user_comment_id", commentId);
        obj.put("content", content);
        WeixinResponse response = weixinRequestClient.post(String.format(news_comment_reply, token.getAccessToken()),
                obj.toJSONString());

        return response.getAsResult();
    }

    /**
     * 删除回复
     *
     * @param msgid
     *            群发返回的msg_data_id
     * @param index
     *            多图文时，用来指定第几篇图文，从0开始，不带默认操作该msg_data_id的第一篇图文
     * @param commentId
     *            用户评论ID
     * @return 操作结果
     * @see #listArticleComments(Pageable, ArticleComment.ArticleCommentType, String, int)
     * @throws WeixinException
     */
    public ApiResult deleteCommentReply(String msgid, int index, String commentId) throws WeixinException {
        String news_comment_reply = WeixinMPlatformApis.news_comment_reply_delete;
        Token token = tokenManager.getCache();
        JSONObject obj = new JSONObject();
        obj.put("msg_data_id", "msgid");
        obj.put("index", index);
        obj.put("user_comment_id", commentId);
        WeixinResponse response = weixinRequestClient.post(String.format(news_comment_reply, token.getAccessToken()),
                obj.toJSONString());

        return response.getAsResult();
    }
}
