package cn.ipokerface.weixin.mp.proxy;

import cn.ipokerface.weixin.Constant;
import cn.ipokerface.weixin.cache.AbstractCacheStorager;
import cn.ipokerface.weixin.cache.MemoryCacheStorager;
import cn.ipokerface.weixin.exception.WeixinException;
import cn.ipokerface.weixin.model.Token;
import cn.ipokerface.weixin.model.WeixinAccount;
import cn.ipokerface.weixin.mp.model.WeixinMPlatformAccount;
import cn.ipokerface.weixin.mp.request.WeixinMPlatformApis;
import cn.ipokerface.weixin.token.TicketManager;
import cn.ipokerface.weixin.utils.StringUtil;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by       PokerFace
 * Create Date      2019-12-28.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class WeixinComponentProxy {

    /**
     * 每个组件授权不一样 componentId - componentApi
     */
    private Map<String, ComponentProxy> componentMap;
    /**
     * 微信账号信息
     */
    private final WeixinMPlatformAccount weixinMpAccount;


    private String componentAuthRedirectUrl;

    private String userAuthRedirectUrl;

    private String openUserAuthRedirectUrl;


    public WeixinComponentProxy(WeixinMPlatformAccount weixinMPlatformAccount){
        this(weixinMPlatformAccount, new MemoryCacheStorager<Token>(), null, null, null);
    }

    public WeixinComponentProxy(WeixinMPlatformAccount weixinMPlatformAccount, AbstractCacheStorager<Token> cacheStorager,
                                String componentAuthRedirectUrl,
                                String userAuthRedirectUrl,
                                String openUserAuthRedirectUrl){
        if (weixinMPlatformAccount == null) {
            throw new IllegalArgumentException("weixinMpAccount must not be empty");
        }
        if (cacheStorager == null) {
            throw new IllegalArgumentException("cacheStorager must not be empty");
        }

        this.weixinMpAccount  = weixinMPlatformAccount;
        this.componentAuthRedirectUrl = componentAuthRedirectUrl;
        this.userAuthRedirectUrl = userAuthRedirectUrl;
        this.openUserAuthRedirectUrl = openUserAuthRedirectUrl;

        this.componentMap = new HashMap<String, ComponentProxy>(weixinMpAccount.getComponents().size());
        for (WeixinAccount component : weixinMpAccount.getComponents()) {

            this.componentMap.put(component.getId(),
                    new ComponentProxy(new TicketManager(component.getId(), component.getSecret(), cacheStorager)));
        }
        this.componentMap.put(null, componentMap.get(weixinMpAccount.getComponents().get(0).getId()));

    }


    /**
     * 获取微信账号信息
     *
     * @return
     */
    public WeixinMPlatformAccount getWeixinMpAccount() {
        return this.weixinMpAccount;
    }

    /**
     * 获取组接口对象(只关注第一个组件
     *
     * @see ComponentProxy
     * @return API实例
     */
    public ComponentProxy component() {
        return this.componentMap.get(null);
    }

    /**
     * 获取套件接口对象(多个组件
     *
     * @see ComponentProxy
     * @param componentId
     *            组件ID
     * @return API实例
     */
    public ComponentProxy component(String componentId) {
        return this.componentMap.get(componentId);
    }



    /**
     * 获取组件的预授权码 <font color="red">需先缓存ticket</font>
     *
     * @param componentId
     *            组件ID
     * @return 预授权码
     * @see #cacheComponentTicket(String, String)
     * @see ComponentProxy
     * @see ComponentProxy#getTicketManager()
     * @see ComponentProxy#getPreCodeManager()
     * @throws WeixinException
     */
    public String getPreComponentTicket(String componentId) throws WeixinException {
        ComponentProxy component = component(componentId);
        Token token = component.getTicketManager().getTicket();
        if (token == null || StringUtil.isBlank(token.getAccessToken())) {
            throw new WeixinException("maybe oauth first?");
        }
        return component.getPreCodeManager().getAccessToken();
    }

    /**
     * 缓存组件ticket
     *
     * @param componentId
     *            组件ID
     * @param componentTicket
     *            组件ticket内容
     * @throws WeixinException
     */
    public void cacheComponentTicket(String componentId, String componentTicket) throws WeixinException {
        component(componentId).getTicketManager().cachingTicket(componentTicket);
    }

    /**
     * 应用组件授权 <font color="red">需先缓存ticket</font>
     * redirectUri默认填写weixin4j.properties#component.oauth.redirect.uri
     * state默认填写state
     *
     * @param componentId
     *            组件ID
     * @return 请求授权的URL
     * @throws WeixinException
     */
    public String getComponentAuthorizationURL(String componentId) throws WeixinException {
        return getComponentAuthorizationURL(componentId, componentAuthRedirectUrl, "state");
    }

    /**
     * 应用组件授权 <font color="red">需先缓存ticket，在授权完成之后需要调用ComponentApi#
     * exchangeAuthorizerToken方法 ,否则无法缓存token相关导致后续的组件接口调用失败</font>
     *
     * @param componentId
     *            组件ID
     * @param redirectUri
     *            授权后重定向url
     * @param state
     *            回调后原样返回
     * @see #cacheComponentTicket(String, String)
     * @see ComponentProxy
     * @see ComponentProxy#getTicketManager()
     * @see ComponentProxy#getPreCodeManager()
     * @see ComponentProxy#exchangeAuthorizerToken(String)
     * @see <a href=
     *      "https://open.weixin.qq.com/cgi-bin/showdocument?action=dir_list&t=resource/res_list&verify=1&id=open1453779503&token=&lang=zh_CN">
     *      应用组件授权</a>
     * @return 请求授权的URL
     * @throws WeixinException
     */
    public String getComponentAuthorizationURL(String componentId, String redirectUri, String state)
            throws WeixinException {
        try {
            return String.format(WeixinMPlatformApis.component_oauth_url, componentId, getPreComponentTicket(componentId),
                    URLEncoder.encode(redirectUri, Constant.UTF_8.name()), state);
        } catch (UnsupportedEncodingException e) {
            ;
        }
        return "";
    }

    /**
     * 创建WeixinProxy对象
     *
     * @param componentId
     *            组件ID
     * @param authAppId
     *            已授权的appid
     * @see WeixinMPlatformProxy
     * @return
     */
    public WeixinMPlatformProxy getWeixinProxy(String componentId, String authAppId) {
        return new WeixinMPlatformProxy(component(componentId).getRefreshTokenManager(authAppId),
                component(componentId).getTokenManager(), userAuthRedirectUrl, openUserAuthRedirectUrl);
    }

}
