package cn.ipokerface.weixin.mp.proxy;

import cn.ipokerface.weixin.exception.WeixinException;
import cn.ipokerface.weixin.model.Token;
import cn.ipokerface.weixin.mp.message.TemplateMessage;
import cn.ipokerface.weixin.mp.message.TemplateMessageInfo;
import cn.ipokerface.weixin.mp.model.IndustryType;
import cn.ipokerface.weixin.mp.request.WeixinMPlatformApis;
import cn.ipokerface.weixin.request.ApiResult;
import cn.ipokerface.weixin.request.WeixinResponse;
import cn.ipokerface.weixin.request.http.NameValue;
import cn.ipokerface.weixin.token.TokenManager;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.NameFilter;

import java.util.List;

/**
 * Created by       PokerFace
 * Create Date      2019-12-28.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class TemplateMessageProxy extends MPlatformProxy {

    public TemplateMessageProxy(TokenManager tokenManager){
        super(tokenManager);
    }


    /**
     * 设置所属行业(每月可修改行业1次，账号仅可使用所属行业中相关的模板)
     *
     * @param industryTypes
     *            所处行业 目前不超过两个
     * @return 操作结果
     * @throws WeixinException
     * @see IndustryType
     * @see <a
     *      href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1433751277&token=&lang=zh_CN">设置所处行业</a>
     */
    public ApiResult setTemplateIndustry(IndustryType... industryTypes)
            throws WeixinException {
        JSONObject obj = new JSONObject();
        for (int i = 0; i < industryTypes.length; i++) {
            obj.put(String.format("industry_id%d", i + 1),
                    Integer.toString(industryTypes[i].getTypeId()));
        }
        Token token = tokenManager.getCache();
        String template_set_industry_uri = WeixinMPlatformApis.template_set_industry_uri;
        WeixinResponse response = weixinRequestClient.post(String.format(
                template_set_industry_uri, token.getAccessToken()), obj
                .toJSONString());

        return response.getAsResult();
    }

    /**
     * 获取设置的行业信息
     *
     * @return 行业信息数组 第一个元素为帐号设置的主营行业 第二个元素为帐号设置的副营行业
     * @throws WeixinException
     * @see IndustryType
     * @see <a
     *      href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1433751277&token=&lang=zh_CN">获取设置的行业信息</a>
     */
    public IndustryType[] getTemplateIndustry() throws WeixinException {
        String template_get_industry_uri = WeixinMPlatformApis.template_get_industry_uri;
        WeixinResponse response = weixinRequestClient.get(String.format(
                template_get_industry_uri, tokenManager.getAccessToken()));
        JSONObject primary = response.getAsJson().getJSONObject(
                "primary_industry");
        JSONObject secondary = response.getAsJson().getJSONObject(
                "secondary_industry");
        return new IndustryType[] {
                primary != null ? IndustryType.getIndustry(
                        primary.getString("first_class"),
                        primary.getString("second_class")) : null,
                secondary != null ? IndustryType.getIndustry(
                        secondary.getString("first_class"),
                        secondary.getString("second_class")) : null };
    }

    /**
     * 获取模板ID
     *
     * @param shortId
     *            模板库中模板的编号，有“TM**”和“OPENTMTM**”等形式
     * @return 模板ID
     * @throws WeixinException
     * @see <a
     *      href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1433751277&token=&lang=zh_CN">获得模板ID</a>
     */
    public String getTemplateId(String shortId) throws WeixinException {
        Token token = tokenManager.getCache();
        String template_getid_uri = WeixinMPlatformApis.template_get_id_uri;
        WeixinResponse response = weixinRequestClient.post(
                String.format(template_getid_uri, token.getAccessToken()),
                String.format("{\"template_id_short\":\"%s\"}", shortId));

        return response.getAsJson().getString("template_id");
    }

    /**
     * 获取模板列表
     *
     * @return 模板列表
     * @see TemplateMessageInfo
     * @see <a
     *      href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1433751277&token=&lang=zh_CN">获取模板列表</a>
     * @throws WeixinException
     */
    public List<TemplateMessageInfo> getAllTemplates() throws WeixinException {
        Token token = tokenManager.getCache();
        String template_getall_uri = WeixinMPlatformApis.template_get_all_uri;
        WeixinResponse response = weixinRequestClient.get(String.format(
                template_getall_uri, token.getAccessToken()));
        return JSON.parseArray(response.getAsJson().getString("template_list"),
                TemplateMessageInfo.class);
    }

    /**
     * 删除模板
     *
     * @param templateId
     *            公众帐号下模板消息ID
     * @return 处理结果
     * @see <a
     *      href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1433751277&token=&lang=zh_CN">删除模板</a>
     * @throws WeixinException
     */
    public ApiResult deleteTemplate(String templateId) throws WeixinException {
        Token token = tokenManager.getCache();
        String template_del_uri = WeixinMPlatformApis.template_del_uri;
        WeixinResponse response = weixinRequestClient.post(
                String.format(template_del_uri, token.getAccessToken()),
                String.format("{\"template_id\":\"%s\"}", templateId));
        return response.getAsResult();
    }

    /**
     * 发送模板消息
     *
     * @param tplMessage
     *            消息对象
     * @return 发送的消息ID
     * @throws WeixinException
     * @see <a
     *      href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1433751277&token=&lang=zh_CN">模板消息</a>
     * @see <a href=
     *      "https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1433751288&token=&lang=zh_CN"
     *      >运营规范</a>
     * @see TemplateMessage
     */
    public String sendTemplateMessage(TemplateMessage tplMessage)
            throws WeixinException {
        Token token = tokenManager.getCache();
        String template_send_uri = WeixinMPlatformApis.template_send_uri;
        WeixinResponse response = weixinRequestClient.post(
                String.format(template_send_uri, token.getAccessToken()),
                JSON.toJSONString(tplMessage, new NameFilter() {
                    @Override
                    public String process(Object object, String name,
                                          Object value) {
                        if (object instanceof NameValue && name.equals("name")) {
                            return "color";
                        }
                        return name;
                    }
                }));

        return response.getAsJson().getString("msgid");
    }
}
