package cn.ipokerface.weixin.request.http.netty;

import cn.ipokerface.weixin.Constant;
import cn.ipokerface.weixin.concurrent.SettableFuture;
import cn.ipokerface.weixin.request.http.*;
import cn.ipokerface.weixin.request.http.entity.HttpEntity;
import cn.ipokerface.weixin.utils.StringUtil;
import io.netty.bootstrap.Bootstrap;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufOutputStream;
import io.netty.channel.*;
import io.netty.handler.codec.http.DefaultFullHttpRequest;
import io.netty.handler.codec.http.DefaultHttpRequest;
import io.netty.handler.codec.http.FullHttpResponse;
import io.netty.handler.ssl.SslHandler;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.URI;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;

/**
 * Created by       PokerFace
 * Create Date      2019-12-27.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class Netty4HttpClient extends AbstractHttpClient {

    private final Bootstrap bootstrap;
    private final HttpParams params;

    public Netty4HttpClient(Bootstrap bootstrap, HttpParams params) {
        this.bootstrap = bootstrap;
        this.params = params;
    }

    @Override
    public HttpResponse execute(final HttpRequest request)
            throws HttpClientException {
        HttpResponse response = null;
        try {
            final URI uri = request.getURI();
            final SettableFuture<HttpResponse> future = new SettableFuture<HttpResponse>();
            ChannelFutureListener listener = new ChannelFutureListener() {
                @Override
                public void operationComplete(ChannelFuture channelFuture)
                        throws Exception {
                    if (channelFuture.isSuccess()) {
                        Channel channel = channelFuture.channel();
                        if ("https".equals(uri.getScheme())) {
                            SSLContext sslContext;
                            if (params != null
                                    && params.getSSLContext() != null) {
                                sslContext = params.getSSLContext();
                            } else {
                                sslContext = HttpClientFactory
                                        .allowSSLContext();
                            }
                            SSLEngine sslEngine = sslContext.createSSLEngine();
                            sslEngine.setUseClientMode(true);
                            channel.pipeline().addFirst(
                                    new SslHandler(sslEngine));
                        }
                        channel.pipeline().addLast(new RequestHandler(future));
                        DefaultHttpRequest uriRequest = createRequest(request);
                        channel.writeAndFlush(uriRequest);
                    } else {
                        future.setException(channelFuture.cause());
                    }
                }
            };
            InetSocketAddress address = params != null
                    && params.getProxy() != null ? (InetSocketAddress) params
                    .getProxy().address() : new InetSocketAddress(
                    InetAddress.getByName(uri.getHost()), getPort(uri));
            bootstrap.connect(address).addListener(listener);
            response = future.get();
            handleResponse(response);
        } catch (IOException e) {
            throw new HttpClientException("I/O error on "
                    + request.getMethod().name() + " request for \""
                    + request.getURI().toString(), e);
        } catch (InterruptedException e) {
            throw new HttpClientException("Execute error on "
                    + request.getMethod().name() + " request for \""
                    + request.getURI().toString(), e);
        } catch (ExecutionException e) {
            throw new HttpClientException("Execute error on "
                    + request.getMethod().name() + " request for \""
                    + request.getURI().toString(), e);
        } finally {
            if (response != null) {
                response.close();
            }
        }
        return response;
    }

    private DefaultHttpRequest createRequest(HttpRequest request)
            throws IOException {
        io.netty.handler.codec.http.HttpMethod method = io.netty.handler.codec.http.HttpMethod.valueOf(request.getMethod().name());
        URI uri = request.getURI();
        String url = StringUtil.isBlank(uri.getRawPath()) ? "/" : uri
                .getRawPath();
        if (StringUtil.isNotBlank(uri.getRawQuery())) {
            url += "?" + uri.getRawQuery();
        }
        DefaultHttpRequest uriRequest = new DefaultHttpRequest(
                io.netty.handler.codec.http.HttpVersion.HTTP_1_1, method, url);
        // entity
        HttpEntity entity = request.getEntity();
        if (entity != null) {
            ByteBuf byteBuf = ByteBufAllocator.DEFAULT.buffer();
            ByteBufOutputStream out = new ByteBufOutputStream(byteBuf);
            entity.writeTo(out);
            out.flush();
            out.close();
            uriRequest = new DefaultFullHttpRequest(
                    uriRequest.getProtocolVersion(), uriRequest.getMethod(),
                    uriRequest.getUri(), byteBuf);
            if (entity.getContentType() != null) {
                uriRequest.headers().add(HttpHeaders.CONTENT_TYPE,
                        entity.getContentType().toString());
            }
            if (entity.getContentLength() < 0) {
                uriRequest.headers().add(HttpHeaders.TRANSFER_ENCODING,
                        io.netty.handler.codec.http.HttpHeaders.Values.CHUNKED);
            } else {
                uriRequest.headers().add(HttpHeaders.CONTENT_LENGTH,
                        entity.getContentLength());
            }
        }
        // header
        HttpHeaders headers = request.getHeaders();
        if (headers == null) {
            headers = new HttpHeaders();
        }
        if (!headers.containsKey(HttpHeaders.HOST)) {
            headers.set(HttpHeaders.HOST, uri.getHost());
        }
        // Add default accept headers
        if (!headers.containsKey(HttpHeaders.ACCEPT)) {
            headers.set(HttpHeaders.ACCEPT, "*/*");
        }
        // Add default user agent
        if (!headers.containsKey(HttpHeaders.USER_AGENT)) {
            headers.set(HttpHeaders.USER_AGENT, "netty/httpclient");
        }
        for (Map.Entry<String, List<String>> header : headers.entrySet()) {
            uriRequest.headers().set(header.getKey(), header.getValue());
        }
        uriRequest.headers().set(HttpHeaders.ACCEPT_CHARSET,
                Constant.UTF_8.displayName());
        uriRequest.headers().set(HttpHeaders.CONNECTION,
                io.netty.handler.codec.http.HttpHeaders.Values.CLOSE);
        return uriRequest;
    }

    private int getPort(URI uri) {
        int port = uri.getPort();
        if (port == -1) {
            if ("http".equalsIgnoreCase(uri.getScheme())) {
                port = 80;
            } else if ("https".equalsIgnoreCase(uri.getScheme())) {
                port = 443;
            }
        }
        return port;
    }

    private static class RequestHandler extends
            SimpleChannelInboundHandler<FullHttpResponse> {

        private final SettableFuture<HttpResponse> future;

        public RequestHandler(SettableFuture<HttpResponse> future) {
            this.future = future;
        }

        @Override
        protected void channelRead0(ChannelHandlerContext context,
                                    FullHttpResponse response) throws Exception {
            byte[] content = null;
            ByteBuf byteBuf = response.content();
            if (byteBuf.hasArray()) {
                content = byteBuf.array();
            } else {
                content = new byte[byteBuf.readableBytes()];
                byteBuf.readBytes(content);
            }
            future.set(new Netty4HttpResponse(context, response, content));
        }

        @Override
        public void exceptionCaught(ChannelHandlerContext context,
                                    Throwable cause) throws Exception {
            future.setException(cause);
        }
    }
}