package cn.ipokerface.weixin.jssdk;

import cn.ipokerface.weixin.exception.WeixinException;
import cn.ipokerface.weixin.token.TokenManager;
import cn.ipokerface.weixin.utils.*;
import com.alibaba.fastjson.JSONObject;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Created by       PokerFace
 * Create Date      2019-12-28.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class JssdkConfigurator {
    private final TokenManager ticketTokenManager;
    private JSONObject config;
    private Set<JssdkApis> apis;

    /**
     * ticket保存类 可调用WeixinProxy#getTicketManager获取
     *
     * @param ticketTokenManager
     */
    public JssdkConfigurator(TokenManager ticketTokenManager) {
        this.ticketTokenManager = ticketTokenManager;
        this.config = new JSONObject();
        this.apis = new HashSet<JssdkApis>();
    }

    /**
     * 开启调试模式,调用的所有api的返回值会在客户端alert出来，若要查看传入的参数，可以在pc端打开，参数信息会通过log打出，
     * 仅在pc端时才会打印。
     *
     * @return
     */
    public JssdkConfigurator debugMode() {
        config.put("debug", true);
        return this;
    }

    /**
     * 需要使用的JS接口列表
     *
     * @see JssdkApis
     * @param apis
     * @return
     */
    public JssdkConfigurator apis(JssdkApis... apis) {
        for (JssdkApis api : apis) {
            this.apis.add(api);
        }
        return this;
    }

    /**
     * 需要使用的JS接口列表
     *
     * @see JssdkApis
     * @param apis
     * @return
     */
    public JssdkConfigurator apis(JssdkApis[]... apis) {
        for (JssdkApis[] api : apis) {
            apis(api);
        }
        return this;
    }

    /**
     * 生成config配置JSON串
     *
     * @param url
     *            当前网页的URL，不包含#及其后面部分
     * @return jssdk配置JSON字符串
     * @see <a href=
     *      "https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421141115&token=&lang=zh_CN">公众号JSSDK</a>
     * @see <a href=
     *      "http://qydev.weixin.qq.com/wiki/index.php?title=%E5%BE%AE%E4%BF%A1JS-SDK%E6%8E%A5%E5%8F%A3">企业号JSSDK</a>
     * @throws WeixinException
     */
    public String toJSONConfig(String url) throws WeixinException {
        if (apis.isEmpty()) {
            throw new WeixinException("jsapilist not be empty");
        }
        Map<String, String> signMap = new HashMap<String, String>();
        String timestamp = DateUtils.timestamp2string();
        String noncestr = RandomUtil.generateString(24);
        signMap.put("timestamp", timestamp);
        signMap.put("noncestr", noncestr);
        signMap.put("jsapi_ticket", this.ticketTokenManager.getAccessToken());
        signMap.put("url", url);
        String sign = DigestUtils.SHA1(MapUtils.toJoinString(signMap, false, false));
        config.put("appId", ticketTokenManager.getWeixinId());
        if (StringUtil.isBlank(config.getString("debug"))) {
            config.put("debug", false);
        }
        config.put("timestamp", timestamp);
        config.put("nonceStr", noncestr);
        config.put("signature", sign);
        config.put("jsApiList", apis.toArray());
        return config.toJSONString();
    }
}