package cn.ipokerface.weixin.cache;

import cn.ipokerface.weixin.Constant;
import cn.ipokerface.weixin.utils.SerializationUtils;
import redis.clients.jedis.HostAndPort;
import redis.clients.jedis.JedisCluster;
import redis.clients.jedis.JedisPoolConfig;

import java.util.Set;

/**
 * Created by       PokerFace
 * Create Date      2020-01-12.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class RedisClusterCacheStorager<T extends Cacheable> implements
        AbstractCacheStorager<T>  {

    private String keys = "weixin_cache_keys";

    private final static int CONNECTION_TIMEOUT = 5000;
    private final static int SO_TIMEOUT = 5000;
    private final static int MAX_REDIRECTIONS = 5;
    private final static int MAX_TOTAL = 50;
    private final static int MAX_IDLE = 5;
    private final static int MAX_WAIT_MILLIS = 5000;
    private final static boolean TEST_ON_BORROW = false;
    private final static boolean TEST_ON_RETURN = true;
    private final JedisCluster jedisCluster;

    public RedisClusterCacheStorager(Set<HostAndPort> nodes) {
        JedisPoolConfig poolConfig = new JedisPoolConfig();
        poolConfig.setMaxTotal(MAX_TOTAL);
        poolConfig.setMaxIdle(MAX_IDLE);
        poolConfig.setMaxWaitMillis(MAX_WAIT_MILLIS);
        poolConfig.setTestOnBorrow(TEST_ON_BORROW);
        poolConfig.setTestOnReturn(TEST_ON_RETURN);
        this.jedisCluster = new JedisCluster(nodes, CONNECTION_TIMEOUT,
                SO_TIMEOUT, MAX_REDIRECTIONS, poolConfig);
    }

    public RedisClusterCacheStorager(Set<HostAndPort> nodes,
                                     JedisPoolConfig poolConfig) {
        this(nodes, CONNECTION_TIMEOUT, SO_TIMEOUT, MAX_REDIRECTIONS,
                poolConfig);
    }

    public RedisClusterCacheStorager(Set<HostAndPort> nodes,
                                     int connectionTimeout, int soTimeout, int maxRedirections,
                                     JedisPoolConfig poolConfig) {
        this(new JedisCluster(nodes, connectionTimeout, soTimeout,
                maxRedirections, poolConfig));
    }

    public RedisClusterCacheStorager(JedisCluster jedisCluster) {
        this.jedisCluster = jedisCluster;
    }




    @Override
    public T lookup(String key) {
        byte[] value = jedisCluster.get(key.getBytes(Constant.UTF_8));
        return value != null ? (T) SerializationUtils.deserialize(value) : null;
    }

    @Override
    public void caching(String key, T cache) {
        byte[] cacheKey = key.getBytes(Constant.UTF_8);
        byte[] value = SerializationUtils.serialize(cache);
        if (cache.expiredTime() > 0) {
            jedisCluster.setex(cacheKey,
                    (int) (cache.expiredTime() - timeout) / 1000, value);
        } else {
            jedisCluster.set(cacheKey, value);
        }
        jedisCluster.sadd(keys, key);
    }

    @Override
    public T delete(String key) {
        T cache = lookup(key);
        jedisCluster.del(key);
        jedisCluster.srem(keys, key);
        return cache;
    }

    @Override
    public void clear() {
        Set<String> cacheKeys = jedisCluster.smembers(keys);
        if (!cacheKeys.isEmpty()) {
            cacheKeys.add(keys);
            jedisCluster.del(cacheKeys.toArray(new String[cacheKeys.size()]));
        }
    }
}
