package cn.ipokerface.weixin.cache;

import cn.ipokerface.weixin.Constant;
import cn.ipokerface.weixin.utils.SerializationUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;
import redis.clients.jedis.util.Pool;

import java.util.Set;

/**
 * Created by       PokerFace
 * Create Date      2019-12-27.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class RedisCacheStorager<T extends Cacheable> implements AbstractCacheStorager<T> {

    private static final String keys = "weixin_cache_keys";

    private Pool<Jedis> jedisPool;

    private final static int TIMEOUT = 5000;
    private final static int MAX_TOTAL = 50;
    private final static int MAX_IDLE = 5;
    private final static int MAX_WAIT_MILLIS = 5000;
    private final static boolean TEST_ON_BORROW = false;
    private final static boolean TEST_ON_RETURN = true;
    private final static JedisPoolConfig POOLCONFIG;
    static {
        POOLCONFIG = new JedisPoolConfig();
        POOLCONFIG.setMaxTotal(MAX_TOTAL);
        POOLCONFIG.setMaxIdle(MAX_IDLE);
        POOLCONFIG.setMaxWaitMillis(MAX_WAIT_MILLIS);
        POOLCONFIG.setTestOnBorrow(TEST_ON_BORROW);
        POOLCONFIG.setTestOnReturn(TEST_ON_RETURN);
    }


    public RedisCacheStorager(String host, int port, int timeout) {
        this(host, port, timeout, null, POOLCONFIG);
    }

    public RedisCacheStorager(String host, int port, int timeout, String password) {
        this(host, port, timeout, password, POOLCONFIG);
    }
    public RedisCacheStorager(String host, int port, int timeout, String password, JedisPoolConfig poolConfig) {
        this(new JedisPool(poolConfig, host, port, timeout, password));
    }

    public RedisCacheStorager(Pool<Jedis> jedisPool) {
        this.jedisPool = jedisPool;
    }


    @Override
    public T lookup(String key) {

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            byte[] value = jedis.get(key.getBytes(Constant.UTF_8));
            return value != null ? (T) SerializationUtils.deserialize(value) : null;
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }

    @Override
    public void caching(String key, T cache) {

        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            byte[] cacheKey = key.getBytes(Constant.UTF_8);
            byte[] value = SerializationUtils.serialize(cache);
            if (cache.expiredTime() > 0) {
                jedis.setex(cacheKey, (int) (cache.expiredTime() - timeout) / 1000, value);
            } else {
                jedis.set(cacheKey, value);
            }
            jedis.sadd(keys, key);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }

    public T delete(String key) {
        T cache = lookup(key);
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            jedis.del(key);
            jedis.srem(keys, key);
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
        return cache;
    }

    public void clear() {
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            Set<String> cacheKeys = jedis.smembers(keys);
            if (!cacheKeys.isEmpty()) {
                cacheKeys.add(keys);
                jedis.del(cacheKeys.toArray(new String[cacheKeys.size()]));
            }
        } finally {
            if (jedis != null) {
                jedis.close();
            }
        }
    }
}
