package cn.ipokerface.weixin.proxy;

import cn.ipokerface.weixin.Constant;
import cn.ipokerface.weixin.exception.WeixinException;
import cn.ipokerface.weixin.model.WeixinPayAccount;
import cn.ipokerface.weixin.model.payment.TradeType;
import cn.ipokerface.weixin.proxy.merchant.*;
import cn.ipokerface.weixin.proxy.payment.*;
import cn.ipokerface.weixin.proxy.refund.RefundAccountType;
import cn.ipokerface.weixin.proxy.refund.RefundRecord;
import cn.ipokerface.weixin.proxy.refund.RefundResult;
import cn.ipokerface.weixin.request.ApiXmlResult;
import cn.ipokerface.weixin.request.WeixinApis;
import cn.ipokerface.weixin.request.WeixinResponse;
import cn.ipokerface.weixin.sign.SignType;
import cn.ipokerface.weixin.utils.*;
import cn.ipokerface.weixin.xml.ListSuffixResultDeserializer;
import cn.ipokerface.weixin.xml.XmlFormatter;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;

import java.io.*;
import java.net.URLEncoder;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by       PokerFace
 * Create Date      2019-12-28.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class PaymentProxy extends MerchantProxy {



    public PaymentProxy(WeixinPayAccount weixinAccount) {
        super(weixinAccount);
    }



    /**
     * 统一下单接口
     * 除被扫支付场景以外，商户系统先调用该接口在微信支付服务后台生成预支付交易单，返回正确的预支付交易回话标识后再按扫码、JSAPI
     * 、APP等不同场景生成交易串调起支付。
     *
     * @param payPackage
     *            包含订单信息的对象
     * @see cn.ipokerface.weixin.proxy.payment.MerchantPaymentPackage
     * @see PrePayment
     * @see <a href=
     *      "http://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_1">统一下单接口
     *      </a>
     * @return 预支付对象
     */
    public PrePayment createPrePay(MerchantPaymentPackage payPackage) throws WeixinException {
        super.declareMerchant(payPackage);
        payPackage.setSign(weixinSignature.sign(payPackage));
        String payJsRequestXml = XmlFormatter.toXML(payPackage);
        WeixinResponse response = weixinRequestClient.post(
                WeixinApis.order_create_uri, payJsRequestXml);
        return response.getAsObject(new TypeReference<PrePayment>() {
        });
    }




    /**
     * 创建支付请求对象
     *
     * @param payPackage
     *            支付详情
     * @return 支付请求对象
     * @see cn.ipokerface.weixin.proxy.payment.JsapiPaymentRequest JS支付
     * @see cn.ipokerface.weixin.proxy.payment.NativePaymentRequest 扫码支付
     * @see cn.ipokerface.weixin.proxy.payment.MicroPaymentRequest 刷卡支付
     * @see cn.ipokerface.weixin.proxy.payment.AppPaymentRequest APP支付
     * @see cn.ipokerface.weixin.proxy.payment.WapPaymentRequest WAP支付
     * @throws WeixinException
     */
    public MerchantPaymentRequest createPayRequest(MerchantPaymentPackage payPackage)
            throws WeixinException {
        if (StringUtil.isBlank(payPackage.getTradeType())) {
            throw new WeixinException("tradeType not be empty");
        }
        String tradeType = payPackage.getTradeType().toUpperCase();
        if (TradeType.MICROPAY.name().equals(tradeType)) {
            MerchantPaymentPackage _payPackage = new MerchantPaymentPackage(payPackage.getBody(),
                    payPackage.getDetail(), payPackage.getOutTradeNo(),
                    DateUtils.formatFee2Yuan(payPackage.getTotalFee()), null,
                    null, payPackage.getCreateIp(), null, null,
                    payPackage.getAuthCode(), null, payPackage.getAttach(),
                    null, null, payPackage.getGoodsTag(),
                    payPackage.getLimitPay(), payPackage.getSubAppId());
            super.declareMerchant(_payPackage);
            _payPackage.setSign(weixinSignature.sign(_payPackage));
            String para = XmlFormatter.toXML(_payPackage);
            WeixinResponse response = weixinRequestClient.post(
                    WeixinApis.micro_pay_uri, para);
            MicroPaymentRequest microPayRequest = response
                    .getAsObject(new TypeReference<MicroPaymentRequest>() {
                    });
            microPayRequest.setPaymentAccount(weixinPayAccount);
            return microPayRequest;
        }
        PrePayment prePay = createPrePay(payPackage);
        if (TradeType.APP.name().equals(tradeType)) {
            return new AppPaymentRequest(prePay.getPrepayId(), weixinPayAccount);
        } else if (TradeType.JSAPI.name().equals(tradeType)) {
            return new JsapiPaymentRequest(prePay.getPrepayId(), weixinPayAccount);
        } else if (TradeType.NATIVE.name().equals(tradeType)) {
            return new NativePaymentRequest(prePay.getPrepayId(),
                    prePay.getPayUrl(), weixinPayAccount);
        } else if (TradeType.MWEB.name().equals(tradeType)) {
            return new WapPaymentRequest(prePay.getPrepayId(), prePay.getPayUrl(),
                    weixinPayAccount);
        } else {
            throw new WeixinException("unknown tradeType:" + tradeType);
        }
    }



    /**
     * 创建JSAPI支付请求对象
     *
     * @param openId
     *            用户ID
     * @param body
     *            订单描述
     * @param outTradeNo
     *            订单号
     * @param totalFee
     *            订单总额(元)
     * @param notifyUrl
     *            支付通知地址
     * @param createIp
     *            ip地址
     * @param attach
     *            附加数据 非必填
     * @see JsapiPaymentRequest
     * @return JSAPI支付对象
     * @throws WeixinException
     */
    public MerchantPaymentRequest createJSPayRequest(String openId, String body,
                                            String outTradeNo, double totalFee, String notifyUrl,
                                            String createIp, String attach) throws WeixinException {
        MerchantPaymentPackage payPackage = new MerchantPaymentPackage(body, outTradeNo,
                totalFee, notifyUrl, createIp, TradeType.JSAPI, openId, null,
                null, attach);
        return createPayRequest(payPackage);
    }



    /**
     * <p>
     * 生成编辑地址请求
     * </p>
     *
     * err_msg edit_address:ok获取编辑收货地址成功 edit_address:fail获取编辑收货地址失败
     * userName 收货人姓名 telNumber 收货人电话 addressPostalCode 邮编
     * proviceFirstStageName 国标收货地址第一级地址 addressCitySecondStageName
     * 国标收货地址第二级地址 addressCountiesThirdStageName 国标收货地址第三级地址
     * addressDetailInfo 详细收货地址信息nationalCode 收货地址国家码
     *
     * @param url
     *            当前访问页的URL
     * @param oauthToken
     *            oauth授权时产生的token
     * @see <a href=
     *      "https://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=7_8&index=7">
     *      收货地址共享</a>
     * @return 编辑地址请求JSON串
     */
    public String createAddressRequestJSON(String url, String oauthToken) {
        Map<String, String> map = new HashMap<String, String>();
        map.put("appId", weixinPayAccount.getId());
        map.put("timeStamp", DateUtils.timestamp2string());
        map.put("nonceStr", RandomUtil.generateString(16));
        map.put("url", url);
        map.put("accessToken", oauthToken);
        String sign = DigestUtils.SHA1(MapUtils.toJoinString(map, false, true));
        map.remove("url");
        map.remove("accessToken");
        map.put("scope", "jsapi_address");
        map.put("signType", SignType.SHA1.name().toLowerCase());
        map.put("addrSign", sign);
        return JSON.toJSONString(map);
    }


    /**
     * 创建Native支付(扫码支付)链接【模式一】
     *
     * @param productId
     *            与订单ID等价
     * @return 支付链接
     * @see <a href=
     *      "https://pay.weixin.qq.com/wiki/doc/api/native.php?chapter=6_1">扫码支付
     *      </a>
     * @see <a href=
     *      "https://pay.weixin.qq.com/wiki/doc/api/native.php?chapter=6_4">模式一
     *      </a>
     */
    public String createNativePayRequest(String productId) {
        Map<String, String> map = new HashMap<String, String>();
        String timestamp = DateUtils.timestamp2string();
        String noncestr = RandomUtil.generateString(16);
        map.put("appid", weixinPayAccount.getId());
        map.put("mch_id", weixinPayAccount.getMchId());
        map.put("time_stamp", timestamp);
        map.put("nonce_str", noncestr);
        map.put("product_id", productId);
        String sign = weixinSignature.sign(map);
        return String.format(WeixinApis.native_pay_uri, sign,
                weixinPayAccount.getId(), weixinPayAccount.getMchId(), productId,
                timestamp, noncestr);
    }



    /**
     * 创建Native支付(扫码支付)回调对象【模式一】
     *
     * @param productId
     *            商品ID
     * @param body
     *            商品描述
     * @param outTradeNo
     *            商户内部唯一订单号
     * @param totalFee
     *            商品总额 单位元
     * @param notifyUrl
     *            支付回调URL
     * @param createIp
     *            订单生成的机器 IP
     * @param attach
     *            附加数据 非必填
     * @return Native回调对象
     * @see NativePaymentResponse
     * @see <a href=
     *      "https://pay.weixin.qq.com/wiki/doc/api/native.php?chapter=6_1">扫码支付
     *      </a>
     * @see <a href=
     *      "https://pay.weixin.qq.com/wiki/doc/api/native.php?chapter=6_4">模式一
     *      </a>
     * @throws WeixinException
     */
    public NativePaymentResponse createNativePayResponse(String productId,
                                                     String body, String outTradeNo, double totalFee, String notifyUrl,
                                                     String createIp, String attach) throws WeixinException {
        MerchantPaymentPackage payPackage = new MerchantPaymentPackage(body, outTradeNo,
                totalFee, notifyUrl, createIp, TradeType.NATIVE, null, null,
                productId, attach);
        PrePayment prePay = createPrePay(payPackage);
        return new NativePaymentResponse(weixinPayAccount, prePay.getPrepayId());
    }


    /**
     * 创建Native支付(扫码支付)链接【模式二】
     *
     * @param productId
     *            商品ID
     * @param body
     *            商品描述
     * @param outTradeNo
     *            商户内部唯一订单号
     * @param totalFee
     *            商品总额 单位元
     * @param notifyUrl
     *            支付回调URL
     * @param createIp
     *            订单生成的机器 IP
     * @param attach
     *            附加数据 非必填
     * @return Native支付对象
     * @see NativePaymentRequest
     * @see <a href=
     *      "https://pay.weixin.qq.com/wiki/doc/api/native.php?chapter=6_1">扫码支付
     *      </a>
     * @see <a href=
     *      "https://pay.weixin.qq.com/wiki/doc/api/native.php?chapter=6_5">模式二
     *      </a>
     * @throws WeixinException
     */
    public MerchantPaymentRequest createNativePayRequest(String productId, String body,
                                                String outTradeNo, double totalFee, String notifyUrl,
                                                String createIp, String attach) throws WeixinException {
        MerchantPaymentPackage payPackage = new MerchantPaymentPackage(body, outTradeNo,
                totalFee, notifyUrl, createIp, TradeType.NATIVE, null, null,
                productId, attach);
        return createPayRequest(payPackage);
    }


    /**
     * 创建APP支付请求对象
     *
     * @param body
     *            商品描述
     * @param outTradeNo
     *            商户内部唯一订单号
     * @param totalFee
     *            商品总额 单位元
     * @param notifyUrl
     *            支付回调URL
     * @param createIp
     *            订单生成的机器 IP
     * @param attach
     *            附加数据 非必填
     * @param store
     *            门店信息 非必填
     * @return APP支付对象
     * @see SceneInfoStore
     * @see AppPaymentRequest
     * @see <a href=
     *      "https://pay.weixin.qq.com/wiki/doc/api/app/app.php?chapter=8_1">
     *      APP支付</a>
     * @throws WeixinException
     */
    public MerchantPaymentRequest createAppPayRequest(String body, String outTradeNo,
                                             double totalFee, String notifyUrl, String createIp, String attach,
                                             SceneInfoStore store) throws WeixinException {
        MerchantPaymentPackage payPackage = new MerchantPaymentPackage(body, outTradeNo,
                totalFee, notifyUrl, createIp, TradeType.APP, null, null, null,
                attach);
        if (store != null) {
            payPackage.setSceneInfo(String.format(
                    "{\"store_id\": \"%s\", \"store_name\":\"%s\"}",
                    store.getId(), store.getName()));
        }
        return createPayRequest(payPackage);
    }



    /**
     * 创建WAP支付请求对象：正常流程用户支付完成后会返回至发起支付的页面，如需返回至指定页面，
     * 则可以在MWEB_URL后拼接上redirect_url参数，来指定回调页面
     *
     * @param body
     *            商品描述
     * @param outTradeNo
     *            商户内部唯一订单号
     * @param totalFee
     *            商品总额 单位元
     * @param notifyUrl
     *            支付回调URL
     * @param createIp
     *            订单生成的机器 IP
     * @param attach
     *            附加数据 非必填
     * @param app
     *            应用信息
     * @return WAP支付对象
     * @see SceneInfoApp
     * @see WapPaymentRequest
     * @see <a href=
     *      "https://pay.weixin.qq.com/wiki/doc/api/wap.php?chapter=15_1">WAP支付
     *      </a>
     * @throws WeixinException
     */
    public MerchantPaymentRequest createWapPayRequest(String body, String outTradeNo,
                                             double totalFee, String notifyUrl, String createIp, String attach,
                                             SceneInfoApp app) throws WeixinException {
        MerchantPaymentPackage payPackage = new MerchantPaymentPackage(body, outTradeNo,
                totalFee, notifyUrl, createIp, TradeType.MWEB, null, null,
                null, attach);
        if (app != null) {
            payPackage.setSceneInfo(String.format("{\"h5_info\":\"%s\"}",
                    app.getSceneInfo()));
        }
        return createPayRequest(payPackage);
    }



    /**
     * 提交被扫支付
     *
     * @param authCode
     *            扫码支付授权码 ,设备读取用户微信中的条码或者二维码信息
     * @param body
     *            商品描述
     * @param outTradeNo
     *            商户内部唯一订单号
     * @param totalFee
     *            商品总额 单位元
     * @param createIp
     *            订单生成的机器 IP
     * @param attach
     *            附加数据 非必填
     * @param store
     *            门店信息 非必填
     * @return 支付的订单信息
     * @see MicroPaymentRequest
     * @see cn.ipokerface.weixin.proxy.merchant.Order
     * @see SceneInfoStore
     * @see <a href=
     *      "http://pay.weixin.qq.com/wiki/doc/api/micropay.php?chapter=9_10">
     *      提交被扫支付API</a>
     * @throws WeixinException
     */
    public MerchantPaymentRequest createMicroPayRequest(String authCode, String body,
                                               String outTradeNo, double totalFee, String createIp, String attach,
                                               SceneInfoStore store) throws WeixinException {
        MerchantPaymentPackage payPackage = new MerchantPaymentPackage(body, outTradeNo,
                totalFee, null, createIp, TradeType.MICROPAY, null, authCode,
                null, attach);
        if (store != null) {
            payPackage.setSceneInfo(String.format("{\"store_info\":\"%s\"}",
                    JSON.toJSONString(store)));
        }
        return createPayRequest(payPackage);
    }



    /**
     * 订单查询
     * <p>
     * 当商户后台、网络、服务器等出现异常，商户系统最终未接收到支付通知； 调用支付接口后，返回系统错误或未知交易状态情况；
     * 调用被扫支付API，返回USERPAYING的状态； 调用关单或撤销接口API之前，需确认支付状态；
     * </P>
     *
     * @param idQuery
     *            商户系统内部的订单号, transaction_id、out_trade_no 二 选一,如果同时存在优先级:
     *            transaction_id &gt; out_trade_no
     * @return 订单信息
     * @see Order
     * @see <a href=
     *      "http://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_2">
     *      订单查询API</a>
     * @since V3
     * @throws WeixinException
     */
    public Order queryOrder(IdQuery idQuery) throws WeixinException {
        Map<String, String> map = createBaseRequestMap(idQuery);
        map.put("sign", weixinSignature.sign(map));
        String param = XmlFormatter.map2xml(map);
        WeixinResponse response = weixinRequestClient.post(
                WeixinApis.order_query_uri, param);
        return ListSuffixResultDeserializer.deserialize(response.getAsString(),
                Order.class);
    }


    /**
     * 申请退款(请求需要双向证书)
     *
     * <p>
     * 当交易发生之后一段时间内，由于买家或者卖家的原因需要退款时，卖家可以通过退款接口将支付款退还给买家，微信支付将在收到退款请求并且验证成功之后，
     * 按照退款规则将支付款按原路退到买家帐号上。
     * </p>
     *
     * <ol>
     *   <li>交易时间超过一年的订单无法提交退款；</li>
     *   <li>
     *     微信支付退款支持单笔交易分多次退款，多次退款需要提交原支付订单的商户订单号和设置不同的退款单号。
     *     申请退款总金额不能超过订单金额。
     *     <span style="color:red">一笔退款失败后重新提交，请不要更换退款单号，请使用原商户退款单号。</span>
     *   </li>
     *   <li>
     *     请求频率限制：150qps，即每秒钟正常的申请退款请求次数不超过150次。
     *     错误或无效请求频率限制：6qps，即每秒钟异常或错误的退款申请请求不超过6次。
     *   </li>
     *   <li>每个支付订单的部分退款次数不能超过50次。</li>
     * </ol>
     *
     * @param idQuery
     *            商户系统内部的订单号, transaction_id 、 out_trade_no 二选一,如果同时存在优先级:
     *            transaction_id &gt; out_trade_no
     * @param outRefundNo
     *            商户系统内部的退款单号,商 户系统内部唯一,同一退款单号多次请求只退一笔
     * @param totalFee
     *            订单总金额,单位为元
     * @param refundFee
     *            退款总金额,单位为元,可以做部分退款
     * @param refundFeeType
     *            货币类型，符合ISO 4217标准的三位字母代码，默认人民币：CNY
     * @param opUserId
     *            操作员帐号, 默认为商户号
     * @param refundDesc
     *            退款原因，若商户传入，会在下发给用户的退款消息中体现退款原因
     * @param refundAccountType
     *            退款资金来源,默认使用未结算资金退款：REFUND_SOURCE_UNSETTLED_FUNDS
     * @return 退款申请结果
     * @see RefundResult
     * @see <a href=
     *      "http://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_4">
     *      申请退款API</a>
     * @since V3
     * @throws WeixinException
     */
    public RefundResult applyRefund(IdQuery idQuery, String outRefundNo,
                                    double totalFee, double refundFee, CurrencyType refundFeeType,
                                    String opUserId, String refundDesc,
                                    RefundAccountType refundAccountType) throws WeixinException {
        Map<String, String> map = createBaseRequestMap(idQuery);
        map.put("out_refund_no", outRefundNo);
        map.put("total_fee",
                Integer.toString(DateUtils.formatYuan2Fen(totalFee)));
        map.put("refund_fee",
                Integer.toString(DateUtils.formatYuan2Fen(refundFee)));
        if (StringUtil.isBlank(opUserId)) {
            opUserId = weixinPayAccount.getMchId();
        }
        map.put("op_user_id", opUserId);
        if (refundFeeType == null) {
            refundFeeType = CurrencyType.CNY;
        }
        if (refundAccountType == null) {
            refundAccountType = RefundAccountType.REFUND_SOURCE_UNSETTLED_FUNDS;
        }
        if (StringUtil.isNotBlank(refundDesc)) {
            map.put("refund_desc", refundDesc);
        }
        map.put("refund_fee_type", refundFeeType.name());
        map.put("refund_account", refundAccountType.name());
        map.put("sign", weixinSignature.sign(map));
        String param = XmlFormatter.map2xml(map);
        WeixinResponse response = getWeixinSSLExecutor().post(
                WeixinApis.refund_apply_uri, param);
        return response.getAsObject(new TypeReference<RefundResult>() {
        });
    }


    /**
     * 退款申请(全额退款)
     *
     * @param idQuery
     *            商户系统内部的订单号, transaction_id 、 out_trade_no 二选一,如果同时存在优先级:
     *            transaction_id &gt; out_trade_no
     * @param outRefundNo
     *            商户系统内部的退款单号,商 户系统内部唯一,同一退款单号多次请求只退一笔
     * @param totalFee
     *            订单总金额,单位为元
     * @see #applyRefund(IdQuery, String, double, double, CurrencyType, String, String, RefundAccountType)
     */
    public RefundResult applyRefund(IdQuery idQuery, String outRefundNo,
                                    double totalFee) throws WeixinException {
        return applyRefund(idQuery, outRefundNo, totalFee, totalFee, null,
                null, null, null);
    }


    /**
     * 冲正订单(需要证书)当支付返回失败,或收银系统超时需要取消交易,可以调用该接口 接口逻辑:支
     * 付失败的关单,支付成功的撤销支付<font color="red">7天以内的单可撤销,其他正常支付的单
     * 如需实现相同功能请调用退款接口</font> <font
     * color="red">调用扣款接口后请勿立即调用撤销,需要等待5秒以上。先调用查单接口,如果没有确切的返回,再调用撤销</font>
     *
     * @param idQuery
     *            商户系统内部的订单号, transaction_id 、 out_trade_no 二选一,如果同时存在优先级:
     *            transaction_id &gt; out_trade_no
     * @return 撤销结果
     * @since V3
     * @throws WeixinException
     */
    public MerchantResult reverseOrder(IdQuery idQuery) throws WeixinException {
        Map<String, String> map = createBaseRequestMap(idQuery);
        map.put("sign", weixinSignature.sign(map));
        String param = XmlFormatter.map2xml(map);
        WeixinResponse response = getWeixinSSLExecutor().post(
                WeixinApis.order_reverse_uri, param);
        return response.getAsObject(new TypeReference<MerchantResult>() {
        });
    }


    /**
     * native支付URL转短链接：用于扫码原生支付模式一中的二维码链接转成短链接(weixin://wxpay/s/XXXXXX)，减小二维码数据量
     * ，提升扫描速度和精确度。
     *
     * @param url
     *            具有native标识的支付URL
     * @return 转换后的短链接
     * @throws WeixinException
     * @see <a href=
     *      "http://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_9">
     *      转换短链接API</a>
     */
    public String getShorturl(String url) throws WeixinException {
        Map<String, String> map = createBaseRequestMap(null);
        try {
            map.put("long_url", URLEncoder.encode(url, Constant.UTF_8.name()));
        } catch (UnsupportedEncodingException e) {
            ;
        }
        map.put("sign", weixinSignature.sign(map));
        String param = XmlFormatter.map2xml(map);
        WeixinResponse response = weixinRequestClient.post(
                WeixinApis.longurl_convert_uri, param);
        map = XmlFormatter.xml2map(response.getAsString());
        return map.get("short_url");
    }


    /**
     * 关闭订单
     * <p>
     * 商户订单支付失败需要生成新单号重新发起支付，要对原订单号调用关单，避免重复支付；系统下单后，用户支付超时，系统退出不再受理，避免用户继续
     * ，请调用关单接口,如果关单失败,返回已完 成支付请按正常支付处理。如果出现银行掉单,调用关单成功后,微信后台会主动发起退款。
     * </p>
     *
     * @param outTradeNo
     *            商户系统内部的订单号
     * @return 处理结果
     * @since V3
     * @throws WeixinException
     * @see <a href=
     *      "http://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_3">
     *      关闭订单API</a>
     */
    public MerchantResult closeOrder(String outTradeNo) throws WeixinException {
        Map<String, String> map = createBaseRequestMap(new IdQuery(outTradeNo,
                IdType.TRADENO));
        map.put("sign", weixinSignature.sign(map));
        String param = XmlFormatter.map2xml(map);
        WeixinResponse response = weixinRequestClient.post(
                WeixinApis.order_close_uri, param);
        return response.getAsObject(new TypeReference<MerchantResult>() {
        });
    }



    /**
     * 下载对账单<br>
     * 1.微信侧未成功下单的交易不会出现在对账单中。支付成功后撤销的交易会出现在对账 单中,跟原支付单订单号一致,bill_type 为
     * REVOKED;<br>
     * 2.微信在次日 9 点启动生成前一天的对账单,建议商户 9 点半后再获取;<br>
     * 3.对账单中涉及金额的字段单位为“元”。<br>
     *
     * @param billDate
     *            下载对账单的日期
     * @param billType
     *            下载对账单的类型 ALL,返回当日所有订单信息, 默认值 SUCCESS,返回当日成功支付的订单
     *            REFUND,返回当日退款订单
     * @param outputStream
     *            输出流
     * @param tarType
     *            非必传参数，固定值：GZIP，返回格式为.gzip的压缩包账单。不传则默认为数据流形式。
     * @since V3
     * @see <a href=
     *      "http://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_6">
     *      下载对账单API</a>
     * @throws WeixinException
     */
    public void downloadBill(Date billDate, BillType billType,
                             OutputStream outputStream, ZipType tarType) throws WeixinException {
        if (billDate == null) {
            Calendar now = Calendar.getInstance();
            now.add(Calendar.DAY_OF_MONTH, -1);
            billDate = now.getTime();
        }
        if (billType == null) {
            billType = BillType.ALL;
        }
        String formatBillDate = DateUtils.fortmat2yyyyMMdd(billDate);
        Map<String, String> map = createBaseRequestMap(null);
        map.put("bill_date", formatBillDate);
        map.put("bill_type", billType.name());
        if (tarType != null) {
            map.put("tar_type", tarType.name());
        }
        map.put("sign", weixinSignature.sign(map));
        String param = XmlFormatter.map2xml(map);
        WeixinResponse response = weixinRequestClient.post(
                WeixinApis.download_bill_uri, param);

        if (ZipType.GZIP == tarType) {
            try {
                IOUtils.copy(response.getBody(), outputStream);
            } catch (IOException e) {
                ;
            }
        } else {
            BufferedReader reader = null;
            BufferedWriter writer = null;
            try {
                writer = new BufferedWriter(new OutputStreamWriter(
                        outputStream, Constant.UTF_8));
                reader = new BufferedReader(new InputStreamReader(
                        response.getBody(), Constant.UTF_8));
                String line = null;
                while ((line = reader.readLine()) != null) {
                    writer.write(line);
                    writer.newLine();
                }
            } catch (IOException e) {
                throw new WeixinException(e);
            } finally {
                try {
                    if (reader != null) {
                        reader.close();
                    }
                    if (writer != null) {
                        writer.close();
                    }
                } catch (IOException ignore) {
                    ;
                }
            }
        }
    }


    /**
     * 退款查询
     *
     * <p>
     * 提交退款申请后，通过调用该接口查询退款状态。退款有一定延时，用零钱支付的退款20分钟内到账，银行卡支付的退款3个工作日后重新查询退款状态。
     * </p>
     *
     * @param idQuery
     *            单号 refund_id、out_refund_no、 out_trade_no 、 transaction_id
     *            四个参数必填一个,优先级为:
     *            refund_id &gt; out_refund_no &gt; transaction_id &gt; out_trade_no
     * @return 退款记录
     * @see RefundRecord
     * @see cn.ipokerface.weixin.proxy.refund.RefundDetail
     * @see <a href=
     *      "http://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_5">
     *      退款查询API</a>
     * @since V3
     * @throws WeixinException
     */
    public RefundRecord queryRefund(IdQuery idQuery) throws WeixinException {
        Map<String, String> map = createBaseRequestMap(idQuery);
        map.put("sign", weixinSignature.sign(map));
        String param = XmlFormatter.map2xml(map);
        WeixinResponse response = weixinRequestClient.post(
                WeixinApis.refund_query_uri, param);
        return ListSuffixResultDeserializer.deserialize(response.getAsString(),
                RefundRecord.class);
    }


    /**
     * 接口上报
     *
     * @param interfaceUrl
     *            上报对应的接口的完整 URL, 类似: https://api.mch.weixin.q
     *            q.com/pay/unifiedorder
     * @param executeTime
     *            接口耗时情况,单位为毫秒
     * @param outTradeNo
     *            商户系统内部的订单号,商 户可以在上报时提供相关商户订单号方便微信支付更好 的提高服务质量。
     * @param ip
     *            发起接口调用时的机器 IP
     * @param time
     *            ￼商户调用该接口时商户自己 系统的时间
     * @param returnXml
     *            调用接口返回的基本数据
     * @return 处理结果
     * @throws WeixinException
     * @see <a href=
     *      "http://pay.weixin.qq.com/wiki/doc/api/jsapi.php?chapter=9_8">
     *      交易保障</a>
     */
    @SuppressWarnings("unchecked")
    public ApiXmlResult reportInterface(String interfaceUrl, int executeTime,
                                        String outTradeNo, String ip, Date time, ApiXmlResult returnXml)
            throws WeixinException {
        Map<String, String> map = createBaseRequestMap(null);
        map.put("interface_url", interfaceUrl);
        map.put("execute_time_", Integer.toString(executeTime));
        map.put("out_trade_no", outTradeNo);
        map.put("user_ip", ip);
        map.put("time", DateUtils.fortmat2yyyyMMddHHmmss(time));
        map.putAll((Map<String, String>) JSON.toJSON(returnXml));
        map.put("sign", weixinSignature.sign(map));
        String param = XmlFormatter.map2xml(map);
        WeixinResponse response = weixinRequestClient.post(
                WeixinApis.interface_report_uri, param);
        return response.getAsXml();
    }

    /**
     * 授权码查询OPENID接口
     *
     * @param authCode
     *            扫码支付授权码，设备读取用户微信中的条码或者二维码信息
     * @return 查询结果
     * @see OpenidResult
     * @see <a href=
     *      "https://pay.weixin.qq.com/wiki/doc/api/micropay.php?chapter=9_13&index=9">
     *      授权码查询OPENID</a>
     * @throws WeixinException
     */
    public OpenidResult authCode2openId(String authCode) throws WeixinException {
        Map<String, String> map = createBaseRequestMap(null);
        map.put("auth_code", authCode);
        map.put("sign", weixinSignature.sign(map));
        String param = XmlFormatter.map2xml(map);
        WeixinResponse response = weixinRequestClient.post(
                WeixinApis.authcode_openid_uri, param);
        return response.getAsObject(new TypeReference<OpenidResult>() {
        });
    }
}
