package cn.ipokerface.weixin.proxy;

import cn.ipokerface.weixin.exception.WeixinException;
import cn.ipokerface.weixin.model.WeixinPayAccount;
import cn.ipokerface.weixin.proxy.merchant.IdQuery;
import cn.ipokerface.weixin.proxy.merchant.MerchantResult;
import cn.ipokerface.weixin.request.WeixinRequestClient;
import cn.ipokerface.weixin.sign.WeixinPaymentSignature;
import cn.ipokerface.weixin.sign.WeixinSignature;
import cn.ipokerface.weixin.utils.RandomUtil;
import cn.ipokerface.weixin.utils.StringUtil;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by       PokerFace
 * Create Date      2019-12-27.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class MerchantProxy extends AbstractProxy {


    protected final WeixinPayAccount weixinPayAccount;
    protected final WeixinSignature weixinSignature;


    private volatile WeixinRequestClient weixinSslRequestClient;


    public MerchantProxy(WeixinPayAccount payAccount) {
        this.weixinPayAccount = payAccount;
        this.weixinSignature = new WeixinPaymentSignature(weixinPayAccount.getPaySignKey());
    }



    /**
     * 获取微信签名工具
     *
     * @return
     */
    public WeixinSignature getWeixinSignature() {
        return this.weixinSignature;
    }





    /**
     * 支付接口请求基本数据
     *
     * @param idQuery
     *            ID信息 可为空
     * @return 基础map
     */
    protected Map<String, String> createBaseRequestMap(IdQuery idQuery) {
        Map<String, String> map = new HashMap<String, String>();
        map.put("appid", weixinPayAccount.getId());
        map.put("mch_id", weixinPayAccount.getMchId());
        map.put("nonce_str", RandomUtil.generateString(16));
        if (StringUtil.isNotBlank(weixinPayAccount.getDeviceInfo())) {
            map.put("device_info", weixinPayAccount.getDeviceInfo());
        }
        if (StringUtil.isNotBlank(weixinPayAccount.getSubId())) {
            map.put("sub_appid", weixinPayAccount.getSubId());
        }
        if (StringUtil.isNotBlank(weixinPayAccount.getSubMchId())) {
            map.put("sub_mch_id", weixinPayAccount.getSubMchId());
        }
        if (idQuery != null) {
            map.put(idQuery.getType().getName(), idQuery.getId());
        }
        return map;
    }


    /**
     * 微信SSL
     *
     * @return
     */
    protected WeixinRequestClient getWeixinSSLExecutor() throws WeixinException {
        if (weixinSslRequestClient == null) {
            try {
                InputStream is = null;
                File certificate = new File(weixinPayAccount.getCertificateFile());
                if (!certificate.exists() || !certificate.isFile()) {
                    is = Thread.currentThread().getContextClassLoader().getResourceAsStream(certificate.getName());
                } else {
                    is = new FileInputStream(certificate);
                }
                if (is == null) {
                    throw new WeixinException("Invalid certificate file : " + certificate.toString());
                }
                this.weixinSslRequestClient = weixinRequestClient.createSSLRequestExecutor(weixinPayAccount.getCertificateKey(), is);
            } catch (IOException e) {
                throw new WeixinException("IO Error on createSSLRequestExecutor", e);
            }
        }
        return this.weixinSslRequestClient;
    }


    /**
     * 设置商户信息
     *
     * @param merchant
     */
    protected <T extends MerchantResult> void declareMerchant(T merchant) {
        merchant.setAppId(weixinPayAccount.getId());
        merchant.setMchId(weixinPayAccount.getMchId());
        merchant.setDeviceInfo(weixinPayAccount.getDeviceInfo());
        merchant.setSubAppId(weixinPayAccount.getSubId());
        merchant.setSubMchId(weixinPayAccount.getSubMchId());
        merchant.setNonceStr(RandomUtil.generateString(16));
    }
}
