package cn.ipokerface.weixin.proxy;

import cn.ipokerface.weixin.exception.WeixinException;
import cn.ipokerface.weixin.model.WeixinPayAccount;
import cn.ipokerface.weixin.proxy.coupon.CouponDetail;
import cn.ipokerface.weixin.proxy.coupon.CouponResult;
import cn.ipokerface.weixin.proxy.coupon.CouponStock;
import cn.ipokerface.weixin.request.WeixinApis;
import cn.ipokerface.weixin.request.WeixinResponse;
import cn.ipokerface.weixin.utils.StringUtil;
import cn.ipokerface.weixin.xml.XmlFormatter;
import com.alibaba.fastjson.TypeReference;

import java.util.Map;

/**
 * Created by       PokerFace
 * Create Date      2019-12-28.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class CouponProxy extends MerchantProxy {

    public CouponProxy(WeixinPayAccount weixinAccount) {
        super(weixinAccount);
    }


    /**
     * 发放代金券(需要证书)
     *
     * @param couponStockId
     *            代金券批次id
     * @param partnerTradeNo
     *            商户发放凭据号（格式：商户id+日期+流水号），商户侧需保持唯一性
     * @param openId
     *            用户的openid
     * @param opUserId
     *            操作员帐号, 默认为商户号 可在商户平台配置操作员对应的api权限 可为空
     * @return 发放结果
     * @see CouponResult
     * @see <a href=
     *      "https://pay.weixin.qq.com/wiki/doc/api/tools/sp_coupon.php?chapter=12_3">发放代金券接口</a>
     * @throws WeixinException
     */
    public CouponResult sendCoupon(String couponStockId, String partnerTradeNo,
                                   String openId, String opUserId) throws WeixinException {
        Map<String, String> map = createBaseRequestMap(null);
        map.put("coupon_stock_id", couponStockId);
        map.put("partner_trade_no", partnerTradeNo);
        map.put("openid", openId);
        // openid记录数（目前支持num=1）
        map.put("openid_count", "1");
        // 操作员帐号, 默认为商户号 可在商户平台配置操作员对应的api权限
        if (StringUtil.isBlank(opUserId)) {
            opUserId = weixinPayAccount.getMchId();
        }
        map.put("op_user_id", opUserId);
        map.put("version", "1.0");
        map.put("type", "XML");
        map.put("sign", weixinSignature.sign(map));
        String param = XmlFormatter.map2xml(map);
        WeixinResponse response = getWeixinSSLExecutor().post(
                WeixinApis.coupon_send_uri, param);
        return response.getAsObject(new TypeReference<CouponResult>() {
        });
    }



    /**
     * 查询代金券批次
     *
     * @param couponStockId
     *            代金券批次ID
     * @return 代金券批次信息
     * @see CouponStock
     * @see <a href=
     *      "https://pay.weixin.qq.com/wiki/doc/api/tools/sp_coupon.php?chapter=12_4">查询代金券批次信息接口</a>
     * @throws WeixinException
     */
    public CouponStock queryCouponStock(String couponStockId)
            throws WeixinException {
        Map<String, String> map = createBaseRequestMap(null);
        map.put("coupon_stock_id", couponStockId);
        map.put("sign", weixinSignature.sign(map));
        String param = XmlFormatter.map2xml(map);
        WeixinResponse response = weixinRequestClient.post(
                WeixinApis.coupon_stock_query_uri, param);
        return response.getAsObject(new TypeReference<CouponStock>() {
        });
    }


    /**
     * 查询代金券详细
     *
     * @param openId
     *            用户ID
     * @param couponId
     *            代金券ID
     * @param stockId
     *            代金劵对应的批次号
     * @return 代金券详细信息
     * @see CouponDetail
     * @see <a href=
     *      "https://pay.weixin.qq.com/wiki/doc/api/tools/sp_coupon.php?chapter=12_5">查询代金券详细信息接口</a>
     * @throws WeixinException
     */
    public CouponDetail queryCouponDetail(String openId, String couponId,
                                          String stockId) throws WeixinException {
        Map<String, String> map = createBaseRequestMap(null);
        map.put("openid", openId);
        map.put("coupon_id", couponId);
        map.put("stock_id", stockId);
        map.put("sign", weixinSignature.sign(map));
        String param = XmlFormatter.map2xml(map);
        WeixinResponse response = weixinRequestClient.post(
                WeixinApis.coupon_detail_query_uri, param);
        return response.getAsObject(new TypeReference<CouponDetail>() {
        });
    }

}
