package cn.ipokerface.web.spring;


import cn.ipokerface.common.fastjson.Fastjson;
import cn.ipokerface.common.utils.LocalTimeUtils;
import cn.ipokerface.web.spring.converter.LocalDateTimeConverter;
import cn.ipokerface.web.spring.bind.ServletModelAttributeMethodProcessor;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.alibaba.fastjson.support.config.FastJsonConfig;
import com.alibaba.fastjson.support.spring.FastJsonHttpMessageConverter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.http.HttpMessageConverters;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.format.FormatterRegistry;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurationSupport;

import java.util.List;

/**
 * Created by       PokerFace
 * Create Date      2019-10-22.
 * Email:
 * <p>
 * Description:     Use this filter to wrapper request so that request's inputstream could be use any times.
 *                  so you should set config <code>spring.mvc.filter.request-wrapper.enable = true</code>
 *                  more:
 *                  <code>spring.mvc.filter.request-wrapper.filter-path = /path/to/filter</code> default<code>/api/*</code>
 *
 */

@Configuration
@EnableConfigurationProperties({ WebMvcProperties.class})
public class WebAutoConfiguration extends WebMvcConfigurationSupport {

    @Autowired
    WebMvcProperties webMvcProperties;




    @Bean
    @Primary
    public FastJsonHttpMessageConverter fastJsonHttpMessageConverter() {
        FastJsonHttpMessageConverter fastConvert = new FastJsonHttpMessageConverter();
        FastJsonConfig fastJsonConfig = new FastJsonConfig();
        fastJsonConfig.setSerializeConfig(Fastjson.serializeConfig());

        fastJsonConfig.setSerializeFilters(Fastjson.serializeFilters());
        fastJsonConfig.setSerializerFeatures(webMvcProperties.getFastjsonFeatures());
        fastJsonConfig.setDateFormat(LocalTimeUtils.fmt_date_time);
        fastConvert.setFastJsonConfig(fastJsonConfig);
        return fastConvert;
    }


    /**
     *  Auto config fastjson mvc cofiguration
     *  set<code>spring.mvc.fast-json.converters-enable=true</code>
     *  set<code>spring.mvc.fast-json.features=PrettyFormat,QuoteFieldNames,...</code>{@link SerializerFeature}
     *
     *
     * @return {@link HttpMessageConverter}
     */
    @Bean
    @ConditionalOnProperty(value = "spring.web.fastjson-enabled", havingValue = "true", matchIfMissing = true)
    @ConditionalOnMissingBean(HttpMessageConverters.class)
    public HttpMessageConverters fastJsonHttpMessageConverters(FastJsonHttpMessageConverter fastJsonHttpMessageConverter) {
        return new HttpMessageConverters(fastJsonHttpMessageConverter);
    }


    @Override
    protected void addFormatters(FormatterRegistry registry) {
        registry.addConverter(new LocalDateTimeConverter());
    }

    @Override
    public void addArgumentResolvers(List<HandlerMethodArgumentResolver> resolvers) {
        if (webMvcProperties.isUnderlineToCamelcaseEnabled()) {
            resolvers.add(new ServletModelAttributeMethodProcessor(true));
        }
    }
}
