package cn.ipokerface.common.async;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.util.concurrent.*;

/**
 * Created by       PokerFace
 * Create Date      2020-11-08.
 * Email:
 * Version          1.0.0
 * <p>
 * Description:     自动装配
 */


@Configuration
@ConditionalOnProperty(name = "spring.async.enabled", matchIfMissing = false)
@EnableConfigurationProperties({ThreadPoolProperties.class})
public class ThreadPoolAutoConfiguration {


    @Autowired
    ThreadPoolProperties threadPoolProperties;


    @Bean
    public ThreadPoolExecutor threadPoolExecutor(){

        /**
         * 参数keepAliveTime的时间单位，有7种取值，在TimeUnit类中有7种静态属性：
         */
        TimeUnit timeUnit = threadPoolProperties.getTimeUnit();

        /**
         * 一个阻塞队列，用来存储等待执行的任务，这个参数的选择也很重要，会对线程池的运行
         *  过程产生重大影响，一般来说，这里的阻塞队列有以下几种选择：
         *  ArrayBlockingQueue;
         *  LinkedBlockingQueue;
         *  SynchronousQueue;
         *
         */
        BlockingQueue<Runnable> taskQueue = null;
        switch (threadPoolProperties.getQueueType()) {
            case "LinkedBlockingDeque":
                taskQueue = new LinkedBlockingDeque<>();
                break;
            case "ArrayBlockingQueue":
                taskQueue = new ArrayBlockingQueue(threadPoolProperties.getArrayBlockingQueueCapacity());
                break;
            case "SynchronousQueue":
                taskQueue = new SynchronousQueue();
        }

        /**
         *  当任务超过队列长度时，又不能启动线程时，会抛出异常
         *
         */
        RejectedExecutionHandler handler = null;
        switch (threadPoolProperties.getHandlerPolicy()) {
            case "DiscardPolicy":
                handler = new ThreadPoolExecutor.DiscardPolicy();
                break;
            case "AbortPolicy":
                handler = new ThreadPoolExecutor.AbortPolicy();
                break;
            case "CallerRunsPolicy":
                handler = new ThreadPoolExecutor.CallerRunsPolicy();
                break;
            case "DiscardOldestPolicy":
                handler = new ThreadPoolExecutor.DiscardOldestPolicy();
                break;
        }



        ThreadPoolExecutor threadPoolExecutor = new ThreadPoolExecutor(threadPoolProperties.getCorePoolSize(),
                threadPoolProperties.getMaximumPoolSize(),
                threadPoolProperties.getKeepAliveTime(),
                timeUnit,
                taskQueue,
                handler);
        // 预先启动核心线程
        threadPoolExecutor.prestartCoreThread();
        return threadPoolExecutor;

    }

}
