package cn.ipokerface.cloud.config.repository;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.Banner;
import org.springframework.boot.WebApplicationType;
import org.springframework.boot.autoconfigure.context.PropertyPlaceholderAutoConfiguration;
import org.springframework.boot.builder.SpringApplicationBuilder;
import org.springframework.boot.context.config.ConfigFileApplicationListener;
import org.springframework.cloud.config.environment.Environment;
import org.springframework.cloud.config.environment.PropertySource;
import org.springframework.cloud.config.server.environment.EnvironmentRepository;
import org.springframework.cloud.config.server.environment.NativeEnvironmentProperties;
import org.springframework.cloud.config.server.environment.PassthruEnvironmentRepository;
import org.springframework.cloud.config.server.environment.SearchPathLocator;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.Ordered;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MapPropertySource;
import org.springframework.core.env.StandardEnvironment;
import org.springframework.util.StringUtils;

import java.io.File;
import java.util.*;

/**
 * Created by       PokerFace
 * Create Date      2020-06-19.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class LocalEnvironmentRepository
        implements EnvironmentRepository, SearchPathLocator, Ordered {

    private Logger logger = LoggerFactory.getLogger(LocalEnvironmentRepository.class);



    private static final String[] DEFAULT_LOCATIONS = new String[] { "classpath:/",
            "classpath:/config/", "file:./", "file:./config/" };



    private ConfigurableEnvironment environment;

    private int order;

    private String version;

    private String[] searchLocations;






    public LocalEnvironmentRepository(ConfigurableEnvironment environment, LocalEnvironmentProperties properties){

        this.environment = environment;
        this.order = properties.getOrder();
        this.searchLocations = properties.getSearchLocations();
        this.version = properties.getVersion();
    }


    @Override
    public Environment findOne(String config, String profile, String label, boolean includeOrigin) {
        return findOne(config, profile, label);
    }

    @Override
    public Environment findOne(String config, String profile, String label) {
        SpringApplicationBuilder builder = new SpringApplicationBuilder(
                PropertyPlaceholderAutoConfiguration.class);
        ConfigurableEnvironment environment = getEnvironment(profile);
        builder.environment(environment);
        builder.web(WebApplicationType.NONE).bannerMode(Banner.Mode.OFF);
        if (!logger.isDebugEnabled()) {
            // Make the mini-application startup less verbose
            builder.logStartupInfo(false);
        }
        String[] args = getArgs(config, profile, label);
        // Explicitly set the listeners (to exclude logging listener which would change
        // log levels in the caller)
        builder.application()
                .setListeners(Arrays.asList(new ConfigFileApplicationListener()));
        ConfigurableApplicationContext context = builder.run(args);
        environment.getPropertySources().remove("profiles");
        try {
            return clean(new PassthruEnvironmentRepository(environment).findOne(config,
                    profile, label));
        }
        finally {
            context.close();
        }
    }


    /**
     *  启用 label
     *
     * @param application
     * @param profile
     * @param label
     * @return
     */
    private String[] getArgs(String application, String profile, String label) {
        List<String> list = new ArrayList<String>();
        String config = application;
        // 兼容可以获取到以label 开头的配置
        if (!StringUtils.isEmpty(label) && !label.equals(application)){
            config =  label;
        }
        list.add("--spring.config.name=" + config);
        list.add("--spring.cloud.bootstrap.enabled=false");
        list.add("--encrypt.failOnError=false");
        list.add("--spring.config.location=" + StringUtils.arrayToCommaDelimitedString(
                getLocations(application, profile, label).getLocations()));
        return list.toArray(new String[0]);
    }



    private ConfigurableEnvironment getEnvironment(String profile) {
        ConfigurableEnvironment environment = new StandardEnvironment();
        Map<String, Object> map = new HashMap<String, Object>();
        map.put("spring.profiles.active", profile);
        map.put("spring.main.web-application-type", "none");
        environment.getPropertySources().addFirst(new MapPropertySource("profiles", map));
        return environment;
    }


    protected Environment clean(Environment value) {
        Environment result = new Environment(value.getName(), value.getProfiles(),
                value.getLabel(), getVersion(), value.getState());

        String profile = result.getProfiles() == null ? null
                : StringUtils.arrayToCommaDelimitedString(result.getProfiles());
        Locations locations = getLocations(result.getName(), profile, result.getLabel());

        for (PropertySource source : value.getPropertySources()) {
            String name = source.getName();
            if (this.environment.getPropertySources().contains(name)) {
                continue;
            }
            name = name.replace("applicationConfig: [", "");
            name = name.replace("]", "");
            if (getSearchLocations() != null) {
                boolean matches = false;
                String normal = name;
                if (normal.startsWith("file:")) {
                    normal = StringUtils
                            .cleanPath(new File(normal.substring("file:".length()))
                                    .getAbsolutePath());
                }

                for (String pattern : locations.getLocations()) {
                    if (!pattern.contains(":")) {
                        pattern = "file:" + pattern;
                    }
                    if (pattern.startsWith("file:")) {
                        pattern = StringUtils
                                .cleanPath(new File(pattern.substring("file:".length()))
                                        .getAbsolutePath())
                                + "/";
                    }
                    if (logger.isTraceEnabled()) {
                        logger.trace("Testing pattern: " + pattern
                                + " with property source: " + name);
                    }
                    if (normal.startsWith(pattern)
                            && !normal.substring(pattern.length()).contains("/")) {
                        matches = true;
                        break;
                    }
                }
                if (!matches) {
                    // Don't include this one: it wasn't matched by our search locations
                    if (logger.isDebugEnabled()) {
                        logger.debug("Not adding property source: " + name);
                    }
                    continue;
                }
            }
            logger.info("Adding property source: " + name);
            result.add(new PropertySource(name, source.getSource()));
        }
        return result;
    }







    @Override
    public Locations getLocations(String application, String profile, String label) {
        String[] locations = getSearchLocations();
        if (getSearchLocations() == null || getSearchLocations().length == 0 ||  StringUtils.isEmpty(label)) {
            locations = DEFAULT_LOCATIONS;
        }
        Collection<String> output = new LinkedHashSet<String>();

        for (String location : locations) {
            String[] profiles = new String[] { profile };
            if (profile != null) {
                profiles = StringUtils.commaDelimitedListToStringArray(profile);
            }
            String[] apps = new String[] { application };
            if (application != null) {
                apps = StringUtils.commaDelimitedListToStringArray(application);
            }
            for (String prof : profiles) {
                for (String app : apps) {
                    String value = location;
                    if (application != null) {
                        value = value.replace("{application}", app);
                    }
                    if (prof != null) {
                        value = value.replace("{profile}", prof);
                    }
                    if (label != null) {
                        value = value.replace("{label}", label);
                    }
                    if (!value.endsWith("/")) {
                        value = value + "/";
                    }
                    if (isDirectory(value)) {
                        output.add(value);
                    }
                }
            }
        }



        return new Locations(application, profile, label, getVersion(),
                output.toArray(new String[0]));
    }

    private boolean isDirectory(String location) {
        return !location.contains("{") && !location.endsWith(".properties")
                && !location.endsWith(".yml") && !location.endsWith(".yaml");
    }


    public String getVersion() {
        return version;
    }

    public String[] getSearchLocations() {
        return this.searchLocations;
    }

    public void setSearchLocations(String... locations) {
        this.searchLocations = locations;
        if (locations != null) {
            for (int i = 0; i < locations.length; i++) {
                String location = locations[i];
                if (isDirectory(location) && !location.endsWith("/")) {
                    location = location + "/";
                }
                locations[i] = location;
            }
        }
    }

    @Override
    public int getOrder() {
        return this.order;
    }
}
