package cn.ipokerface.snowflake;

/**
 * Created by       PokerFace
 * Create Date      2019-11-25.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class SnowflakeIdGenerator {

    /** start timestamp (2020-06-01 00:00:00) */
    private long twepoch = 1590940800000L;

    /** worker id bit */
    private final long workerIdBits = 5L;

    /** data center bit */
    private final long datacenterIdBits = 5L;

    /** max worker id */
    private final long maxWorkerId = -1L ^ (-1L << workerIdBits);

    /**  max data center id <code>=31</code> */
    private final long maxDatacenterId = -1L ^ (-1L << datacenterIdBits);

    /** sequence bit */
    private final long sequenceBits = 12L;

    /** worker id shift */
    private final long workerIdShift = sequenceBits;

    /** data center id shift  */
    private final long datacenterIdShift = sequenceBits + workerIdBits;

    /** timestamp shift */
    private final long timestampLeftShift = sequenceBits + workerIdBits + datacenterIdBits;

    /** sequence mask max = 4095 (0b111111111111=0xfff=4095) */
    private final long sequenceMask = -1L ^ (-1L << sequenceBits);

    /** worker id ID(0~31) */
    private long workerId;

    /** data center ID(0~31) */
    private long datacenterId;

    /** sequence (0~4095) */
    private long sequence = 0L;

    /** last timestamp */
    private long lastTimestamp = -1L;


    //==============================Constructors=====================================
    /**
     *  Constructors. Under distributed systems. Every service may has sorts of instance to obtain.
     *  So you may have some strategy to differ system's and instance.
     *
     * @param workerId worker id number between(0~31)
     * @param dataCenterId data center ID  number between (0~31)
     *
     */
    public SnowflakeIdGenerator(long workerId, long dataCenterId) {
        if (workerId > maxWorkerId || workerId < 0) {
            throw new IllegalArgumentException(String.format("worker Id can't be greater than %d or less than 0",
                    maxWorkerId));
        }
        if (datacenterId > maxDatacenterId || datacenterId < 0) {
            throw new IllegalArgumentException(String.format("datacenter Id can't be greater than %d or less than 0",
                    maxDatacenterId));
        }
        this.workerId = workerId;
        this.datacenterId = datacenterId;
    }


    /**
     *  Constructors. Under distributed systems. Every service may has sorts of instance to obtain.
     *  So you may have some strategy to differ system's and instance.
     *
     * @param workerId worker id number between(0~31)
     * @param dataCenterId data center ID  number between (0~31)
     * @param startTimestamp define a start timestamp  show not change when restart instance.
     *
     */
    public SnowflakeIdGenerator(long workerId, long dataCenterId,long startTimestamp) {
        if (workerId > maxWorkerId || workerId < 0) {
            throw new IllegalArgumentException(String.format("worker Id can't be greater than %d or less than 0",
                    maxWorkerId));
        }
        if (datacenterId > maxDatacenterId || datacenterId < 0) {
            throw new IllegalArgumentException(String.format("dataCenter Id can't be greater than %d or less than 0",
                    maxDatacenterId));
        }
        if (startTimestamp >= System.currentTimeMillis())  {
            throw new IllegalArgumentException("start timestamp should not greater than now");
        }
        this.workerId = workerId;
        this.twepoch = startTimestamp;
        this.datacenterId = datacenterId;
    }




    // ==============================Methods==========================================
    /**
     *
     * Generator an Id of this generator. this generator do not return same id between some calls.
     * So you want use it on different places. please store it by your own collection.
     *
     * @return SnowflakeId
     */
    public synchronized long nextId() {
        long timestamp = System.currentTimeMillis();

        if (timestamp < lastTimestamp) {
            throw new RuntimeException(
                    String.format("Clock moved backwards.  Refusing to generate id for %d milliseconds",
                            lastTimestamp - timestamp));
        }


        if (lastTimestamp == timestamp) {
            sequence = (sequence + 1) & sequenceMask;

            if (sequence == 0) {

                timestamp = tillNextMillis(lastTimestamp);
            }
        }
        else {
            sequence = 0L;
        }

        lastTimestamp = timestamp;

        return ((timestamp - twepoch) << timestampLeftShift)
                | (datacenterId << datacenterIdShift)
                | (workerId << workerIdShift)
                | sequence;
    }


    /**
     * 阻塞到下一个毫秒，直到获得新的时间戳
     *
     * @param lastTimestamp
     *          上次生成ID的时间截
     * @return
     *          当前时间戳
     */
    private long tillNextMillis(long lastTimestamp) {
        long timestamp = System.currentTimeMillis();
        while (timestamp <= lastTimestamp) {
            timestamp = System.currentTimeMillis();
        }
        return timestamp;
    }



}