package cn.ipokerface.redis;

import redis.clients.jedis.*;
import redis.clients.jedis.params.GeoRadiusParam;
import redis.clients.jedis.params.SetParams;
import redis.clients.jedis.params.ZAddParams;
import redis.clients.jedis.params.ZIncrByParams;
import redis.clients.jedis.util.SafeEncoder;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by       PokerFace
 * Create Date      2019-11-27.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class JedisClusterClient implements JedisClient {


    private JedisCluster jedisCluster;

    public JedisClusterClient(JedisCluster jedisCluster){
        this.jedisCluster = jedisCluster;
    }

    public String set(String key, String value) {

        return jedisCluster.set(key, value);
    }

    public String set(String key, String value, long expireInMillis) {
        return jedisCluster.set(key, value, SetParams.setParams().px(expireInMillis));
    }

    public String set(String key, String value, SetParams setParams) {
        return jedisCluster.set(key, value, setParams);
    }

    public String multiSet(String... keyValues) {
        return jedisCluster.mset(keyValues);
    }

    public Long multiSetNx(String... keyValues) {
        return jedisCluster.msetnx(keyValues);
    }

    public String get(String key) {
        return jedisCluster.get(key);
    }


    public List<String> get(String... keys) {
        return jedisCluster.mget(keys);
    }

    public Long exists(String... keys) {
        return jedisCluster.exists(keys);
    }

    public Boolean exists(String key) {
        return jedisCluster.exists(key);
    }

    public Long expire(String key, int seconds) {
        return jedisCluster.expire(key, seconds);
    }

    public Long expire(String key, long timeMillis) {
        return this.expire(key, new Long(timeMillis / 1000).intValue());
    }

    public Long expireAt(String key, long unixTimestamp) {
        return jedisCluster.expireAt(key, unixTimestamp);
    }

    public Long persist(String key) {
        return jedisCluster.persist(key);
    }

    public Long ttl(String key) {
        return jedisCluster.ttl(key);
    }

    public String echo(String string) {
        return jedisCluster.echo(string);
    }

    public Long delete(String... keys) {
        return jedisCluster.unlink(keys);
    }


    public String type(String key) {
        return jedisCluster.type(key);
    }

    public Set<String> keys(String pattern) {
        return jedisCluster.keys(pattern);
    }

    public Long touch(String... keys) {
        return jedisCluster.touch(keys);
    }


    public String getSet(String key, String value) {
        return jedisCluster.getSet(key, value);
    }

    public Long increase(String key) {
        return jedisCluster.incr(key);
    }

    public Long increaseBy(String key, long increment) {
        return jedisCluster.incrBy(key, increment);
    }

    public Long decrease(String key) {
        return jedisCluster.decr(key);
    }

    public Long decreaseBy(String key, long decrement) {
        return jedisCluster.decrBy(key, decrement);
    }

    public Double increaseByFloat(String key, float increment) {
        return jedisCluster.incrByFloat(key, increment);
    }

    public Long append(String key, String value) {
        return jedisCluster.append(key, value);
    }

    public String subString(String key, int start, int end) {
        return jedisCluster.substr(key, start, end);
    }

    public Boolean setBit(String key, long offset, boolean value) {
        return jedisCluster.setbit(key, offset, value);
    }

    public Boolean setBit(String key, long offset, String value) {
        return jedisCluster.setbit(key, offset, value);
    }

    public Boolean getBit(String key, long offset) {
        return jedisCluster.getbit(key, offset);
    }

    public Long bigCount(String key) {
        return jedisCluster.bitcount(key);
    }

    public Long bitCount(String key, long start, long end) {
        return jedisCluster.bitcount(key, start, end);
    }

    public Long bitOp(BitOP op, String destKey, String... srcKeys) {
        return jedisCluster.bitop(op, destKey, srcKeys);
    }

    public List<Long> bitField(String key, String... arguments) {
        return jedisCluster.bitfield(key, arguments);
    }

    public Long setRange(String key, long offset, String value) {
        return jedisCluster.setrange(key, offset, value);
    }

    public String getRange(String key, long offset, long endOffset) {
        return jedisCluster.getrange(key, offset, endOffset);
    }

    // ========================== map ===============================


    public Long mapSet(String key, String field, String value) {
        return jedisCluster.hset(key, field, value);
    }

    public Long mapSet(String key, Map<String, String> map) {
        return jedisCluster.hset(key, map);
    }

    public Long mapSetNx(String key, String field, String value) {
        return jedisCluster.hsetnx(key, field, value);
    }

    public Long mapIncreaseBy(String key, String field, long value) {
        return jedisCluster.hincrBy(key, field, value);
    }

    public Double mapIncreaseByFloat(String key, String field, float value) {

        return jedisCluster.hincrByFloat(SafeEncoder.encode(key), SafeEncoder.encode(field), value);
    }

    public String mapGet(String key, String field) {
        return jedisCluster.hget(key, field);
    }

    public List<String> mapMultiGet(String key, String... fields) {
        return jedisCluster.hmget(key, fields);
    }

    public Long mapLength(String key) {
        return jedisCluster.hlen(key);
    }

    public Boolean mapExist(String key, String field) {
        return jedisCluster.hexists(key, field);
    }

    public Set<String> mapKeys(String key) {
        return jedisCluster.hkeys(key);
    }

    public List<String> mapValues(String key) {
        return jedisCluster.hvals(key);
    }

    public Map<String, String> mapAll(String key) {
        return jedisCluster.hgetAll(key);
    }

    public Long mapDelete(String key, String... fields) {
        return jedisCluster.hdel(key, fields);
    }



    // ===================== list ==============================


    public Long listPush(String key, String... values) {
        return jedisCluster.rpush(key, values);
    }


    public Long listInsert(String key, ListPosition where, String anchor, String value) {
        return jedisCluster.linsert(key, where, anchor, value);
    }

    public Long listRightPush(String key, String... values) {
        return jedisCluster.rpush(key, values);
    }

    public Long listRightPushExist(String key, String... values) {
        return jedisCluster.rpushx(key, values);
    }

    public Long listLeftPush(String key, String... values) {
        return jedisCluster.lpush(key, values);
    }

    public Long listLeftPushExist(String key, String... values) {
        return jedisCluster.lpushx(key, values);
    }

    public Long listLength(String key) {
        return jedisCluster.llen(key);
    }

    public List<String> listRange(String key, long start, long end) {
        return jedisCluster.lrange(key, start, end);
    }

    public String listTrim(String key, long start, long end) {
        return jedisCluster.ltrim(key, start, end);
    }

    public String listIndex(String key, long index) {
        return jedisCluster.lindex(key, index);
    }

    public String listSet(String key, long index, String value) {
        return jedisCluster.lset(key, index, value);
    }

    public Long listRemove(String key, long count, String value) {
        return jedisCluster.lrem(key, count, value);
    }

    public String listPop(String key) {
        return this.listRightPop(key);
    }

    public String listRightPop(String key) {
        return jedisCluster.rpop(key);
    }

    public String listLeftPop(String key) {
        return jedisCluster.lpop(key);
    }

    public String listRightPopLeftPush(String srcKey, String targetKey) {
        return jedisCluster.rpoplpush(srcKey, targetKey);
    }


    public List<String> listBlockingLeftPop(int timeout, String... keys) {
        return jedisCluster.blpop(timeout, keys);
    }


    public List<String> listBlockingRightPop(int timeout, String... keys) {
        return jedisCluster.brpop(timeout, keys);
    }

    public String listBlockingRightPopLeftPush(String srcKey, String targetKey, int timeout) {
        return jedisCluster.brpoplpush(srcKey, targetKey, timeout);
    }


    // =========================== set ==============================


    public Long setsAdd(String key, String... values) {
        return jedisCluster.sadd(key, values);
    }

    public Set<String> setsValues(String key) {
        return jedisCluster.smembers(key);
    }

    public Long setsDelete(String key, String... values) {
        return jedisCluster.srem(key, values);
    }

    public String setsPop(String key) {
        return jedisCluster.spop(key);
    }

    public Set<String> setsPop(String key, long count) {
        return jedisCluster.spop(key, count);
    }

    public Long setsLength(String key) {
        return jedisCluster.scard(key);
    }

    public Long setsMove(String srcKey, String targetKey, String value) {
        return jedisCluster.smove(srcKey, targetKey, value);
    }

    public Boolean setsExist(String key, String value) {
        return jedisCluster.sismember(key, value);
    }

    public Set<String> setsInter(String... keys) {
        return jedisCluster.sinter(keys);
    }

    public Long setsInterStore(String destKey, String... keys) {
        return jedisCluster.sinterstore(destKey, keys);
    }

    public Set<String> setsUnion(String... keys) {
        return jedisCluster.sunion(keys);
    }

    public Long setsUnionStore(String destKey, String... keys) {
        return jedisCluster.sinterstore(destKey, keys);
    }

    public Set<String> setsDifferent(String... keys) {
        return jedisCluster.sdiff(keys);
    }

    public Long setsDifferentStore(String destKey, String... keys) {
        return jedisCluster.sdiffstore(destKey, keys);
    }

    public String setsRandomValue(String key) {
        return jedisCluster.srandmember(key);
    }

    public List<String> setsRandomValue(String key, int count) {
        return jedisCluster.srandmember(key, count);
    }


    // ========================= zset ===========================

    public Long zsetAdd(String key, double score, String value) {
        return jedisCluster.zadd(key, score, value);
    }

    public Long zsetAdd(String key, double score, String value, ZAddParams params) {
        return jedisCluster.zadd(key, score, value, params);
    }

    public Long zsetAdd(String key, Map<String, Double> valueScores) {
        return jedisCluster.zadd(key, valueScores);
    }

    public Long zsetAdd(String key, Map<String, Double> valueScores, ZAddParams params) {
        return jedisCluster.zadd(key, valueScores, params);
    }

    public Set<String> zsetRange(String key, long start, long end) {
        return jedisCluster.zrange(key, start, end);
    }

    public Long zsetDelete(String key, String... values) {
        return jedisCluster.zrem(key, values);
    }

    public Double zsetIncreaseBy(String key, double increment, String value) {
        return jedisCluster.zincrby(key, increment, value);
    }

    public Double zsetIncreaseBy(String key, double increment, String value, ZIncrByParams params) {
        return jedisCluster.zincrby(key, increment, value, params);
    }

    public Long zsetIndexOf(String key, String value) {
        return jedisCluster.zrank(key, value);
    }

    public Long zsetReverseIndexOf(String key, String value) {
        return jedisCluster.zrevrank(key,value);
    }

    public Set<String> zsetReverseRange(String key, long start, long end) {
        return jedisCluster.zrevrange(key, start, end);
    }

    public Set<Tuple> zsetRangeWithScores(String key, long start, long end) {
        return jedisCluster.zrangeWithScores(key, start, end);
    }

    public Set<Tuple> zsetReverseRangeWithScores(String key, long start, long end) {
        return jedisCluster.zrevrangeWithScores(key, start, end);
    }

    public Long zsetLength(String key) {
        return jedisCluster.zcard(key);
    }

    public Double zsetScore(String key, String value) {
        return jedisCluster.zscore(key, value);
    }


    public List<String> sort(String key) {
        return jedisCluster.sort(key);
    }

    public List<String> sort(String key, SortingParams sortingParameters) {
        return jedisCluster.sort(key, sortingParameters);
    }

    public Long sortStore(String key, String destKey) {
        return jedisCluster.sort(key, destKey);
    }

    public Long sortStore(String key, SortingParams sortingParameters, String destKey) {
        return jedisCluster.sort(key, sortingParameters, destKey);
    }

    public Set<String> zsetRangeByScore(String key, double min, double max) {
        return jedisCluster.zrangeByScore(key, min, max);
    }

    public Set<String> zsetRangeByScore(String key, String min, String max) {
        return jedisCluster.zrangeByScore(key, min, max);
    }

    public Set<String> zsetRangeByScore(String key, double min, double max, int offset, int count) {
        return jedisCluster.zrangeByScore(key, min, max, offset, count);
    }

    public Set<String> zsetRangeByScore(String key, String min, String max, int offset, int count) {
        return jedisCluster.zrangeByScore(key, min, max, offset, count);
    }

    public Set<Tuple> zsetRangeByScoreWithScores(String key, double min, double max) {
        return jedisCluster.zrangeByScoreWithScores(key, min, max);
    }

    public Set<Tuple> zsetRangeByScoreWithScores(String key, String min, String max) {
        return jedisCluster.zrangeByScoreWithScores(key, min, max);
    }

    public Set<Tuple> zsetRangeByScoreWithScores(String key, double min, double max, int offset, int count) {
        return jedisCluster.zrangeByScoreWithScores(key, min, max, offset, count);
    }

    public Set<Tuple> zsetRangeByScoreWithScores(String key, String min, String max, int offset, int count) {
        return jedisCluster.zrangeByScoreWithScores(key, min, max, offset, count);
    }

    public Set<String> zsetReverseRangeByScore(String key, double min, double max) {
        return jedisCluster.zrangeByScore(key, min, max);
    }

    public Set<String> zsetReverseRangeByScore(String key, String min, String max) {
        return jedisCluster.zrangeByScore(key, min, max);
    }

    public Set<String> zsetReverseRangeByScore(String key, double min, double max, int offset, int count) {
        return jedisCluster.zrangeByScore(key, min, max, offset, count);
    }

    public Set<String> zsetReverseRangeByScore(String key, String min, String max, int offset, int count) {
        return jedisCluster.zrangeByScore(key, min, max, offset, count);
    }

    public Set<Tuple> zsetReverseRangeByScoreWithScores(String key, double min, double max) {
        return jedisCluster.zrangeByScoreWithScores(key, min, max);
    }

    public Set<Tuple> zsetReverseRangeByScoreWithScores(String key, String min, String max) {
        return jedisCluster.zrangeByScoreWithScores(key, min, max);
    }

    public Set<Tuple> zsetReverseRangeByScoreWithScores(String key, double min, double max, int offset, int count) {
        return jedisCluster.zrangeByScoreWithScores(key, min, max, offset, count);
    }

    public Set<Tuple> zsetReverseRangeByScoreWithScores(String key, String min, String max, int offset, int count) {
        return jedisCluster.zrangeByScoreWithScores(key, min, max, offset, count);
    }

    public Long zsetDeleteRangeByIndex(String key, long start, long end) {
        return jedisCluster.zremrangeByRank(key, start, end);
    }

    public Long zsetDeleteRangeByScore(String key, double min, double max) {
        return jedisCluster.zremrangeByScore(key, min, max);
    }

    public Long zsetDeleteRangeByScore(String key, String min, String max) {
        return jedisCluster.zremrangeByScore(key, min, max);
    }

    public Long zsetUnionStore(String destKey, String... keys) {
        return jedisCluster.zunionstore(destKey, keys);
    }

    public Long zsetUnionStore(String destKey, ZParams params, String... keys) {
        return jedisCluster.zunionstore(destKey,params, keys);
    }

    public Long zsetInterStore(String destKey, String... keys) {
        return jedisCluster.zinterstore(destKey, keys);
    }

    public Long zsetInterStore(String destKey, ZParams params, String... keys) {
        return jedisCluster.zinterstore(destKey, params, keys);
    }


    public Long zsetLexLength(String key, String min, String max) {
        return jedisCluster.zlexcount(key, min, max);
    }

    public Set<String> zsetRangeByLex(String key, String min, String max) {
        return jedisCluster.zrangeByLex(key, min, max);
    }

    public Set<String> zsetRangeByLex(String key, String min, String max, int offset, int count) {
        return jedisCluster.zrangeByLex(key, min, max, offset, count);
    }

    public Set<String> zsetReverseRangeByLex(String key, String min, String max) {
        return jedisCluster.zrevrangeByLex(key, min, max);
    }

    public Set<String> zsetReverseRangeByLex(String key, String min, String max, int offset, int count) {
        return jedisCluster.zrevrangeByLex(key, min, max, offset, count);
    }

    public Long zsetDeleteRangeByLex(String key, String min, String max) {
        return jedisCluster.zremrangeByLex(key, min, max);
    }

    // ========================= geo =================================

    public Long geoAdd(String key, double longitude, double latitude, String value) {
        return jedisCluster.geoadd(key, longitude, latitude, value);
    }

    public Long geoAdd(String key, Map<String, GeoCoordinate> memberCoordinateMap) {
        return jedisCluster.geoadd(key, memberCoordinateMap);
    }

    public Double geoDistant(String key, String source, String target) {
        return jedisCluster.geodist(key, source, target);
    }

    public Double geoDistant(String key, String source, String target, GeoUnit unit) {
        return jedisCluster.geodist(key, source, target, unit);
    }

    public List<String> geoHash(String key, String... values) {
        return jedisCluster.geohash(key, values);
    }

    public List<GeoCoordinate> geoPos(String key, String... values) {
        return jedisCluster.geopos(key, values);
    }

    public List<GeoRadiusResponse> geoRadius(String key, double longitude, double latitude, double radius, GeoUnit unit) {
        return jedisCluster.georadius(key, longitude, latitude, radius, unit );
    }

    public List<GeoRadiusResponse> geoRadiusReadonly(String key, double longitude, double latitude, double radius, GeoUnit unit) {
        return jedisCluster.georadiusReadonly(key, longitude, latitude, radius, unit);
    }

    public List<GeoRadiusResponse> geoRadius(String key, double longitude, double latitude, double radius, GeoUnit unit, GeoRadiusParam param) {
        return jedisCluster.georadius(key, longitude, latitude, radius, unit ,param);
    }

    public List<GeoRadiusResponse> geoRadiusReadonly(String key, double longitude, double latitude, double radius, GeoUnit unit, GeoRadiusParam param) {
        return jedisCluster.georadiusReadonly(key, longitude, latitude, radius, unit, param);
    }

    public List<GeoRadiusResponse> geoRadiusByMember(String key, String member, double radius, GeoUnit unit) {
        return jedisCluster.georadiusByMember(key, member, radius, unit);
    }

    public List<GeoRadiusResponse> geoRadiusByMemberReadonly(String key, String member, double radius, GeoUnit unit) {
        return jedisCluster.georadiusByMemberReadonly(key, member, radius, unit);
    }

    public List<GeoRadiusResponse> geoRadiusByMember(String key, String member, double radius, GeoUnit unit, GeoRadiusParam param) {
        return jedisCluster.georadiusByMember(key, member, radius, unit, param);
    }

    public List<GeoRadiusResponse> geoRadiusByMemberReadonly(String key, String member, double radius, GeoUnit unit, GeoRadiusParam param) {
        return jedisCluster.georadiusByMemberReadonly(key, member, radius, unit, param);
    }


    // ========================= other =================================


    public void subscribe(JedisPubSub jedisPubSub, String... channels) {
        jedisCluster.subscribe(jedisPubSub, channels);
    }

    public Long publish(String channel, String message) {
        return jedisCluster.publish(channel, message);
    }

    public void patternSubscribe(JedisPubSub jedisPubSub, String... patterns) {
        jedisCluster.psubscribe(jedisPubSub, patterns);
    }

    public Object eval(String script, List<String> keys, List<String> args) {
        return jedisCluster.eval(script, keys, args);
    }

    public Object eval(String script, int keyCount, String... params) {
        return jedisCluster.eval(script, keyCount, params);
    }

    public Object eval(String script) {
        return eval(script, 0);
    }

    public Object evalSha(String sha1) {
        return jedisCluster.evalsha(sha1,0);
    }

    public Object evalSha(String sha1, List<String> keys, List<String> args) {
        return jedisCluster.evalsha(sha1, keys, args);
    }

    public Object evalSha(String sha1, int keyCount, String... params) {
        return jedisCluster.evalsha(sha1, keyCount, params);
    }

    public Boolean scriptExists(String sha1) {
        return jedisCluster.scriptExists(sha1, "");
    }

    public List<Boolean> scriptExists(String... sha1) {
        return jedisCluster.scriptExists("", sha1);
    }

    public String scriptLoad(String script) {
        return jedisCluster.scriptLoad(script, "");
    }
}
