package cn.ipokerface.redis;

import redis.clients.jedis.*;
import redis.clients.jedis.params.GeoRadiusParam;
import redis.clients.jedis.params.SetParams;
import redis.clients.jedis.params.ZAddParams;
import redis.clients.jedis.params.ZIncrByParams;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by       PokerFace
 * Create Date      2019-11-27.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:     Core Api class of redis-client
 *
 */
public class JedisSimpleClient implements JedisClient {



    JedisPool jedisPool;


    /**
     *  instance a jedis client.
     *
     * @param jedisPool jedisPool instance.
     */
    public JedisSimpleClient(JedisPool jedisPool){
        this.jedisPool = jedisPool;
    }



    public String set(String key, String value)
    {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.set(key, value);
        jedis.close();
        return result;

    }

    public String set(String key, String value, long expireInMillis)
    {
        return this.set(key, value, SetParams.setParams().px(expireInMillis));
    }

    public String set(String key, String value, SetParams setParams)
    {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.set(key, value, setParams);
        jedis.close();
        return result;
    }

    public String multiSet(String... keyValues) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.mset(keyValues);
        jedis.close();
        return result;
    }

    public Long multiSetNx(String... keyValues) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.msetnx(keyValues);
        jedis.close();
        return result;
    }

    public String get(String key){
        Jedis jedis = jedisPool.getResource();
        String result = jedis.get(key);
        jedis.close();
        return result;
    }


    public List<String> get(String... keys) {
        Jedis jedis = jedisPool.getResource();
        List<String> result = jedis.mget(keys);
        jedis.close();
        return result;
    }

    public Long exists(String... keys) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.exists(keys);
        jedis.close();
        return result;
    }

    public Boolean exists(String key) {
        Jedis jedis = jedisPool.getResource();
        Boolean result = jedis.exists(key);
        jedis.close();
        return result;
    }

    public Long expire(String key, int seconds) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.expire(key, seconds);
        jedis.close();
        return result;
    }

    public Long expire(String key, long timeMillis) {
        return this.expire(key, new Long(timeMillis / 1000).intValue());
    }


    public Long expireAt(String key, long unixTimestamp) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.expireAt(key, unixTimestamp);
        jedis.close();
        return result;
    }

    public Long persist(String key) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.persist(key);
        jedis.close();
        return result;
    }

    public Long ttl(String key) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.ttl(key);
        jedis.close();
        return result;
    }


    public String echo(String string) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.echo(string);
        jedis.close();
        return result;
    }

    public Long delete(String... keys) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.unlink(keys);
        jedis.close();
        return result;
    }

    public String type(String key) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.type(key);
        jedis.close();
        return result;
    }

    public Set<String> keys(String pattern) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.keys(pattern);
        jedis.close();
        return result;
    }

    public Long touch(String... keys) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.touch(keys);
        jedis.close();
        return result;
    }



    public String getSet(String key, String value) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.getSet(key, value);
        jedis.close();
        return result;
    }

    public Long increase(String key) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.incr(key);
        jedis.close();
        return result;
    }

    public Long increaseBy(String key, long increment) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.incrBy(key, increment);
        jedis.close();
        return result;
    }

    public Long decrease(String key) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.decr(key);
        jedis.close();
        return result;
    }

    public Long decreaseBy(String key, long decrement) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.decrBy(key, decrement);
        jedis.close();
        return result;
    }

    public Double increaseByFloat(String key, float increment) {
        Jedis jedis = jedisPool.getResource();
        Double result = jedis.incrByFloat(key, increment);
        jedis.close();
        return result;
    }

    public Long append(String key, String value) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.append(key, value);
        jedis.close();
        return result;
    }

    public String subString(String key, int start, int end) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.substr(key, start, end);
        jedis.close();
        return result;
    }

    public Boolean setBit(String key, long offset, boolean value) {
        Jedis jedis = jedisPool.getResource();
        Boolean result = jedis.setbit(key, offset, value);
        jedis.close();
        return result;
    }

    public Boolean setBit(String key, long offset, String value) {
        Jedis jedis = jedisPool.getResource();
        Boolean result = jedis.setbit(key, offset, value);
        jedis.close();
        return result;
    }

    public Boolean getBit(String key, long offset) {
        Jedis jedis = jedisPool.getResource();
        Boolean result = jedis.getbit(key, offset);
        jedis.close();
        return result;
    }


    public Long bigCount(String key) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.bitcount(key);
        jedis.close();
        return result;
    }

    public Long bitCount(String key, long start, long end) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.bitcount(key, start, end);
        jedis.close();
        return result;
    }

    public Long bitOp(BitOP op, String destKey, String... srcKeys) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.bitop(op, destKey, srcKeys);
        jedis.close();
        return result;
    }


    public List<Long> bitField(String key, String... arguments) {
        Jedis jedis = jedisPool.getResource();
        List<Long> result = jedis.bitfield(key, arguments);
        jedis.close();
        return result;
    }

    public Long setRange(String key, long offset, String value) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.setrange(key, offset, value);
        jedis.close();
        return result;
    }

    public String getRange(String key, long offset, long endOffset) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.getrange(key, offset, endOffset);
        jedis.close();
        return result;
    }

    // =========================== map ==========================


    public Long mapSet(String key, String field, String value) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.hset(key, field, value);
        jedis.close();
        return result;
    }

    public Long mapSet(String key, Map<String, String> map) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.hset(key, map);
        jedis.close();
        return result;
    }

    public Long mapSetNx(String key, String field, String value) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.hsetnx(key, field, value);
        jedis.close();
        return result;
    }

    public Long mapIncreaseBy(String key, String field, long value) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.hincrBy(key, field, value);
        jedis.close();
        return result;
    }

    public Double mapIncreaseByFloat(String key, String field, float value) {
        Jedis jedis = jedisPool.getResource();
        Double result = jedis.hincrByFloat(key, field, value);
        jedis.close();
        return result;
    }

    public String mapGet(String key, String field) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.hget(key, field);
        jedis.close();
        return result;
    }

    public List<String> mapMultiGet(String key, String... fields) {
        Jedis jedis = jedisPool.getResource();
        List<String> result = jedis.hmget(key, fields);
        jedis.close();
        return result;
    }

    public Long mapLength(String key) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.hlen(key);
        jedis.close();
        return result;
    }

    public Boolean mapExist(String key, String field) {
        Jedis jedis = jedisPool.getResource();
        Boolean result = jedis.hexists(key, field);
        jedis.close();
        return result;
    }

    public Set<String> mapKeys(String key) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.hkeys(key);
        jedis.close();
        return result;
    }

    public List<String> mapValues(String key) {
        Jedis jedis = jedisPool.getResource();
        List<String> result = jedis.hvals(key);
        jedis.close();
        return result;
    }

    public Map<String, String> mapAll(String key) {
        Jedis jedis = jedisPool.getResource();
        Map<String, String> result = jedis.hgetAll(key);
        jedis.close();
        return result;
    }

    public Long mapDelete(String key, String... fields) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.hdel(key, fields);
        jedis.close();
        return result;
    }

    // ========================== list ==========================


    public Long listPush(String key, String... values) {
        return this.listRightPush(key,values);
    }

    public Long listInsert(String key, ListPosition where, String anchor, String value) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.linsert(key, where, anchor, value);
        jedis.close();
        return result;
    }

    public Long listRightPush(String key, String... values) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.rpush(key, values);
        jedis.close();
        return result;
    }

    public Long listRightPushExist(String key, String... values) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.rpushx(key, values);
        jedis.close();
        return result;
    }

    public Long listLeftPush(String key, String... values) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.lpush(key, values);
        jedis.close();
        return result;
    }

    public Long listLeftPushExist(String key, String... values) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.lpushx(key, values);
        jedis.close();
        return result;
    }

    public Long listLength(String key) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.llen(key);
        jedis.close();
        return result;
    }

    public List<String> listRange(String key, long start, long end) {
        Jedis jedis = jedisPool.getResource();
        List<String> result = jedis.lrange(key, start, end);
        jedis.close();
        return result;
    }

    public String listTrim(String key, long start, long end) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.ltrim(key, start, end);
        jedis.close();
        return result;
    }

    public String listIndex(String key, long index) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.lindex(key, index);
        jedis.close();
        return result;
    }

    public String listSet(String key, long index, String value) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.lset(key, index, value);
        jedis.close();
        return result;
    }

    public Long listRemove(String key, long count, String value) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.lrem(key, count, value);
        jedis.close();
        return result;
    }

    public String listPop(String key) {
        return this.listRightPop(key);
    }

    public String listRightPop(String key) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.rpop(key);
        jedis.close();
        return result;
    }

    public String listLeftPop(String key) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.lpop(key);
        jedis.close();
        return result;
    }

    public String listRightPopLeftPush(String srcKey, String targetKey) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.rpoplpush(srcKey, targetKey);
        jedis.close();
        return result;
    }



    public List<String> listBlockingLeftPop(int timeout, String... keys) {
        Jedis jedis = jedisPool.getResource();
        List<String> result = jedis.blpop(timeout, keys);
        jedis.close();
        return result;
    }



    public List<String> listBlockingRightPop(int timeout, String... keys) {
        Jedis jedis = jedisPool.getResource();
        List<String> result = jedis.brpop(timeout, keys);
        jedis.close();
        return result;
    }

    public String listBlockingRightPopLeftPush(String srcKey, String targetKey, int timeout) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.brpoplpush(srcKey, targetKey, timeout);
        jedis.close();
        return result;
    }

    // =========================== set ==========================


    public Long setsAdd(String key, String... values) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.sadd(key, values);
        jedis.close();
        return result;
    }

    public Set<String> setsValues(String key) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.smembers(key);
        jedis.close();
        return result;
    }

    public Long setsDelete(String key, String... values) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.srem(key, values);
        jedis.close();
        return result;
    }

    public String setsPop(String key) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.spop(key);
        jedis.close();
        return result;
    }

    public Set<String> setsPop(String key, long count) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.spop(key, count);
        jedis.close();
        return result;
    }

    public Long setsLength(String key) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.scard(key);
        jedis.close();
        return result;
    }

    public Long setsMove(String srcKey, String targetKey, String value) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.smove(srcKey, targetKey, value);
        jedis.close();
        return result;
    }

    public Boolean setsExist(String key, String value) {
        Jedis jedis = jedisPool.getResource();
        Boolean result = jedis.sismember(key, value);
        jedis.close();
        return result;
    }

    public Set<String> setsInter(String... keys) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.sinter(keys);
        jedis.close();
        return result;
    }

    public Long setsInterStore(String destKey, String... keys) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.sinterstore(destKey, keys);
        jedis.close();
        return result;
    }

    public Set<String> setsUnion(String... keys) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.sunion(keys);
        jedis.close();
        return result;
    }

    public Long setsUnionStore(String destKey, String... keys) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.sunionstore(destKey, keys);
        jedis.close();
        return result;
    }

    public Set<String> setsDifferent(String... keys) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.sdiff(keys);
        jedis.close();
        return result;
    }

    public Long setsDifferentStore(String destKey, String... keys) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.sdiffstore(destKey, keys);
        jedis.close();
        return result;
    }

    public String setsRandomValue(String key) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.srandmember(key);
        jedis.close();
        return result;
    }

    public List<String> setsRandomValue(String key, int count) {
        Jedis jedis = jedisPool.getResource();
        List<String> result = jedis.srandmember(key, count);
        jedis.close();
        return result;
    }



    //  =========================== zet ==========================


    public Long zsetAdd(String key, double score, String value) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zadd(key, score,value);
        jedis.close();
        return result;
    }

    public Long zsetAdd(String key, double score, String value, ZAddParams params) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zadd(key, score,value);
        jedis.close();
        return result;
    }

    public Long zsetAdd(String key, Map<String, Double> valueScores) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zadd(key, valueScores);
        jedis.close();
        return result;
    }

    public Long zsetAdd(String key, Map<String, Double> valueScores, ZAddParams params) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zadd(key, valueScores,params);
        jedis.close();
        return result;
    }

    public Set<String> zsetRange(String key, long start, long end) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.zrange(key, start,end);
        jedis.close();
        return result;
    }

    public Long zsetDelete(String key, String... values) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zrem(key, values);
        jedis.close();
        return result;
    }

    public Double zsetIncreaseBy(String key, double increment, String value) {
        Jedis jedis = jedisPool.getResource();
        Double result = jedis.zincrby(key, increment, value);
        jedis.close();
        return result;
    }

    public Double zsetIncreaseBy(String key, double increment, String value, ZIncrByParams params) {
        Jedis jedis = jedisPool.getResource();
        Double result = jedis.zincrby(key, increment,value, params);
        jedis.close();
        return result;
    }

    public Long zsetIndexOf(String key, String value) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zrank(key,value);
        jedis.close();
        return result;
    }

    public Long zsetReverseIndexOf(String key, String value) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zrevrank(key,value);
        jedis.close();
        return result;
    }

    public Set<String> zsetReverseRange(String key, long start, long end) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.zrevrange(key, start, end);
        jedis.close();
        return result;
    }

    public Set<Tuple> zsetRangeWithScores(String key, long start, long end) {
        Jedis jedis = jedisPool.getResource();
        Set<Tuple> result = jedis.zrangeWithScores(key, start,end);
        jedis.close();
        return result;
    }

    public Set<Tuple> zsetReverseRangeWithScores(String key, long start, long end) {
        Jedis jedis = jedisPool.getResource();
        Set<Tuple> result = jedis.zrevrangeWithScores(key, start,end);
        jedis.close();
        return result;
    }

    public Long zsetLength(String key) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zcard(key);
        jedis.close();
        return result;
    }

    public Double zsetScore(String key, String value) {
        Jedis jedis = jedisPool.getResource();
        Double result = jedis.zscore(key,value);
        jedis.close();
        return result;
    }


    public List<String> sort(String key) {
        Jedis jedis = jedisPool.getResource();
        List<String> result = jedis.sort(key);
        jedis.close();
        return result;
    }

    public List<String> sort(String key, SortingParams sortingParameters) {
        Jedis jedis = jedisPool.getResource();
        List<String> result = jedis.sort(key,sortingParameters);
        jedis.close();
        return result;
    }

    public Long sortStore(String key, String destKey) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.sort(key,destKey);
        jedis.close();
        return result;
    }

    public Long sortStore(String key, SortingParams sortingParameters, String destKey) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.sort(key,sortingParameters, destKey);
        jedis.close();
        return result;
    }

    public Set<String> zsetRangeByScore(String key, double min, double max) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.zrangeByScore(key,min, max);
        jedis.close();
        return result;
    }

    public Set<String> zsetRangeByScore(String key, String min, String max) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.zrangeByScore(key,min, max);
        jedis.close();
        return result;
    }

    public Set<String> zsetRangeByScore(String key, double min, double max, int offset, int count) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.zrangeByScore(key,min, max,offset, count);
        jedis.close();
        return result;
    }

    public Set<String> zsetRangeByScore(String key, String min, String max, int offset, int count) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.zrangeByScore(key,min, max, offset, count);
        jedis.close();
        return result;
    }

    public Set<Tuple> zsetRangeByScoreWithScores(String key, double min, double max) {
        Jedis jedis = jedisPool.getResource();
        Set<Tuple> result = jedis.zrangeByScoreWithScores(key,min, max);
        jedis.close();
        return result;
    }

    public Set<Tuple> zsetRangeByScoreWithScores(String key, String min, String max) {
        Jedis jedis = jedisPool.getResource();
        Set<Tuple> result = jedis.zrangeByScoreWithScores(key,min, max);
        jedis.close();
        return result;
    }

    public Set<Tuple> zsetRangeByScoreWithScores(String key, double min, double max, int offset, int count) {
        Jedis jedis = jedisPool.getResource();
        Set<Tuple> result = jedis.zrangeByScoreWithScores(key,min, max, offset, count);
        jedis.close();
        return result;
    }

    public Set<Tuple> zsetRangeByScoreWithScores(String key, String min, String max, int offset, int count) {
        Jedis jedis = jedisPool.getResource();
        Set<Tuple> result = jedis.zrangeByScoreWithScores(key,min, max, offset, count);
        jedis.close();
        return result;
    }


    public Set<String> zsetReverseRangeByScore(String key, double min, double max) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.zrangeByScore(key,min, max);
        jedis.close();
        return result;
    }

    public Set<String> zsetReverseRangeByScore(String key, String min, String max) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.zrangeByScore(key,min, max);
        jedis.close();
        return result;
    }

    public Set<String> zsetReverseRangeByScore(String key, double min, double max, int offset, int count) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.zrangeByScore(key,min, max,offset, count);
        jedis.close();
        return result;
    }

    public Set<String> zsetReverseRangeByScore(String key, String min, String max, int offset, int count) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.zrangeByScore(key,min, max, offset, count);
        jedis.close();
        return result;
    }

    public Set<Tuple> zsetReverseRangeByScoreWithScores(String key, double min, double max) {
        Jedis jedis = jedisPool.getResource();
        Set<Tuple> result = jedis.zrangeByScoreWithScores(key,min, max);
        jedis.close();
        return result;
    }

    public Set<Tuple> zsetReverseRangeByScoreWithScores(String key, String min, String max) {
        Jedis jedis = jedisPool.getResource();
        Set<Tuple> result = jedis.zrangeByScoreWithScores(key,min, max);
        jedis.close();
        return result;
    }

    public Set<Tuple> zsetReverseRangeByScoreWithScores(String key, double min, double max, int offset, int count) {
        Jedis jedis = jedisPool.getResource();
        Set<Tuple> result = jedis.zrangeByScoreWithScores(key,min, max, offset, count);
        jedis.close();
        return result;
    }

    public Set<Tuple> zsetReverseRangeByScoreWithScores(String key, String min, String max, int offset, int count) {
        Jedis jedis = jedisPool.getResource();
        Set<Tuple> result = jedis.zrangeByScoreWithScores(key,min, max, offset, count);
        jedis.close();
        return result;
    }

    public Long zsetDeleteRangeByIndex(String key, long start, long stop) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zremrangeByRank(key,start, stop);
        jedis.close();
        return result;
    }

    public Long zsetDeleteRangeByScore(String key, double min, double max) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zremrangeByScore(key, min, max);
        jedis.close();
        return result;
    }

    public Long zsetDeleteRangeByScore(String key, String min, String max) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zremrangeByScore(key, min, max);
        jedis.close();
        return result;
    }

    public Long zsetUnionStore(String destKey, String... keys) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zunionstore(destKey,keys);
        jedis.close();
        return result;
    }

    public Long zsetUnionStore(String destKey, ZParams params, String... keys) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zunionstore(destKey,params, keys);
        jedis.close();
        return result;
    }

    public Long zsetInterStore(String destKey, String... keys) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zinterstore(destKey, keys);
        jedis.close();
        return result;
    }

    public Long zsetInterStore(String destKey, ZParams params, String... keys) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zinterstore(destKey,params, keys);
        jedis.close();
        return result;
    }

    public Long zsetLexLength(String key, String min, String max) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zlexcount(key, min, max);
        jedis.close();
        return result;
    }

    public Set<String> zsetRangeByLex(String key, String min, String max) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.zrangeByLex(key, min, max);
        jedis.close();
        return result;
    }

    public Set<String> zsetRangeByLex(String key, String min, String max, int offset, int count) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.zrangeByLex(key, min, max, offset, count);
        jedis.close();
        return result;
    }

    public Set<String> zsetReverseRangeByLex(String key, String min, String max) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.zrevrangeByLex(key, min, max);
        jedis.close();
        return result;
    }

    public Set<String> zsetReverseRangeByLex(String key, String min, String max, int offset, int count) {
        Jedis jedis = jedisPool.getResource();
        Set<String> result = jedis.zrevrangeByLex(key, min, max, offset, count);
        jedis.close();
        return result;
    }

    public Long zsetDeleteRangeByLex(String key, String min, String max) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zremrangeByLex(key, min, max);
        jedis.close();
        return result;
    }

    // ==================================== geo =================================

    public Long geoAdd(String key, double longitude, double latitude, String value) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.geoadd(key, longitude, latitude, value);
        jedis.close();
        return result;
    }

    public Long geoAdd(String key, Map<String, GeoCoordinate> memberCoordinateMap) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.geoadd(key, memberCoordinateMap);
        jedis.close();
        return result;
    }

    public Double geoDistant(String key, String source, String target) {
        Jedis jedis = jedisPool.getResource();
        Double result = jedis.geodist(key, source, target);
        jedis.close();
        return result;
    }

    public Double geoDistant(String key, String source, String target, GeoUnit unit) {
        Jedis jedis = jedisPool.getResource();
        Double result = jedis.geodist(key, source, target, unit);
        jedis.close();
        return result;
    }

    public List<String> geoHash(String key, String... values) {
        Jedis jedis = jedisPool.getResource();
        List<String> result = jedis.geohash(key, values);
        jedis.close();
        return result;
    }

    public List<GeoCoordinate> geoPos(String key, String... values) {
        Jedis jedis = jedisPool.getResource();
        List<GeoCoordinate> result = jedis.geopos(key, values);
        jedis.close();
        return result;
    }

    public List<GeoRadiusResponse> geoRadius(String key, double longitude, double latitude,
                                             double radius, GeoUnit unit) {
        Jedis jedis = jedisPool.getResource();
        List<GeoRadiusResponse> result = jedis.georadius(key, longitude, latitude, radius, unit);
        jedis.close();
        return result;
    }

    public List<GeoRadiusResponse> geoRadiusReadonly(String key, double longitude, double latitude,
                                                     double radius, GeoUnit unit) {
        Jedis jedis = jedisPool.getResource();
        List<GeoRadiusResponse> result = jedis.georadiusReadonly(key, longitude, latitude, radius, unit);
        jedis.close();
        return result;
    }

    public List<GeoRadiusResponse> geoRadius(String key, double longitude, double latitude,
                                             double radius, GeoUnit unit, GeoRadiusParam param) {
        Jedis jedis = jedisPool.getResource();
        List<GeoRadiusResponse> result = jedis.georadius(key, longitude, latitude, radius, unit, param);
        jedis.close();
        return result;
    }

    public List<GeoRadiusResponse> geoRadiusReadonly(String key, double longitude, double latitude,
                                                     double radius, GeoUnit unit, GeoRadiusParam param) {
        Jedis jedis = jedisPool.getResource();
        List<GeoRadiusResponse> result = jedis.georadiusReadonly(key, longitude, latitude, radius, unit, param);
        jedis.close();
        return result;
    }

    public List<GeoRadiusResponse> geoRadiusByMember(String key, String member, double radius, GeoUnit unit) {
        Jedis jedis = jedisPool.getResource();
        List<GeoRadiusResponse> result = jedis.georadiusByMember(key, member, radius, unit);
        jedis.close();
        return result;
    }

    public List<GeoRadiusResponse> geoRadiusByMemberReadonly(String key, String member, double radius, GeoUnit unit) {
        Jedis jedis = jedisPool.getResource();
        List<GeoRadiusResponse> result = jedis.georadiusByMemberReadonly(key, member, radius, unit);
        jedis.close();
        return result;
    }

    public List<GeoRadiusResponse> geoRadiusByMember(String key, String member,
                                                     double radius, GeoUnit unit, GeoRadiusParam param) {
        Jedis jedis = jedisPool.getResource();
        List<GeoRadiusResponse> result = jedis.georadiusByMember(key, member, radius, unit, param);
        jedis.close();
        return result;
    }

    public List<GeoRadiusResponse> geoRadiusByMemberReadonly(String key, String member,
                                                             double radius, GeoUnit unit, GeoRadiusParam param) {
        Jedis jedis = jedisPool.getResource();
        List<GeoRadiusResponse> result = jedis.georadiusByMemberReadonly(key, member, radius, unit, param);
        jedis.close();
        return result;
    }


    // ==================================== other =========================

    public void subscribe(JedisPubSub jedisPubSub, String... channels) {
        Jedis jedis = jedisPool.getResource();
        jedis.subscribe(jedisPubSub, channels);
        jedis.close();
    }

    public Long publish(String channel, String message) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.publish(channel, message);
        jedis.close();
        return result;
    }

    public void patternSubscribe(JedisPubSub jedisPubSub, String... patterns) {
        Jedis jedis = jedisPool.getResource();
        jedis.psubscribe(jedisPubSub, patterns);
        jedis.close();
    }

    public Object eval(String script, List<String> keys, List<String> args) {
        Jedis jedis = jedisPool.getResource();
        Object result = jedis.eval(script, keys, args);
        jedis.close();
        return result;
    }

    public Object eval(String script, int keyCount, String... params) {
        Jedis jedis = jedisPool.getResource();
        Object result = jedis.eval(script, keyCount, params);
        jedis.close();
        return result;
    }

    public Object eval(String script) {
        return this.eval(script, 0);
    }

    public Object evalSha(String sha1) {
        return this.evalSha(sha1, 0);
    }

    public Object evalSha(String sha1, List<String> keys, List<String> args) {
        Jedis jedis = jedisPool.getResource();
        Object result = jedis.evalsha(sha1, keys, args);
        jedis.close();
        return result;
    }

    public Object evalSha(String sha1, int keyCount, String... params) {
        Jedis jedis = jedisPool.getResource();
        Object result = jedis.evalsha(sha1, keyCount, params);
        jedis.close();
        return result;
    }

    public Boolean scriptExists(String sha1) {
        Jedis jedis = jedisPool.getResource();
        Boolean result = jedis.scriptExists(sha1);
        jedis.close();
        return result;
    }

    public List<Boolean> scriptExists(String... sha1) {
        Jedis jedis = jedisPool.getResource();
        List<Boolean> result = jedis.scriptExists(sha1);
        jedis.close();
        return result;
    }

    public String scriptLoad(String script) {
        Jedis jedis = jedisPool.getResource();
        String result = jedis.scriptLoad(script);
        jedis.close();
        return result;
    }


    // ==================================== self method =========================


    /**
     * Watch keys in transaction. if watched key's value has been changed when execute command exec().
     * All the command after watch will fail.
     *
     *
     * @param keys watch keys
     * @return status
     *
     */
    public String watch(String... keys){
        Jedis jedis = jedisPool.getResource();
        String result = jedis.watch(keys);
        jedis.close();
        return result;
    }

    /**
     * Move the specified key from the currently selected DB to the specified destination DB. Note
     * that this command returns 1 only if the key was successfully moved, and 0 if the target key was
     * already there or if the source key was not found at all, so it is possible to use MOVE as a
     * locking primitive.
     *
     * @param key key to be moved
     * @param dbIndex target dbIndex
     * @return Integer reply
     *
     *
     */
    public Long move(String key, int dbIndex) {
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.move(key, dbIndex);
        jedis.close();
        return result;
    }


    /**
     *  {@link #listBlockingLeftPop(int, String...)}
     *
     * @param keys keys
     * @return pop values
     */
    public List<String> listBlockingRightPop(String... keys) {
        Jedis jedis = jedisPool.getResource();
        List<String> result = jedis.brpop(keys);
        jedis.close();
        return result;
    }

    /**
     *  {@link #listBlockingLeftPop(int, String...)}
     *
     * @param keys pop keys
     * @return pop values
     */
    public List<String> listBlockingLeftPop(String... keys)
    {
        Jedis jedis = jedisPool.getResource();
        List<String> result = jedis.blpop(keys);
        jedis.close();
        return result;
    }


    /**
     *  return count of set value witch score between min and max
     *
     * @param key set key
     * @param min min score
     * @param max max score
     * @return count number
     */
    public Long zsetCount(String key, double min, double max){
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zcount(key, min, max);
        jedis.close();
        return result;
    }

    /**
     *  return count of set value witch score between min and max
     *
     * @param key set key
     * @param min min score
     * @param max max score
     * @return count number
     */
    public Long zsetCount(String key, String min, String max){
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.zcount(key, min, max);
        jedis.close();
        return result;
    }


    /**
     * Return first bit position of value in key.
     * <code>SET mykey "\xff\xf0\x00" "OK"</code>
     * <code>BITPOS mykey 0 (integer) 12</code>
     * <code>SET mykey "\x00\xff\xf0" "OK"</code>
     * <code>BITPOS mykey 1 0 (integer) 8</code>
     *
     * @param key key
     * @param value bit value
     * @return first position
     */
    public Long bitPos(String key, boolean value){
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.bitpos(key, value);
        jedis.close();
        return result;
    }


    /**
     * Return first bit position of value in key.
     * <code>SET mykey "\xff\xf0\x00" "OK"</code>
     * <code>BITPOS mykey 0 (integer) 12</code>
     * <code>SET mykey "\x00\xff\xf0" "OK"</code>
     * <code>BITPOS mykey 1 0 (integer) 8</code>
     *
     * @param key key
     * @param value bit value
     * @param params {@link BitPosParams}
     * @return first position
     */
    public Long bitPos(String key, boolean value, BitPosParams params){
        Jedis jedis = jedisPool.getResource();
        Long result = jedis.bitpos(key, value, params);
        jedis.close();
        return result;
    }


    /**
     * Retrieve the configuration of a running Redis server. Not all the configuration parameters are
     * supported.
     * <p>
     * CONFIG GET returns the current configuration parameters. This sub command only accepts a single
     * argument, that is glob style pattern. All the configuration parameters matching this parameter
     * are reported as a list of key-value pairs.
     * <p>
     * <b>Example:</b>
     *
     * <pre>
     * $ redis-cli config get '*'
     * 1. "dbfilename"
     * 2. "dump.rdb"
     * 3. "requirepass"
     * 4. (nil)
     * 5. "masterauth"
     * 6. (nil)
     * 7. "maxmemory"
     * 8. "0\n"
     * 9. "appendfsync"
     * 10. "everysec"
     * 11. "save"
     * 12. "3600 1 300 100 60 10000"
     *
     * $ redis-cli config get 'm*'
     * 1. "masterauth"
     * 2. (nil)
     * 3. "maxmemory"
     * 4. "0\n"
     * </pre>
     * @param pattern
     * @return Bulk reply.
     */
    public List<String> configGet(final String pattern) {
        Jedis jedis = jedisPool.getResource();
        List<String> result = jedis.configGet(pattern);
        jedis.close();
        return result;
    }


    /**
     * <pre>
     * redis 127.0.0.1:26381&gt; sentinel masters
     * 1)  1) "name"
     *     2) "mymaster"
     *     3) "ip"
     *     4) "127.0.0.1"
     *     5) "port"
     *     6) "6379"
     *     7) "runid"
     *     8) "93d4d4e6e9c06d0eea36e27f31924ac26576081d"
     *     9) "flags"
     *    10) "master"
     *    11) "pending-commands"
     *    12) "0"
     *    13) "last-ok-ping-reply"
     *    14) "423"
     *    15) "last-ping-reply"
     *    16) "423"
     *    17) "info-refresh"
     *    18) "6107"
     *    19) "num-slaves"
     *    20) "1"
     *    21) "num-other-sentinels"
     *    22) "2"
     *    23) "quorum"
     *    24) "2"
     *
     * </pre>
     * @return result
     */
    public List<Map<String, String>> sentinelMasters(){
        Jedis jedis = jedisPool.getResource();
        List<Map<String, String>> result = jedis.sentinelMasters();
        jedis.close();
        return result;
    }



}
