package cn.ipokerface.redis;

import redis.clients.jedis.*;
import redis.clients.jedis.params.GeoRadiusParam;
import redis.clients.jedis.params.SetParams;
import redis.clients.jedis.params.ZAddParams;
import redis.clients.jedis.params.ZIncrByParams;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by       PokerFace
 * Create Date      2019-11-27.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:     Common interface of operation Redis-server
 *
 */
public interface JedisClient {


    //
    public static final String RESULT_OK                   = "OK";

    public static final Long RESULT_WHEN_EXCEPTION         = 0L;

    public static final long DEFAULT_TIMEOUT               = 15000L;

    //
    public static final String SET_IF_EXIST                 = "XX";

    //
    public static final String SET_IF_NOT_EXIST             = "NX";

    //
    public static final String SET_EXPIRE_TIME_SECOND       = "EX";

    //
    public static final String SET_EXPIRE_TIME_MILLISECOND  = "PX";



    /**
     *  Set a value with key with no expire times.
     *   <code>set("key","value")</code>
     *    The string can't be longer than 1073741824 bytes (1GB).
     *  And this will return {@link JedisClient#RESULT_OK} if succeed.
     *
     * @param key   key
     * @param value value
     * @return redis-server reply
     */
    public String set(String key, String value);



    /**
     *  Set value with expire time .
     *  {@link #set(String, String, SetParams)}
     *  <code>set(key, value, SetParams.setParams().px(expireInMillis))</code>
     *  And this will return {@link JedisClient#RESULT_OK} if succeed.
     *
     * @param key   key
     * @param value value
     * @param expireInMillis expire time in milliseconds
     * @return redis-server reply
     */
    public String set(String key, String value, long expireInMillis);


    /**
     *  Set value with some options like:
     *  <ul>
     *      <li>NX: set key only if key not exist</li>
     *      <li>XX: set key only if key exist</li>
     *      <li>PX: set key expire time in milliseconds.</li>
     *      <li>EX: set key expire time in seconds</li>
     *  </ul>
     *  usage: <code>SetParams param = SetParams.setParams().nx().px(123L);</code>
     *  And this will return {@link JedisClient#RESULT_OK} if succeed.
     *
     * @param key   key
     * @param value value
     * @param setParams params
     *
     * @return server reply
     */
    public String set(String key, String value, SetParams setParams);


    /**
     * set key values into a map structure.
     *
     * @param keyValues key value pairs
     * @return status code like OK.
     */
    public String multiSet(String... keyValues);


    /**
     * set key values info a map structure only if all keys not exist in map.
     *  returns:
     *  <ul>
     *      <li>1 if the all the keys were set </li>
     *      <li>0 if no key was set (at least one key already existed)</li>
     *  </ul>
     *
     * @param keyValues
     * @return Integer reply
     */
    public Long multiSetNx(String... keyValues);

    /**
     *  Get value from redis with key.
     *  return null if key not exist.
     *
     * @param key key
     *
     * @return value of key
     */
    public String get(String key);


    /**
     * Get the values of all the specified keys. If one or more keys don't exist or is not of type
     * String, a 'nil' value is returned instead of the value of the specified key, but the operation
     * never fails.
     *
     * @param keys keys to get
     * @return value list of keys
     */
    public List<String> get(String... keys);


    /**
     * Test if the specified keys exist. The command returns the number of keys exist.
     *
     * @param keys keys to be test
     * @return count exist
     */
    public Long exists(String... keys);


    /**
     * if key exist or not . Note that even keys set with an empty string as value will return true.
     *
     * @param key key to be test
     * @return if exist or nor.
     */
    public Boolean exists(String key);


    /**
     * expire key in time of seconds <code>CMD: TTL seconds</code>
     * this key will be removed at time after now in seconds.
     * return :
     *  1: the timeout was set.
     *  0: the timeout was not set since the key already has an associated timeout
     *      (this may happen only in Redis versions &lt; 2.1.3, Redis &gt;= 2.1.3 will
     *      happily update the timeout), or the key does not exist.
     *
     * @param key key
     * @param seconds time in seconds
     * @return Integer reply,
     *
     */
    public Long expire(String key, int seconds);


    /**
     * expire key in time of milliseconds <code>CMD: expire key time</code>
     * this key will be removed at time after now in seconds.
     * return :
     *  1: the timeout was set.
     *  0: the timeout was not set since the key already has an associated timeout
     *      (this may happen only in Redis versions &lt; 2.1.3, Redis &gt;= 2.1.3 will
     *      happily update the timeout), or the key does not exist.
     *
     * @param key key
     * @param timeMillis time in milliseconds
     * @return Integer reply
     *
     */
    public Long expire(String key, long timeMillis);

    /**
     * expire key at time of unix timestamp. it takes an absolute one in the form of a UNIX timestamp (Number of
     * seconds elapsed since 1 Gen 1970).
     * return :
     *  1: the timeout was set.
     *  0: the timeout was not set since the key already has an associated timeout
     *      (this may happen only in Redis versions &lt; 2.1.3, Redis &gt;= 2.1.3 will
     *      happily update the timeout), or the key does not exist.
     *
     * @param key key
     * @param unixTimestamp timestamp
     * @return Integer reply
     */
    public Long expireAt(String key, long unixTimestamp);


    /**
     *  Undo a {@link #expire(String, int) expire} at turning the expire key into a normal key.
     *
     * @param key key
     * @return Integer reply, specifically: 1: the key is now persist. 0: the key is not persist (only
     *     happens when key not set).
     */
    public Long persist(String key);


    /**
     * Get the remaining time to live in seconds of a key.
     * returns the remaining time to live in seconds of a key that has an EXPIRE.
     * In Redis 2.6 or older:
     *      if the Key does not exists or does not have an associated expire, -1 is returned.
     * In Redis 2.8 or newer:
     *      if the Key does not have an associated expire, -1 is returned
     *      if the Key does not exists, -2 is returned.
     *
     * @param key
     * @return
     */
    public Long ttl(String key);


    /**
     * print string in redis
     *
     * @param string string to echo
     * @return string to return
     */
    public String echo(String string);


    /**
     * Delete from redis with keys... .
     * This command is very similar to DEL: it removes the specified keys. Just like DEL a key is
     * ignored if it does not exist. However the command performs the actual memory reclaiming in a
     * different thread, so it is not blocking, while DEL is. This is where the command name comes
     * from: the command just unlinks the keys from the keyspace. The actual removal will happen later
     * asynchronously.
     *  using <code>jedis.unlink(keys)</code> <code>CMD: unlink ...</code>
     *
     *  more. see {@link redis.clients.jedis.Jedis}
     *          or {@link redis.clients.jedis.JedisCluster}
     *
     * @param keys keys to be removed
     * @return number of keys been removed
     */
    public Long delete(String... keys);


    /**
     * get the type of this key. The type can be one of "none", "string", "list", "set".
     * "none" is returned if the key does not exist.
     *  returns:
     *  <ul>
     *      <li>"none"  : if the key does not exist</li>
     *      <li>"string": if the key contains a String value</li>
     *      <li>"list"  : if the key contains a List value</li>
     *      <li>"set"   : if the key contains a Set value</li>
     *      <li>"zset"  : if the key contains a Sorted Set value</li>
     *      <li>"hash"  : if the key contains a Hash value</li>
     *  </ul>
     *
     * @param key key
     * @return type of key
     */
    public String type(String key);


    /**
     * Get a set of keys that exist in redis match with pattern. This may course some problem
     * if volume of data is huge. so do not use this api at any time.
     *
     * @param pattern pattern of keys
     * @return keys
     */
    public Set<String> keys(String pattern);


    /**
     * Alters the last access time of a key(s). A key is ignored if it does not exist.
     *
     * @param keys keys to be touch
     * @return count of keys been touched.
     */
    public Long touch(String... keys);



    /**
     * atomic set this value and return the old value command. Set key to the string
     *  value and return the old value stored at key. The string can't be longer than 1073741824 bytes
     *   (1 GB).
     *
     * @param key key
     * @param value new value of key
     * @return old value of key
     */
    public String getSet(String key, String value);


    /**
     * Increment the number stored at key by one. If the key does not exist or contains a value of a
     * wrong type, set the key to the value of "0" before to perform the increment operation.
     * <p>
     *  INCR commands are limited to 64 bit signed integers.
     * <p>
     * Note: this is actually a string operation, that is, in Redis there are not "integer" types.
     * Simply the string stored at the key is parsed as a base 10 64 bit signed integer, incremented,
     * and then converted back as a string.
     *
     * @param key key
     * @return new value of key.
     */
    public Long increase(String key);


    /**
     * Increment the number stored at key by increment count. If the key does not exist or contains a value of a
     * wrong type, set the key to the value of "0" before to perform the increment operation.
     * <p>
     *  INCR commands are limited to 64 bit signed integers.
     * <p>
     * Note: this is actually a string operation, that is, in Redis there are not "integer" types.
     * Simply the string stored at the key is parsed as a base 10 64 bit signed integer, incremented,
     * and then converted back as a string.
     *
     * @param key key
     * @param increment increment count
     * @return new value of key
     */
    public Long increaseBy(String key, long increment);


    /**
     * Decrement the number stored at key by one. If the key does not exist or contains a value of a
     * wrong type, set the key to the value of "0" before to perform the Decrement operation.
     * <p>
     *  INCR commands are limited to 64 bit signed integers.
     * <p>
     * Note: this is actually a string operation, that is, in Redis there are not "integer" types.
     * Simply the string stored at the key is parsed as a base 10 64 bit signed integer, decremented,
     * and then converted back as a string.
     *
     * @param key key
     * @return new value of key.
     */
    public Long decrease(String key);


    /**
     * Decrement the number stored at key by decrement count. If the key does not exist or contains a value of a
     * wrong type, set the key to the value of "0" before to perform the decrement operation.
     * <p>
     *  INCR commands are limited to 64 bit signed integers.
     * <p>
     * Note: this is actually a string operation, that is, in Redis there are not "integer" types.
     * Simply the string stored at the key is parsed as a base 10 64 bit signed integer, decremented,
     * and then converted back as a string.
     *
     * @param key key
     * @param decrement increment count
     * @return new value of key
     */
    public Long decreaseBy(String key, long decrement);


    /**
     * this is actually a string operation, that is, in Redis there are not "double" types.
     * Simply the string stored at the key is parsed as a base double precision floating point value,
     * incremented, and then converted back as a string. There is no DECRYBYFLOAT but providing a
     * negative value will work as expected.
     *
     *
     * @param key key
     * @param increment increment
     * @return new value
     */
    public Double increaseByFloat(String key, float increment);


    /**
     * If the key already exists and is a string, this command appends the provided value at the end
     * of the string. If the key does not exist it is created and set as an empty string, so APPEND
     * will be very similar to SET in this special case.
     *
     * @param key key
     * @param value value
     * @return Integer reply, specifically the total length of the string after the append operation.
     */
    public Long append(String key, String value);


    /**
     * Return a subset of the string from offset start to offset end (both offsets are inclusive).
     * Negative offsets can be used in order to provide an offset starting from the end of the string.
     * So -1 means the last char, -2 the penultimate and so forth.
     * <p>
     * The function handles out of range requests without raising an error, but just limiting the
     * resulting range to the actual length of the string.
     * </p>
     *
     * @param key key
     * @param start start index
     * @param end end index
     * @return sub string of key
     */
    public String subString(String key, int start, int end);


    /**
     * Sets or clears the bit at offset in the string value stored at key
     * <code>
     *     set value:1 00000000000
     *      setbit value:1 2 1
     * </code>
     * after set value:1 00100000000
     *
     * @param key key
     * @param offset offset
     * @param value bit value
     * @return if success
     */
    public Boolean setBit(String key, long offset, boolean value);


    /**
     * {@link #setBit(String, long, String)}
     *
     * @param key key
     * @param offset offset
     * @param value bit value
     * @return if success
     */
    public Boolean setBit(String key, long offset, String value);


    /**
     *  check if the offset of key is positive.
     * {@link #setBit(String, long, String)}
     *
     * @param key key
     * @param offset offset
     * @return if positive.
     */
    public Boolean getBit(String key, long offset);


    public Long bigCount(String key);


    public Long bitCount(String key, long start, long end);


    public Long bitOp(BitOP op, String destKey, String... srcKeys);


    public List<Long> bitField(String key, String...arguments);

    /**
     * cover the string value of key of new value that offset to key.
     * "Hello News!"
     * <code>set 4 "FOO"</code>
     * become<code>"Hello FOO"</code>
     *
     * Do nothing if key not exist.
     *
     * @param key key
     * @param offset offset to cover
     * @param value new value
     * @return new length of string.
     */
    public Long setRange(String key,long offset, String value);


    /**
     *  get the substring of key that between offset and endOffset.
     *
     * @param key key
     * @param offset start offset of string
     * @param endOffset end offset of string
     * @return substring.
     */
    public String getRange(String key, long offset, long endOffset);

    // ================================   Hash Map ======================================


    /**
     * Set the specified hash map field to the specified value.
     *  returns:
     *  <ul>
     *      <li>0: If the field already exists, and the HSET just produced an update of the value</li>
     *      <li>1: if a new field is created</li>
     *  </ul>
     *
     * @param key key to hold this hash map
     * @param field field name
     * @param value value
     * @return type of result
     */
    public Long mapSet(String key, String field, String value);


    /**
     * Set the specified hash map field to the specified value.
     *  <code>CMD: HMSET key values...</code>
     *  returns:
     *  <ul>
     *      <li>0: If the field already exists, and the HSET just produced an update of the value</li>
     *      <li>1: if a new field is created</li>
     *  </ul>
     *
     * @param key key
     * @param map value map
     * @return type of result
     */
    public Long mapSet(String key, Map<String,String> map);


    /**
     *
     *   Set the specified hash field to the specified value if the field not exists.
     * returns:
     *  <ul>
     *      <li>0: If the field already exists, and the HSET just produced an update of the value</li>
     *      <li>1: if a new field is created</li>
     *  </ul>
     *
     * @param key key
     * @param field field
     * @param value value
     * @return reply
     */
    public Long mapSetNx(String key, String field, String value);


    /**
     * Increment the number stored at field in the hash at key by value. If key does not exist, a new
     *  key holding a hash is created. If field does not exist or holds a string, the value is set to 0
     *  before applying the operation. Since the value argument is signed you can use this command to
     *  perform both increments and decrements.
     *
     *  reply The new value at field after the increment operation.
     *
     * @param key key
     * @param field field
     * @param value increment.
     * @return new value of field
     */
    public Long mapIncreaseBy(String key, String field, long value);


    /**
     * Increment the number stored at field in the hash at key by a double precision floating point
     * value. If key does not exist, a new key holding a hash is created. If field does not exist or
     * holds a string, the value is set to 0 before applying the operation. Since the value argument
     * is signed you can use this command to perform both increments and decrements.
     *
     * Double precision floating point reply The new value at field after the increment
     *    operation.
     *
     * @param key hash key
     * @param field field
     * @param value float increment
     * @return new value of field.
     */
    public Double mapIncreaseByFloat(String key, String field, float value);

    /**
     *  If key holds a hash, retrieve the value associated to the specified field.
     *  If the field is not found or the key does not exist, a special 'nil' value is returned.
     *
     * @param key key
     * @param field field
     * @return value
     */
    public String mapGet(String key, String field);


    /**
     * Retrieve the values associated to the specified fields.
     * If some of the specified fields do not exist, nil values are returned. Non existing keys are
     * considered like empty hashes.
     *   Reply specifically a list of all the values associated with the specified
     *         fields, in the same order of the request.
     *
     * @param key key
     * @param fields fields to request
     * @return reply list
     */
    public List<String> mapMultiGet(String key,String... fields);


    /**
     *  Get the length of a hash key.
     *  The number of entries (fields) contained in the hash stored at key. If the specified
     *   key does not exist, 0 is returned assuming an empty hash.
     *
     * @param key hash key
     * @return number of entries
     */
    public Long mapLength(String key);


    /**
     * Test for existence of a specified field in a hash.
     * Return true if the hash stored at key contains the specified field. Return false if the key is
     *         not found or the field is not present.
     *
     * @param key key
     * @param field field
     * @return if exist
     */
    public Boolean mapExist(String key, String field);


    /**
     * Return all the fields in a hash.
     *
     * @param key hash key
     * @return key set
     */
    public Set<String> mapKeys(String key);

    /**
     * Return all the values in a hash.
     *
     * @param key hash key
     * @return value list
     */
    public List<String> mapValues(String key);


    /**
     * Return all the fields and associated values in a hash.
     *
     *
     * @param key hash key
     * @return hash content.
     */
    public Map<String, String> mapAll(String key);


    /**
     * Remove the specified field from an hash stored at key.
     * returns:
     *  <ul>
     *      <li>0: Negative if operation.</li>
     *      <li>1: If the field was present in the hash it is deleted</li>
     *  </ul>
     *
     * @param key key
     * @param fields fields to be removed
     * @return reply
     */
    public Long mapDelete(String key, String... fields);


    // ================================ List ======================================

    /**
     *  Add the string value to the TAIL of the list. {@link #listRightPush(String, String...)}
     *  If the key does not exist an empty list is created just before the append operation.
     *  If the key exists but is not a List an error is returned.
     *
     *
     * @param key key
     * @param values values to be push
     * @return list length.
     */
    public Long listPush(String key, String... values);


    /**
     *  Insert a value into list of where with anchor.
     *  <code>ListPosition.BEFORE "FOO" will display as "VALUE" "FOO"</code>
     *  <code>ListPosition.AFTER "FOO" will display as "FOO" "VALUE"</code>
     *  if key of list or anchor does not exist . do nothing.
     *
     *  return total size of list after operation.
     *
     * @param key key of list
     * @param where position {@link ListPosition}
     * @param anchor anchor value
     * @param value value
     * @return
     */
    public Long listInsert(String key, ListPosition where, String anchor,String value);


    /**
     *  Add the string value to the TAIL of the list.
     *  If the key does not exist an empty list is created just before the append operation.
     *  If the key exists but is not a List an error is returned.
     *
     *
     * @param key key
     * @param values values to be push
     * @return list length.
     */
    public Long listRightPush(String key, String...values);

    /**
     *  Add the string value to the TAIL of the list.
     *  If the key does not exist it will not operate anything.
     *  If the key exists but is not a List an error is returned.
     *
     *
     * @param key key
     * @param values values to be push
     * @return list length.
     */
    public Long listRightPushExist(String key, String... values);


    /**
     *  Add the string value to the HEAD of the list.
     *  If the key does not exist an empty list is created just before the append operation.
     *  If the key exists but is not a List an error is returned.
     *
     *
     * @param key key
     * @param values values to be push
     * @return list length.
     */
    public Long listLeftPush(String key, String... values);


    /**
     *  Add the string value to the HEAD of the list.
     *  If the key does not exist it will not operate anything.
     *  If the key exists but is not a List an error is returned.
     *
     *
     * @param key key
     * @param values values to be push
     * @return list length.
     */
    public Long listLeftPushExist(String key, String... values);


    /**
     * Return the length of the list stored at the specified key.
     * returns:
     * <ul>
     *     <li>0: dose not exist or empty list</li>
     *     <li>If the value stored at key is not a list an error is returned.</li>
     * </ul>
     *
     * @param key list key
     * @return count of list
     */
    public Long listLength(String key);


    /**
     * Return the specified elements of the list stored at the specified key. Start and end are
     * zero-based indexes. 0 is the first element of the list (the list head), 1 the next element and
     * so on.
     * <p>
     * For example LRANGE foobar 0 2 will return the first three elements of the list.
     * <p>
     * start and end can also be negative numbers indicating offsets from the end of the list. For
     * example -1 is the last element of the list, -2 the penultimate element and so on.
     * <p>
     * <b>Consistency with range functions in various programming languages</b>
     * <p>
     * Note that if you have a list of numbers from 0 to 100, LRANGE 0 10 will return 11 elements,
     * that is, rightmost item is included. This may or may not be consistent with behavior of
     * range-related functions in your programming language of choice (think Ruby's Range.new,
     * Array#slice or Python's range() function).
     * <p>
     * LRANGE behavior is consistent with one of Tcl.
     * <p>
     * <b>Out-of-range indexes</b>
     * <p>
     * Indexes out of range will not produce an error: if start is over the end of the list, or start
     * &gt; end, an empty list is returned. If end is over the end of the list Redis will threat it
     * just like the last element of the list.
     * <p>
     *
     * @param key list key
     * @param start start index
     * @param end end index.
     * @return sub list of list
     */
    public List<String> listRange(String key, long start, long end);


    /**
     * Trim an existing list so that it will contain only the specified range of elements specified.
     * Start and end are zero-based indexes. 0 is the first element of the list (the list head), 1 the
     * next element and so on.
     * <p>
     * For example LTRIM foobar 0 2 will modify the list stored at foobar key so that only the first
     * three elements of the list will remain.
     * <p>
     * start and end can also be negative numbers indicating offsets from the end of the list. For
     * example -1 is the last element of the list, -2 the penultimate element and so on.
     * <p>
     * Indexes out of range will not produce an error: if start is over the end of the list, or start
     * &gt; end, an empty list is left as value. If end over the end of the list Redis will threat it
     * just like the last element of the list.
     * <p>
     * Hint: the obvious use of LTRIM is together with LPUSH/RPUSH. For example:
     * <p>
     * {@code lpush("mylist", "someelement"); ltrim("mylist", 0, 99); * }
     * <p>
     * The above two commands will push elements in the list taking care that the list will not grow
     * without limits. This is very useful when using Redis to store logs for example. It is important
     * to note that when used in this way LTRIM is an O(1) operation because in the average case just
     * one element is removed from the tail of the list.
     * <p>
     *
     * @param key list key
     * @param start start index
     * @param end end index.
     * @return Status code reply
     */
    public String listTrim(String key, long start, long end);


    /**
     * Return the specified element of the list stored at the specified key. 0 is the first element, 1
     * the second and so on. Negative indexes are supported, for example -1 is the last element, -2
     * the penultimate and so on.
     * <p>
     * If the value stored at key is not of list type an error is returned. If the index is out of
     * range a 'nil' reply is returned.
     * <p>
     * Note that even if the average time complexity is O(n) asking for the first or the last element
     * of the list is O(1).
     * <p>
     *
     * @param key list key
     * @param index index
     * @return element at index.
     */
    public String listIndex(String key, long index);


    /**
     * Set a new value as the element at index position of the List at key.
     * <p>
     * Out of range indexes will generate an error.
     * <p>
     * Similarly to other list commands accepting indexes, the index can be negative to access
     * elements starting from the end of the list. So -1 is the last element, -2 is the penultimate,
     * and so forth.
     * <p>
     *
     * @param key list key
     * @param index index
     * @param value value
     * @return Status code reply
     */
    public String listSet(String key, long index, String value);


    /**
     * Remove the first count occurrences of the value element from the list. If count is zero all the
     * elements are removed. If count is negative elements are removed from tail to head, instead to
     * go from head to tail that is the normal behaviour. So for example LREM with count -2 and hello
     * as value to remove against the list (a,b,c,hello,x,hello,hello) will leave the list
     * (a,b,c,hello,x). The number of removed elements is returned as an integer, see below for more
     * information about the returned value. Note that non existing keys are considered like empty
     * lists by LREM, so LREM against non existing keys will always return 0.
     *
     * @param key list key
     * @param count count to be removed
     * @param value value to be removed
     * @return The number of removed elements if the operation succeeded
     */
    public Long listRemove(String key, long count, String value);


    /**
     * Pop out the value just push to the list.
     * Atomically return and remove the first (LPOP) or last (RPOP) element of the list. For example
     * if the list contains the elements "a","b","c" RPOP will return "c" and the list will become
     * "a","b".
     *  {@link #listPush(String, String...)}
     *
     *  If the key does not exist or the list is already empty the special value 'nil' is returned.
     *
     * @param key list key
     * @return pop value
     */
    public String listPop(String key);


    /**
     * Pop out the TAIL value just push to the list.
     * Atomically return and remove the first (LPOP) or last (RPOP) element of the list. For example
     * if the list contains the elements "a","b","c" RPOP will return "c" and the list will become
     * "a","b".
     *  {@link #listPush(String, String...)}
     *
     *  If the key does not exist or the list is already empty the special value 'nil' is returned.
     *
     * @param key list key
     * @return pop value
     */
    public String listRightPop(String key);


    /**
     * Pop out the HEAD value just push to the list.
     * Atomically return and remove the first (LPOP) or last (RPOP) element of the list. For example
     * if the list contains the elements "a","b","c" RPOP will return "c" and the list will become
     * "a","b".
     *  {@link #listPush(String, String...)}
     *
     *  If the key does not exist or the list is already empty the special value 'nil' is returned.
     *
     * @param key list key
     * @return pop value
     */
    public String listLeftPop(String key);


    /**
     * Atomically return and remove the last (tail) element of the srckey list, and push the element
     * as the first (head) element of the dstkey list. For example if the source list contains the
     * elements "a","b","c" and the destination list contains the elements "foo","bar" after an
     * RPOPLPUSH command the content of the two lists will be "a","b" and "c","foo","bar".
     * <p>
     * If the key does not exist or the list is already empty the special value 'nil' is returned. If
     * the srckey and dstkey are the same the operation is equivalent to removing the last element
     * from the list and pushing it as first element of the list, so it's a "list rotation" command.
     * <p>
     *
     * @param srcKey source key
     * @param targetKey target key
     * @return pop value
     */
    public String listRightPopLeftPush(String srcKey, String targetKey);


    /**
     * Pop a value from a list, push it to another list and return it; or block until one is available
     *
     * @param srcKey source key
     * @param targetKey target key
     * @param timeout timeout count.
     * @return pop value
     */
    public String listBlockingRightPopLeftPush(String srcKey, String targetKey, int timeout);



    /**
     * BLPOP (and BRPOP) is a blocking list pop primitive. You can see this commands as blocking
     * versions of LPOP and RPOP able to block if the specified keys don't exist or contain empty
     * lists.
     * <p>
     * The following is a description of the exact semantic. We describe BLPOP but the two commands
     * are identical, the only difference is that BLPOP pops the element from the left (head) of the
     * list, and BRPOP pops from the right (tail).
     * <p>
     * <b>Non blocking behavior</b>
     * <p>
     * When BLPOP is called, if at least one of the specified keys contain a non empty list, an
     * element is popped from the head of the list and returned to the caller together with the name
     * of the key (BLPOP returns a two elements array, the first element is the key, the second the
     * popped value).
     * <p>
     * Keys are scanned from left to right, so for instance if you issue BLPOP list1 list2 list3 0
     * against a dataset where list1 does not exist but list2 and list3 contain non empty lists, BLPOP
     * guarantees to return an element from the list stored at list2 (since it is the first non empty
     * list starting from the left).
     * <p>
     * <b>Blocking behavior</b>
     * <p>
     * If none of the specified keys exist or contain non empty lists, BLPOP blocks until some other
     * client performs a LPUSH or an RPUSH operation against one of the lists.
     * <p>
     * Once new data is present on one of the lists, the client finally returns with the name of the
     * key unblocking it and the popped value.
     * <p>
     * When blocking, if a non-zero timeout is specified, the client will unblock returning a nil
     * special value if the specified amount of seconds passed without a push operation against at
     * least one of the specified keys.
     * <p>
     * The timeout argument is interpreted as an integer value. A timeout of zero means instead to
     * block forever.
     * <p>
     * <b>Multiple clients blocking for the same keys</b>
     * <p>
     * Multiple clients can block for the same key. They are put into a queue, so the first to be
     * served will be the one that started to wait earlier, in a first-blpopping first-served fashion.
     * <p>
     * <b>blocking POP inside a MULTI/EXEC transaction</b>
     * <p>
     * BLPOP and BRPOP can be used with pipelining (sending multiple commands and reading the replies
     * in batch), but it does not make sense to use BLPOP or BRPOP inside a MULTI/EXEC block (a Redis
     * transaction).
     * <p>
     * The behavior of BLPOP inside MULTI/EXEC when the list is empty is to return a multi-bulk nil
     * reply, exactly what happens when the timeout is reached. If you like science fiction, think at
     * it like if inside MULTI/EXEC the time will flow at infinite speed :)
     * <p>
     *
     * @param timeout time out
     * @param keys keys
     * @return pop values
     */
    public List<String> listBlockingLeftPop(int timeout, String... keys);





    /**
     *  {@link #listBlockingLeftPop(int, String...)}
     *
     * @param timeout timeout
     * @param keys keys
     * @return pop values
     */
    public List<String> listBlockingRightPop(int timeout, String... keys);



    // ================================   Set   ======================================

    /**
     * Add the specified member to the set value stored at key. If member is already a member of the
     * set no operation is performed. If key does not exist a new set with the specified member as
     * sole member is created. If the key exists but does not hold a set value an error is returned.
     * <ul>
     *     <li>1 if the new element was added</li>
     *     <li>0 if the element was already a member of the set</li>
     * </ul>
     *
     * @param key set key
     * @param values
     * @return
     */
    public Long setsAdd(String key, String... values);


    /**
     * Return all the members (elements) of the set value stored at key.
     *
     *
     * @param key set key
     * @return set values
     */
    public Set<String> setsValues(String key);


    /**
     * Remove the specified member from the set value stored at key. If member was not a member of the
     * set no operation is performed. If key does not hold a set value an error is returned.
     * returns:
     * <ul>
     *     <li>1 if the new element was removed</li>
     *     <li>0 if the new element was not a member of the set</li>
     * </ul>
     *
     * @param key set key
     * @param values values to be removed
     * @return reply
     */
    public Long setsDelete(String key, String... values);


    /**
     * Remove a random element from a Set returning it as return value. If the Set is empty or the key
     * does not exist, a nil object is returned.
     * {@link #setsPop(String, long)}
     *
     * @param key key
     * @return random value.
     */
    public String setsPop(String key);


    /**
     * Remove a random element from a Set returning it as return value. If the Set is empty or the key
     *   does not exist, a nil object is returned.
     *
     * @param key set key
     * @param count value count to be pop
     * @return removed values
     */
    public Set<String> setsPop(String key, long count);


    /**
     * Return the set cardinality (number of elements). If the key does not exist 0 is returned, like
     *    for empty sets.
     *
     * @param key key
     * @return count of set
     */
    public Long setsLength(String key);


    /**
     * Move the specified member from the set at srckey to the set at dstkey. This operation is
     * atomic, in every given moment the element will appear to be in the source or destination set
     * for accessing clients.
     * <p>
     * If the source set does not exist or does not contain the specified element no operation is
     * performed and zero is returned, otherwise the element is removed from the source set and added
     * to the destination set. On success one is returned, even if the element was already present in
     * the destination set.
     * <p>
     * An error is raised if the source or destination keys contain a non Set value.
     * <p>
     *  returns:
     *  <ul>
     *      <li>1 if the element was moved </li>
     *      <li>0 if the element was not found on the first set and no operation was performed</li>
     *  </ul>
     *
     * @param srcKey source key
     * @param targetKey target key
     * @param value value to be moved
     * @return reply
     */
    public Long setsMove(String srcKey, String targetKey, String value);


    /**
     * Return true if member is a member of the set stored at key, otherwise false is returned.
     *
     * @param key key
     * @param value value
     * @return if value exist
     */
    public Boolean setsExist(String key, String value);



    /**
     * Return the members of a set resulting from the intersection of all the sets hold at the
     * specified keys. Like in {@link #listRange(String, long, long)} (String, long, long) LRANGE}
     * the result is sent to the client as a multi-bulk reply (see the protocol specification for more information).
     * If just a single key is specified, then this command produces the same result as
     * {@link #setsValues(String)} (String) SMEMBERS}. Actually SMEMBERS is just syntax sugar for SINTER.
     * <p>
     * Non existing keys are considered like empty sets, so if one of the keys is missing an empty set
     * is returned (since the intersection with an empty set always is an empty set).
     * <p>
     *
     * @param keys set keys
     * @return total set.
     */
    public Set<String> setsInter(String... keys);


    /**
     * This command works exactly like {@link #setsInter(String...)}  SINTER} but instead of being returned
     * the resulting set is stored as dstkey.
     * <p>
     *
     * @param destKey target key
     * @param keys source keys
     * @return Status code reply
     */
    public Long setsInterStore(String destKey, String... keys);


    /**
     * Return the members of a set resulting from the union of all the sets hold at the specified
     * keys. Like in {@link #listRange(String, long, long)} LRANGE} the result is sent to the client as a
     * multi-bulk reply (see the protocol specification for more information). If just a single key is
     * specified, then this command produces the same result as {@link #setsValues(String)}  SMEMBERS}.
     * <p>
     * Non existing keys are considered like empty sets.
     * <p>
     *
     * @param keys set keys
     * @return union values of keys
     */
    public Set<String> setsUnion(String... keys);


    /**
     * This command works exactly like {@link #setsUnion(String...)} SUNION} but instead of being returned
     * the resulting set is stored as dstkey. Any existing value in dstkey will be over-written.
     * <p>
     *
     * @param destKey dest key
     * @param keys keys
     * @return Status code reply
     */
    public Long setsUnionStore(String destKey, String... keys);


    /**
     * Return the difference between the Set stored at key1 and all the Sets key2, ..., keyN
     * <p>
     * <b>Example:</b>
     *
     * <pre>
     * key1 = [x, a, b, c]
     * key2 = [c]
     * key3 = [a, d]
     * SDIFF key1,key2,key3 =&gt; [x, b]
     * </pre>
     *
     * Non existing keys are considered like empty sets.
     *
     * @param keys key1 to keys all.
     * @return different between set keys.
     */
    public Set<String> setsDifferent(String ... keys);


    /**
     * This command works exactly like {@link #setsDifferent(String...)} but instead of being returned
     * the resulting set is stored in destkey.
     *
     * @param destKey dest key
     * @param keys keys
     * @return Status code reply
     */
    public Long setsDifferentStore(String destKey, String... keys);


    /**
     *
     * Return a random element from a Set, without removing the element. If the Set is empty or the
     * key does not exist, a nil object is returned.
     *
     * @param key set key
     * @return random value.
     */
    public String setsRandomValue(String key);


    /**
     * Return a count of random elements from a Set, without removing the element. If the Set is empty or the
     * key does not exist, a nil list is returned.
     *
     * @param key set key
     * @param count value count
     * @return value set.
     */
    public List<String> setsRandomValue(String key, int count);



    // ============================ zset ==============================

    /**
     * Add the specified member having the specified score to the sorted set stored at key. If member
     * is already a member of the sorted set the score is updated, and the element reinserted in the
     * right position to ensure sorting. If key does not exist a new sorted set with the specified
     * member as sole member is created. If the key exists but does not hold a sorted set value an
     * error is returned.
     * <p>
     * The score value can be the string representation of a double precision floating point number.
     * returns:
     * <ul>
     *     <li>1 if the new element was added </li>
     *     <li>0 if the element was already a member of the sorted set and the score was updated</li>
     * </ul>
     *
     * @param key key
     * @param score sort score of this value
     * @param value value
     * @return status code
     */
    public Long zsetAdd(String key, double score, String value);


    /**
     * Add the specified member having the specified score to the sorted set stored at key.
     *
     * @param key key
     * @param score the score of value
     * @param value value of score
     * @param params set params
     * @return status code
     */
    public Long zsetAdd( String key, double score, String value, ZAddParams params);


    /**
     * Add the specified member having the specified score to the sorted set stored at key.
     * with a map contains value and value score.
     *
     * @param key key
     * @param valueScores value score pair
     * @return  status code
     */
    public Long zsetAdd(String key, Map<String, Double> valueScores);


    /**
     * Add the specified member having the specified score to the sorted set stored at key.
     * with a map contains value and value score and Zset add Params.
     *
     * @param key key
     * @param valueScores value score pair
     * @param params add param
     * @return status code
     */
    public Long zsetAdd(String key, Map<String, Double> valueScores,ZAddParams params);


    /**
     * Return a range of zset . this function such as {@link #listRange(String, long, long)}
     *
     *
     * @param key set key
     * @param start start index
     * @param end end index.
     * @return sub range of zset.
     */
    public Set<String> zsetRange(String key, long start, long end);


    /**
     * Remove the specified member from the sorted set value stored at key. If member was not a member
     * of the set no operation is performed. If key does not not hold a set value an error is
     * returned.
     *
     * @param key zset key
     * @param values values
     * @return Integer reply, specifically: 1 if the new element was removed 0 if the new element was
     *     not a member of the set
     */
    public Long zsetDelete(String key, String... values);


    /**
     * If member already exists in the sorted set adds the increment to its score and updates the
     * position of the element in the sorted set accordingly. If member does not already exist in the
     * sorted set it is added with increment as score (that is, like if the previous score was
     * virtually zero). If key does not exist a new sorted set with the specified member as sole
     * member is created. If the key exists but does not hold a sorted set value an error is returned.
     * <p>
     * The score value can be the string representation of a double precision floating point number.
     * It's possible to provide a negative value to perform a decrement.
     * <p>
     * For an introduction to sorted sets check the Introduction to Redis data types page.
     *
     * @param key set key
     * @param increment increment
     * @param value value
     * @return The new score
     */
    public Double zsetIncreaseBy(String key, double increment, String value);


    /**
     * If member already exists in the sorted set adds the increment to its score and updates the
     * position of the element in the sorted set accordingly. If member does not already exist in the
     * sorted set it is added with increment as score (that is, like if the previous score was
     * virtually zero). If key does not exist a new sorted set with the specified member as sole
     * member is created. If the key exists but does not hold a sorted set value an error is returned.
     * <p>
     * The score value can be the string representation of a double precision floating point number.
     * It's possible to provide a negative value to perform a decrement.
     * <p>
     * For an introduction to sorted sets check the Introduction to Redis data types page.
     *
     * @param key set key
     * @param increment increment
     * @param value value
     * @param params incrBy params
     * @return  The new score
     */
    public Double zsetIncreaseBy(String key, double increment, String value, ZIncrByParams params);


    /**
     * Return the rank (or index) of member in the sorted set at key, with scores being ordered from
     * low to high.
     * <p>
     * When the given member does not exist in the sorted set, the special value 'nil' is returned.
     * The returned rank (or index) of the member is 0-based for both commands.
     * <p>
     *
     * @param key zset key
     * @param value zset value
     * @return Integer reply or a nil bulk reply, specifically: the rank of the element as an integer
     *       reply if the element exists. A nil bulk reply if there is no such element.
     */
    public Long zsetIndexOf(String key, String value);


    /**
     * Return the rank (or index) of member in the sorted set at key, with scores being ordered from
     * high to low.
     * <p>
     * When the given member does not exist in the sorted set, the special value 'nil' is returned.
     * The returned rank (or index) of the member is 0-based for both commands.
     *
     * @param key zset key
     * @param value zset value
     * @return Integer reply or a nil bulk reply, specifically: the rank of the element as an integer
     *       reply if the element exists. A nil bulk reply if there is no such element.
     */
    public Long zsetReverseIndexOf(String key, String value);


    /**
     * return reverse range of zset. {@link #zsetRange(String, long, long)}
     *
     *
     * @param key zset key
     * @param start start index
     * @param end end index
     * @return set values
     */
    public Set<String> zsetReverseRange(String key, long start, long end);



    /**
     * Return the sorted set cardinality (number of elements). If the key does not exist 0 is
     * returned, like for empty sorted sets.
     *
     * @param key key
     * @return length of zset
     */
    public Long zsetLength(String key);


    /**
     * Return the score of the specified element of the sorted set at key. If the specified element
     * does not exist in the sorted set, or the key does not exist at all, a special 'nil' value is
     * returned.
     *
     * @param key zset key
     * @param value  zset value
     * @return score of value in key.
     */
    public Double zsetScore(String key, String value);





    /**
     * return range of zset. {@link #zsetRange(String, long, long)}
     * witch is contain in {@link Tuple} with element and score.
     *
     *
     * @param key zset key
     * @param start start index
     * @param end end index
     * @return set values
     */
    public Set<Tuple> zsetRangeWithScores(String key, long start, long end);



    /**
     * Return the all the elements in the sorted set at key with a score between min and max
     * (including elements with score equal to min or max).
     * <p>
     * The elements having the same score are returned sorted lexicographically as ASCII strings (this
     * follows from a property of Redis sorted sets and does not involve further computation).
     * <p>
     * Using the optional {@link #zsetRangeByScore(String, double, double)} LIMIT} it's possible
     * to get only a range of the matching elements in an SQL-alike way. Note that if offset is large
     * the commands needs to traverse the list for offset elements and this adds up to the O(M)
     * figure.
     * <p>
     * The  command is similar to
     * {@link #zsetRangeByScore(String, double, double)}ZRANGEBYSCORE} but instead of returning the
     * actual elements in the specified interval, it just returns the number of matching elements.
     * <p>
     * <b>Exclusive intervals and infinity</b>
     * <p>
     * min and max can be -inf and +inf, so that you are not required to know what's the greatest or
     * smallest element in order to take, for instance, elements "up to a given value".
     * <p>
     * Also while the interval is for default closed (inclusive) it's possible to specify open
     * intervals prefixing the score with a "(" character, so for instance:
     * <p>
     * {@code ZRANGEBYSCORE zset (1.3 5}
     * <p>
     * Will return all the values with score &gt; 1.3 and &lt;= 5, while for instance:
     * <p>
     * {@code ZRANGEBYSCORE zset (5 (10}
     * <p>
     * Will return all the values with score &gt; 5 and &lt; 10 (5 and 10 excluded).
     *
     * @see #zsetRangeByScore(String, double, double)
     * @see #zsetRangeByScore(String, double, double, int, int)
     * @see #zsetRangeByScoreWithScores(String, double, double)
     * @see #zsetRangeByScoreWithScores(String, double, double, int, int)
     *
     *
     * @param key set key
     * @param min min score
     * @param max max score
     *
     * @return Multi bulk reply specifically a list of elements in the specified score range.
     */
    public Set<String> zsetRangeByScore(String key, double min, double max);


    /**
     *  {@link #zsetRangeByScore(String, String, String)}
     *
     * @param key set key
     * @param min min score
     * @param max max score
     *
     * @return Multi bulk reply specifically a list of elements in the specified score range.
     */
    public Set<String> zsetRangeByScore(String key, String  min, String max);


    /**
     * Return the all the elements in the sorted set at key with a score between min and max
     * (including elements with score equal to min or max).
     * <p>
     * The elements having the same score are returned sorted lexicographically as ASCII strings (this
     * follows from a property of Redis sorted sets and does not involve further computation).
     * <p>
     * Using the optional {@link #zsetRangeByScore(String, double, double, int, int) LIMIT} it's possible
     * to get only a range of the matching elements in an SQL-alike way. Note that if offset is large
     * the commands needs to traverse the list for offset elements and this adds up to the O(M)
     * figure.
     * <p>
     * The command is similar to
     * {@link #zsetRangeByScore(String, double, double) ZRANGEBYSCORE} but instead of returning the
     * actual elements in the specified interval, it just returns the number of matching elements.
     * <p>
     * <b>Exclusive intervals and infinity</b>
     * <p>
     * min and max can be -inf and +inf, so that you are not required to know what's the greatest or
     * smallest element in order to take, for instance, elements "up to a given value".
     * <p>
     * Also while the interval is for default closed (inclusive) it's possible to specify open
     * intervals prefixing the score with a "(" character, so for instance:
     * <p>
     * {@code ZRANGEBYSCORE zset (1.3 5}
     * <p>
     * Will return all the values with score &gt; 1.3 and &lt;= 5, while for instance:
     * <p>
     * {@code ZRANGEBYSCORE zset (5 (10}
     * <p>
     * Will return all the values with score &gt; 5 and &lt; 10 (5 and 10 excluded).
     * <p>
     *
     * @param key set key
     * @param min min score
     * @param max max score
     * @param offset offset index
     * @param count count of set
     *
     * @return Multi bulk reply specifically a list of elements in the specified score range.
     */
    public Set<String> zsetRangeByScore(String key, double min, double max, int offset, int count);

    /**
     *  {@link #zsetRangeByScore(String, String, String, int, int)}
     *
     * @param key set key
     * @param min min score
     * @param max max score
     * @param offset offset index
     * @param count count of set
     *
     * @return Multi bulk reply specifically a list of elements in the specified score range.
     */

    public Set<String> zsetRangeByScore(String key, String min, String max, int offset, int count);


    /**
     * Return the all the elements in the sorted set at key with a score between min and max
     * (including elements with score equal to min or max).
     * <p>
     * The elements having the same score are returned sorted lexicographically as ASCII strings (this
     * follows from a property of Redis sorted sets and does not involve further computation).
     * <p>
     * Using the optional {@link #zsetRangeByScore(String, String, String, int, int)}  it's possible
     * to get only a range of the matching elements in an SQL-alike way. Note that if offset is large
     * the commands needs to traverse the list for offset elements and this adds up to the O(M)
     * figure.
     * <p>
     * The command is similar to
     * {@link #zsetRangeByScore(String, double, double) ZRANGEBYSCORE} but instead of returning the
     * actual elements in the specified interval, it just returns the number of matching elements.
     * <p>
     * <b>Exclusive intervals and infinity</b>
     * <p>
     * min and max can be -inf and +inf, so that you are not required to know what's the greatest or
     * smallest element in order to take, for instance, elements "up to a given value".
     * <p>
     * Also while the interval is for default closed (inclusive) it's possible to specify open
     * intervals prefixing the score with a "(" character, so for instance:
     * <p>
     * {@code ZRANGEBYSCORE zset (1.3 5}
     * <p>
     * Will return all the values with score &gt; 1.3 and &lt;= 5, while for instance:
     * <p>
     * {@code ZRANGEBYSCORE zset (5 (10}
     * <p>
     * Will return all the values with score &gt; 5 and &lt; 10 (5 and 10 excluded).
     *
     * @param key key
     * @param min min score
     * @param max max score
     * @return value score pair
     */
    public Set<Tuple> zsetRangeByScoreWithScores(String key, double min, double max);

    /**
     *  {@link #zsetRangeByScoreWithScores(String, String, String)}
     *
     *
     * @param key key
     * @param min min score
     * @param max max score
     * @return value score pair
     */
    public Set<Tuple> zsetRangeByScoreWithScores(String key, String min, String max);

    /**
     *  {@link #zsetRangeByScoreWithScores(String, String, String)}
     *  {@link #zsetRangeByScore(String, double, double, int ,int)}
     *
     * @param key key
     * @param min min score
     * @param max max score
     * @return value score pair
     */
    public Set<Tuple> zsetRangeByScoreWithScores(String key, double min, double max, int offset, int count);

    /**
     *  {@link #zsetRangeByScoreWithScores(String, String, String)}
     *  {@link #zsetRangeByScore(String, double, double, int ,int)}
     *
     * @param key key
     * @param min min score
     * @param max max score
     * @return value score pair
     */
    public Set<Tuple> zsetRangeByScoreWithScores(String key, String min, String max, int offset, int count);


    /**
     * return reverse range of zset. {@link #zsetRange(String, long, long)}
     * witch is contain in {@link Tuple} with element and score.
     *
     *
     * @param key zset key
     * @param start start index
     * @param end end index
     * @return set values
     */
    public Set<Tuple> zsetReverseRangeWithScores(String key, long start, long end);


    /**
     * {@link #zsetRangeByScore(String, double, double)}
     *
     * @param key key
     * @param min min
     * @param max max
     * @return values
     */
    public Set<String> zsetReverseRangeByScore(String key, double min, double max);

    public Set<String> zsetReverseRangeByScore(String key, String  min, String max);

    public Set<String> zsetReverseRangeByScore(String key, double min, double max, int offset, int count);

    public Set<String> zsetReverseRangeByScore(String key, String min, String max, int offset, int count);

    public Set<Tuple> zsetReverseRangeByScoreWithScores(String key, double min, double max);

    public Set<Tuple> zsetReverseRangeByScoreWithScores(String key, String min, String max);

    public Set<Tuple> zsetReverseRangeByScoreWithScores(String key, double min, double max, int offset, int count);

    public Set<Tuple> zsetReverseRangeByScoreWithScores(String key, String min, String max, int offset, int count);


    /**
     * Remove all elements in the sorted set at key with rank between start and end. Start and end are
     * 0-based with rank 0 being the element with the lowest score. Both start and end can be negative
     * numbers, where they indicate offsets starting at the element with the highest rank. For
     * example: -1 is the element with the highest score, -2 the element with the second highest score
     * and so forth.
     *
     * @param key set key
     * @param start start index
     * @param stop end index
     * @return status reply
     */
    public Long zsetDeleteRangeByIndex(String key, long start, long stop);


    /**
     * Remove all the elements in the sorted set at key with a score between min and max (including
     *  elements with score equal to min or max).
     *
     * @param key key
     * @param min min
     * @param max max
     * @return status reply
     */
    public Long zsetDeleteRangeByScore(String key, double min, double max);

    /**
     * {@link #zsetDeleteRangeByScore(String, String, String)}
     *
     * @param key key
     * @param min min
     * @param max max
     * @return status reply
     */
    public Long zsetDeleteRangeByScore(String key, String min, String max);


    /**
     * Creates a union or intersection of N sorted sets given by keys k1 through kN, and stores it at
     * dstkey. It is mandatory to provide the number of input keys N, before passing the input keys
     * and the other (optional) arguments.
     * <p>
     * As the terms imply, the {@link #zsetInterStore(String, String...) ZINTERSTORE} command requires an
     * element to be present in each of the given inputs to be inserted in the result. The
     * {@link #zsetUnionStore(String, String...) ZUNIONSTORE} command inserts all elements across all
     * inputs.
     * <p>
     * Using the WEIGHTS option, it is possible to add weight to each input sorted set. This means
     * that the score of each element in the sorted set is first multiplied by this weight before
     * being passed to the aggregation. When this option is not given, all weights default to 1.
     * <p>
     * With the AGGREGATE option, it's possible to specify how the results of the union or
     * intersection are aggregated. This option defaults to SUM, where the score of an element is
     * summed across the inputs where it exists. When this option is set to be either MIN or MAX, the
     * resulting set will contain the minimum or maximum score of an element across the inputs where
     * it exists.
     * <p>
     *
     * @param destKey target key
     * @param keys source set's keys
     * @return Integer reply, specifically the number of elements in the sorted set at dstkey
     */
    public Long zsetUnionStore(String destKey, String... keys);


    /**
     * Creates a union or intersection of N sorted sets given by keys k1 through kN, and stores it at
     * dstkey. It is mandatory to provide the number of input keys N, before passing the input keys
     * and the other (optional) arguments.
     * <p>
     * As the terms imply, the {@link #zsetInterStore(String, String...) ZINTERSTORE} command requires an
     * element to be present in each of the given inputs to be inserted in the result. The
     * {@link #zsetUnionStore(String, String...) ZUNIONSTORE} command inserts all elements across all
     * inputs.
     * <p>
     * Using the WEIGHTS option, it is possible to add weight to each input sorted set. This means
     * that the score of each element in the sorted set is first multiplied by this weight before
     * being passed to the aggregation. When this option is not given, all weights default to 1.
     * <p>
     * With the AGGREGATE option, it's possible to specify how the results of the union or
     * intersection are aggregated. This option defaults to SUM, where the score of an element is
     * summed across the inputs where it exists. When this option is set to be either MIN or MAX, the
     * resulting set will contain the minimum or maximum score of an element across the inputs where
     * it exists.
     * <p>
     *
     * @param destKey target key
     * @param params set params
     * @param keys source set's keys
     * @return Integer reply, specifically the number of elements in the sorted set at dstkey
     */
    public Long zsetUnionStore(String destKey, ZParams params, String... keys);


    /**
     * Creates a union or intersection of N sorted sets given by keys k1 through kN, and stores it at
     * dstkey. It is mandatory to provide the number of input keys N, before passing the input keys
     * and the other (optional) arguments.
     * <p>
     * As the terms imply, the {@link #zsetInterStore(String, String...) ZINTERSTORE} command requires an
     * element to be present in each of the given inputs to be inserted in the result. The
     * {@link #zsetUnionStore(String, String...) ZUNIONSTORE} command inserts all elements across all
     * inputs.
     * <p>
     * Using the WEIGHTS option, it is possible to add weight to each input sorted set. This means
     * that the score of each element in the sorted set is first multiplied by this weight before
     * being passed to the aggregation. When this option is not given, all weights default to 1.
     * <p>
     * With the AGGREGATE option, it's possible to specify how the results of the union or
     * intersection are aggregated. This option defaults to SUM, where the score of an element is
     * summed across the inputs where it exists. When this option is set to be either MIN or MAX, the
     * resulting set will contain the minimum or maximum score of an element across the inputs where
     * it exists.
     * <p>
     *
     * @param destKey target key
     * @param keys source set's keys
     * @return Integer reply, specifically the number of elements in the sorted set at dstkey
     */
    public Long zsetInterStore(String destKey, String... keys);


    /**
     * Creates a union or intersection of N sorted sets given by keys k1 through kN, and stores it at
     * dstkey. It is mandatory to provide the number of input keys N, before passing the input keys
     * and the other (optional) arguments.
     * <p>
     * As the terms imply, the {@link #zsetInterStore(String, String...) ZINTERSTORE} command requires an
     * element to be present in each of the given inputs to be inserted in the result. The
     * {@link #zsetUnionStore(String, String...) ZUNIONSTORE} command inserts all elements across all
     * inputs.
     * <p>
     * Using the WEIGHTS option, it is possible to add weight to each input sorted set. This means
     * that the score of each element in the sorted set is first multiplied by this weight before
     * being passed to the aggregation. When this option is not given, all weights default to 1.
     * <p>
     * With the AGGREGATE option, it's possible to specify how the results of the union or
     * intersection are aggregated. This option defaults to SUM, where the score of an element is
     * summed across the inputs where it exists. When this option is set to be either MIN or MAX, the
     * resulting set will contain the minimum or maximum score of an element across the inputs where
     * it exists.
     * <p>
     *
     * @param destKey target key
     * @param params set params
     * @param keys source set's keys
     * @return Integer reply, specifically the number of elements in the sorted set at dstkey
     */
    public Long zsetInterStore(String destKey, ZParams params, String... keys);


    /**
     *  Return count of set values that lex value between min and max
     *  such as:
     *  a -&gt; z
     *
     *
     * @param key key
     * @param min min lex
     * @param max max lex
     * @return count
     */
    public Long zsetLexLength(String key, String min, String max);


    /**
     *  {@link #zsetRangeByScore(String, double, double)}
     *  {@link #zsetLexLength(String, String, String)}
     *
     * @param key key
     * @param min min lex
     * @param max max lex
     * @return values between min and max
     */
    public Set<String> zsetRangeByLex(String key, String min, String max);

    public Set<String> zsetRangeByLex(String key, String min, String max, int offset, int count);

    public Set<String> zsetReverseRangeByLex(String key, String min, String max);

    public Set<String> zsetReverseRangeByLex(String key, String min, String max,int offset, int count);

    public Long zsetDeleteRangeByLex(String key, String min, String max);


    /**
     * Sort a Set or a List.
     * <p>
     * Sort the elements contained in the List, Set, or Sorted Set value at key. By default sorting is
     * numeric with elements being compared as double precision floating point numbers. This is the
     * simplest form of SORT.
     *
     * @see #sortStore(String, String)
     * @see #sort(String, SortingParams)
     * @see #sortStore(String, SortingParams, String)
     *
     * @param key key
     * @return sorted values
     */
    public List<String> sort(String key);


    /**
     * Sort a Set or a List accordingly to the specified parameters.
     * <p>
     * <b>examples:</b>
     * <p>
     * Given are the following sets and key/values:
     *
     * <pre>
     * x = [1, 2, 3]
     * y = [a, b, c]
     *
     * k1 = z
     * k2 = y
     * k3 = x
     *
     * w1 = 9
     * w2 = 8
     * w3 = 7
     * </pre>
     *
     * Sort Order:
     *
     * <pre>
     * sort(x) or sort(x, sp.asc())
     * -&gt; [1, 2, 3]
     *
     * sort(x, sp.desc())
     * -&gt; [3, 2, 1]
     *
     * sort(y)
     * -&gt; [c, a, b]
     *
     * sort(y, sp.alpha())
     * -&gt; [a, b, c]
     *
     * sort(y, sp.alpha().desc())
     * -&gt; [c, a, b]
     * </pre>
     *
     * Limit (e.g. for Pagination):
     *
     * <pre>
     * sort(x, sp.limit(0, 2))
     * -&gt; [1, 2]
     *
     * sort(y, sp.alpha().desc().limit(1, 2))
     * -&gt; [b, a]
     * </pre>
     *
     * Sorting by external keys:
     *
     * <pre>
     * sort(x, sb.by(w*))
     * -&gt; [3, 2, 1]
     *
     * sort(x, sb.by(w*).desc())
     * -&gt; [1, 2, 3]
     * </pre>
     *
     * Getting external keys:
     *
     * <pre>
     * sort(x, sp.by(w*).get(k*))
     * -&gt; [x, y, z]
     *
     * sort(x, sp.by(w*).get(#).get(k*))
     * -&gt; [3, x, 2, y, 1, z]
     * </pre>
     *
     *
     * @param key key
     * @param sortingParameters sorting parameters
     * @return sorted values
     */
    public List<String> sort(String key, SortingParams sortingParameters);


    /**
     * Sort a Set or a List and Store the Result at dstkey.
     * <p>
     * Sort the elements contained in the List, Set, or Sorted Set value at key and store the result
     * at dstkey. By default sorting is numeric with elements being compared as double precision
     * floating point numbers. This is the simplest form of SORT.
     *
     * @param key key
     * @param destKey target key
     * @return The number of elements of the list at dstkey.
     */
    public Long sortStore(String key, String destKey);


    /**
     * Sort a Set or a List and Store the Result at dstkey.
     * <p>
     * Sort the elements contained in the List, Set, or Sorted Set value at key and store the result
     * at dstkey. By default sorting is numeric with elements being compared as double precision
     * floating point numbers. This is the simplest form of SORT.
     *
     * @param key key
     * @param sortingParameters param
     * @param destKey target key
     * @return The number of elements of the list at dstkey.
     */
    public Long sortStore(String key, SortingParams sortingParameters, String destKey);


    // ============================= geo ==================================


    /**
     *  Add a geo location of value
     *
     * @param key key to geo
     * @param longitude longitude
     * @param latitude latitude
     * @param value value
     * @return status code
     */
    public Long geoAdd(String key, double longitude, double latitude, String value);


    public Long geoAdd( String key, Map<String, GeoCoordinate> memberCoordinateMap);


    public Double geoDistant( String key, String source, String target);


    public Double geoDistant( String key, String source, String target, GeoUnit unit);


    public List<String> geoHash( String key, String... values);


    public List<GeoCoordinate> geoPos( String key, String... values);


    /**
     * Find values that distant is in radius with longitude and latitude.
     *
     * @param key key
     * @param longitude lon
     * @param latitude lat
     * @param radius radius
     * @param unit unit
     * @return members.
     */
    public List<GeoRadiusResponse> geoRadius( String key, double longitude, double latitude, double radius, GeoUnit unit);


    public List<GeoRadiusResponse> geoRadiusReadonly(String key, double longitude, double latitude, double radius, GeoUnit unit);


    public List<GeoRadiusResponse> geoRadius(String key, double longitude, double latitude, double radius,
                                             GeoUnit unit, GeoRadiusParam param);


    public List<GeoRadiusResponse> geoRadiusReadonly(String key, double longitude, double latitude,
                                                     double radius, GeoUnit unit, GeoRadiusParam param);

    public List<GeoRadiusResponse> geoRadiusByMember( String key, String member,
                                                      double radius, GeoUnit unit);


    public List<GeoRadiusResponse> geoRadiusByMemberReadonly(String key, String member, double radius, GeoUnit unit);


    public List<GeoRadiusResponse> geoRadiusByMember(String key, String member, double radius,
                                                     GeoUnit unit, GeoRadiusParam param);


    public List<GeoRadiusResponse> geoRadiusByMemberReadonly(String key, String member, double radius,
                                                             GeoUnit unit, GeoRadiusParam param);

    // ============================= other ==================================


    /**
     * subscribe channels ..
     *
     * @param jedisPubSub callback when channels has publish message
     * @param channels changes
     */
    public void subscribe(JedisPubSub jedisPubSub, String... channels);


    /**
     * publish message to channel.
     *
     * @param channel target channel.
     * @param message message
     * @return status code
     */
    public Long publish(String channel, String message);


    /**
     *  subscribe channels in patterns...
     *
     * @param jedisPubSub callback
     * @param patterns channel pattern
     */
    public void patternSubscribe(JedisPubSub jedisPubSub, String... patterns);




    /**
     * Eval a script in lua .
     * <code>
     *  redis 127.0.0.1:6379&gt; eval "return {KEYS[1],KEYS[2],ARGV[1],ARGV[2]}" 2 key1 key2 first second
     * 1) "key1"
     * 2) "key2"
     * 3) "first"
     * 4) "second"
     * </code>
     *
     * @param script script content
     * @param keys keys
     * @param args arg
     * @return object
     */
    public Object eval(String script, List<String> keys, List<String> args);


    public Object eval(String script, int keyCount, String... params);


    public Object eval(String script);


    public Object evalSha(String sha1);


    public Object evalSha(String sha1, List<String> keys, List<String> args);


    public Object evalSha(String sha1, int keyCount, String... params);


    public Boolean scriptExists(String sha1);


    public List<Boolean> scriptExists(String... sha1);


    public String scriptLoad(final String script);

}
