package cn.ipokerface.redis.spring;

import cn.ipokerface.redis.JedisClient;
import cn.ipokerface.redis.JedisClientCluster;
import cn.ipokerface.redis.JedisClientStandalone;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.util.CollectionUtils;
import redis.clients.jedis.HostAndPort;
import redis.clients.jedis.JedisCluster;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;

import java.util.HashSet;
import java.util.Set;

/**
 * Created by       mac
 * Create Date      2020/12/11
 * Version          1.0.0
 * Description:
 */


@ConditionalOnProperty(name = "spring.redis.enabled", havingValue = "true", matchIfMissing = false)
@EnableConfigurationProperties({JedisProperties.class})
@Configuration
public class JedisAutoConfiguration {

    @Autowired
    private JedisProperties jedisProperties;


    @Bean
    @ConditionalOnMissingBean(JedisPoolConfig.class)
    public JedisPoolConfig jedisPoolConfig() {
        JedisPoolConfig jedisPoolConfig = new JedisPoolConfig();
        jedisPoolConfig.setMaxTotal(jedisProperties.getMaxTotal());
        jedisPoolConfig.setMaxIdle(jedisProperties.getMaxIdle());
        jedisPoolConfig.setMinIdle(jedisProperties.getMinIdle());
        jedisPoolConfig.setMaxWaitMillis(jedisProperties.getMaxWaitMillis());
        jedisPoolConfig.setMinEvictableIdleTimeMillis(jedisProperties.getMinEvictableIdleTimeMillis());
        jedisPoolConfig.setNumTestsPerEvictionRun(jedisProperties.getNumTestsPerEvictionRun());
        jedisPoolConfig.setTimeBetweenEvictionRunsMillis(jedisProperties.getTimeBetweenEvictionRunsMillis());
        jedisPoolConfig.setTestOnBorrow(jedisProperties.isTestOnBorrow());
        jedisPoolConfig.setTestWhileIdle(jedisProperties.isTestWhileIdle());
        jedisPoolConfig.setTestOnCreate(jedisProperties.isTestOnCreate());
        return jedisPoolConfig;
    }


    @Primary
    @Bean
    @ConditionalOnProperty(value = "spring.redis.mode",havingValue = "standalone", matchIfMissing = true)
    public JedisPool jedisPool(JedisPoolConfig jedisPoolConfig) {
        return new JedisPool(jedisPoolConfig,
                jedisProperties.getHost(),
                jedisProperties.getPort(),
                jedisProperties.getTimeout(),
                jedisProperties.getPassword(),
                jedisProperties.getDatabase());
    }

    @Primary
    @Bean
    @ConditionalOnProperty(value = "spring.redis.mode",havingValue = "standalone", matchIfMissing = true)
    public JedisClient jedisClientStandalone(JedisPool jedisPool) {
        return new JedisClientStandalone(jedisPool);
    }






    @Bean
    @ConditionalOnMissingBean({JedisPool.class, JedisCluster.class})
    @ConditionalOnProperty(name = "spring.redis.mode", havingValue = "cluster", matchIfMissing = false)
    public JedisCluster jedisCluster(JedisPoolConfig jedisPoolConfig)
    {
        Set<HostAndPort> hostAndPorts = new HashSet<HostAndPort>();
        if (CollectionUtils.isEmpty(jedisProperties.getClusters()))
            throw new IllegalArgumentException("redis cluster nodes is not set!");

        for(String ipPort: jedisProperties.getClusters()){
            String[] ipAndPorts = ipPort.split(":");
            if (ipAndPorts == null || ipAndPorts.length != 2)
                throw new IllegalArgumentException("Invalid ip and port of node:"+ ipPort);
            hostAndPorts.add(new HostAndPort(ipAndPorts[0], Integer.parseInt(ipAndPorts[1])));
        }

        return new JedisCluster(hostAndPorts,
                jedisProperties.getConnectionTimeout(),
                jedisProperties.getSocketTimeout(),
                jedisProperties.getMaxAttempts(),
                jedisProperties.getPassword(),
                jedisProperties.getClusterName(),
                jedisPoolConfig);
    }



    @Bean
    @ConditionalOnMissingBean({JedisClient.class, JedisClientStandalone.class})
    @ConditionalOnProperty(name = "spring.redis.mode", havingValue = "cluster", matchIfMissing = false)
    public JedisClient jedisClientCluster(JedisCluster jedisCluster) {
        return new JedisClientCluster(jedisCluster);
    }

}
