package cn.ipokerface.redis.spring;

import cn.ipokerface.redis.JedisClient;
import cn.ipokerface.redis.JedisClientCluster;
import cn.ipokerface.redis.JedisClientStandalone;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.util.CollectionUtils;
import redis.clients.jedis.HostAndPort;
import redis.clients.jedis.JedisCluster;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;

import java.util.HashSet;
import java.util.Set;

/**
 * Created by       mac
 * Create Date      2020/12/11
 * Version          1.0.0
 * Description:
 */


@ConditionalOnProperty(name = "spring.redis.enabled", matchIfMissing = true)
@EnableConfigurationProperties({JedisPoolProperties.class, JedisProperties.class, JedisClusterProperties.class})
@Configuration
public class JedisAutoConfiguration {




    @Bean
    @ConditionalOnMissingBean(JedisPoolConfig.class)
    public JedisPoolConfig jedisPoolConfig(JedisPoolProperties jedisPoolProperties) {
        JedisPoolConfig jedisPoolConfig = new JedisPoolConfig();
        jedisPoolConfig.setMaxTotal(jedisPoolProperties.getMaxTotal());
        jedisPoolConfig.setMaxIdle(jedisPoolProperties.getMaxIdle());
        jedisPoolConfig.setMinIdle(jedisPoolProperties.getMinIdle());
        jedisPoolConfig.setMaxWaitMillis(jedisPoolProperties.getMaxWaitMillis());
        jedisPoolConfig.setMinEvictableIdleTimeMillis(jedisPoolProperties.getMinEvictableIdleTimeMillis());
        jedisPoolConfig.setNumTestsPerEvictionRun(jedisPoolProperties.getNumTestsPerEvictionRun());
        jedisPoolConfig.setTimeBetweenEvictionRunsMillis(jedisPoolProperties.getTimeBetweenEvictionRunsMillis());
        jedisPoolConfig.setTestOnBorrow(jedisPoolProperties.isTestOnBorrow());
        jedisPoolConfig.setTestWhileIdle(jedisPoolProperties.isTestWhileIdle());
        jedisPoolConfig.setTestOnCreate(jedisPoolProperties.isTestOnCreate());
        return jedisPoolConfig;
    }


    @Primary
    @Bean
    @ConditionalOnProperty(value = "spring.redis.standalone", matchIfMissing = true)
    public JedisPool jedisPool(JedisPoolConfig jedisPoolConfig,JedisProperties jedisProperties) {
        return new JedisPool(jedisPoolConfig,
                jedisProperties.getHost(),
                jedisProperties.getPort(),
                jedisProperties.getTimeout(),
                jedisProperties.getPassword(),
                jedisProperties.getDatabase());
    }

    @Primary
    @Bean
    @ConditionalOnProperty(value = "spring.redis.standalone", matchIfMissing = true)
    public JedisClient jedisClientStandalone(JedisPool jedisPool) {
        return new JedisClientStandalone(jedisPool);
    }






    @Bean
    @ConditionalOnMissingBean({JedisPool.class, JedisCluster.class})
    @ConditionalOnProperty(name = "spring.redis.standalone", havingValue = "false", matchIfMissing = false)
    public JedisCluster jedisCluster(JedisPoolConfig jedisPoolConfig,JedisClusterProperties jedisClusterProperties)
    {
        Set<HostAndPort> hostAndPorts = new HashSet<HostAndPort>();
        if (CollectionUtils.isEmpty(jedisClusterProperties.getClusterNodes()))
            throw new IllegalArgumentException("redis cluster nodes is not set!");

        for(String ipPort: jedisClusterProperties.getClusterNodes()){
            String[] ipAndPorts = ipPort.split(":");
            if (ipAndPorts == null || ipAndPorts.length != 2)
                throw new IllegalArgumentException("Invalid ip and port of node:"+ ipPort);
            hostAndPorts.add(new HostAndPort(ipAndPorts[0], Integer.parseInt(ipAndPorts[1])));
        }

        return new JedisCluster(hostAndPorts,
                jedisClusterProperties.getConnectionTimeout(),
                jedisClusterProperties.getSocketTimeout(),
                jedisClusterProperties.getMaxAttempts(),
                jedisClusterProperties.getPassword(),
                jedisClusterProperties.getClusterName(),
                jedisPoolConfig);
    }



    @Bean
    @ConditionalOnMissingBean({JedisClient.class, JedisClientStandalone.class})
    @ConditionalOnProperty(name = "spring.redis.standalone", havingValue = "false", matchIfMissing = false)
    public JedisClient jedisClientCluster(JedisCluster jedisCluster) {
        return new JedisClientCluster(jedisCluster);
    }

}
