package cn.ipokerface.netty.client;

import cn.ipokerface.netty.client.connection.ConnectionIdleManager;
import cn.ipokerface.netty.client.connection.ConnectionListener;
import cn.ipokerface.netty.NettyContext;
import cn.ipokerface.netty.client.connection.ConnectionInboundHandlerAdapter;
import cn.ipokerface.netty.client.connection.ConnectionManager;
import cn.ipokerface.netty.connection.Connection;
import cn.ipokerface.netty.message.Request;
import cn.ipokerface.netty.protocol.PacketInboundHandlerAdapter;
import cn.ipokerface.netty.protocol.ProtocolInboundHandlerFrameDecoder;
import cn.ipokerface.netty.request.RequestInboundHandlerAdapter;
import cn.ipokerface.netty.request.RequestHandler;
import cn.ipokerface.netty.request.ResponseListener;
import cn.ipokerface.netty.request.interceptor.AfterHandlerInterceptor;
import cn.ipokerface.netty.request.interceptor.BeforeHandlerInterceptor;
import cn.ipokerface.netty.transaction.ResponseInboundHandlerAdapter;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.ChannelOption;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Created by       PokerFace
 * Create Date      2020-03-10.
 * Email:           <a href="mailto:214888341@163.com">214888341@163.com</a>
 * Version          1.0.0
 * <p>
 * Description:
 */
public class NettyClient extends NettyContext {

    private static final Logger logger = LoggerFactory.getLogger(NettyClient.class);


    /**
     * connection bootstrap
     *
     */
    private Bootstrap bootstrap;

    /**
     *  server host
     *
     */
    private String server;

    /**
     * server port
     *
     */
    private int port;


    private int connectionTimeout;


    private  ConnectionManager connectionManager;

    private ConnectionIdleManager connectionIdleManager;



    public NettyClient()
    {
        super();
        initialBootstrap();
    }



    public NettyClient(int protocol, int version)
    {
        super();
        this.protocol = protocol;
        this.version = version;
        initialBootstrap();
    }

    public NettyClient(int protocol, int version, boolean socketKeepAlive, boolean tcpNoDelay, int connectionTimeout)
    {
        super();
        this.protocol = protocol;
        this.version = version;
        this.socketKeepAlive = socketKeepAlive;
        this.tcpNoDelay = tcpNoDelay;
        this.connectionTimeout = connectionTimeout;
        this.connectionManager = new ConnectionManager(this);
        initialBootstrap();
    }



    private void initialBootstrap()
    {
        NioEventLoopGroup workerGroup = new NioEventLoopGroup();

        bootstrap = new Bootstrap();
        bootstrap
                // 1.指定线程模型
                .group(workerGroup)
                // 2.指定 IO 类型为 NIO
                .channel(NioSocketChannel.class)
                .option(ChannelOption.SO_KEEPALIVE,socketKeepAlive)
                .option(ChannelOption.TCP_NODELAY, tcpNoDelay)
                .option(ChannelOption.CONNECT_TIMEOUT_MILLIS, connectionTimeout)
                // 3.IO 处理逻辑
                .handler(new ChannelInitializer<SocketChannel>() {
                    @Override
                    public void initChannel(SocketChannel ch) {
                        // Add connection handler
                        ch.pipeline().addLast(new ConnectionInboundHandlerAdapter(connectionManager));
                        // add packet decoder.
                        ch.pipeline().addLast(new ProtocolInboundHandlerFrameDecoder());
                        // Add packet checker
                        ch.pipeline().addLast(new PacketInboundHandlerAdapter(NettyClient.this, NettyClient.this.decoder));
                        // Add response handler
                        ch.pipeline().addLast(new ResponseInboundHandlerAdapter(NettyClient.this, NettyClient.this.transactionManager));
                        // Add Request handler
                        ch.pipeline().addLast(new RequestInboundHandlerAdapter(NettyClient.this, NettyClient.this.requestMappingManager));

                    }
                });
    }


    /**
     *  Set connection listener of connection.
     *
     * @param clientConnectionListener
     * @return
     */
    public NettyClient listener(ConnectionListener clientConnectionListener)
    {
        connectionManager.setConnectionListener(clientConnectionListener);
        return this;
    }


    /**
     *  开启心跳, 默认间隔20秒
     *
     * @param heartbeatMessage
     * @return
     */
    public NettyClient heartbeat(Request heartbeatMessage)
    {
        connectionIdleManager.setHeartMessage(heartbeatMessage, ConnectionIdleManager.default_idle);
        return this;
    }

    /**
     *  开启心跳
     *
     * @param heartbeatMessage
     * @return
     */
    public NettyClient heartbeat(Request heartbeatMessage,long timeIdle)
    {
        connectionIdleManager.setHeartMessage(heartbeatMessage, timeIdle);
        return this;
    }


    /**
     *  try to connect to server.
     *
     * @param server  server host or servername to be connected
     * @param port server port to be connected.
     */
    public void connect(String server, int port){
        this.server = server;
        this.port = port;
        start(0);
    }

    /**
     *  try to reconnect to host and port by the first time.
     *
     */
    public void reconnect(){
        logger.info("Try to rebuild connection with server!!");

        start(0);
    }

    /**
     *  try to connect server . If failed by 3 times  will
     *
     * @param reties
     */
    private void start(final int reties){
        // 4.建立连接
        bootstrap.connect(server, port).addListener(new GenericFutureListener<Future<? super Void>>()
        {
            public void operationComplete(Future<? super Void> future) throws Exception
            {
                if (future.isSuccess()) {
                    logger.info("NettyClient connect success !!!");

                } else {
                    if (reties > 2){
                        logger.error("NettyClient connect to server failed !!");
                    }else{
                        start(reties + 1 );
                    }
                }
            }
        });
    }


    @Override
    public Connection getConnection(ChannelHandlerContext context) {
        return this.connectionManager.getConnection();
    }

    /**
     * request to server
     * @param request
     */
    public void request(Request request)
    {
        sendRequest(this.connectionManager.getConnection(), request);
    }

    /**
     *  Request to server
     *
     * @param request
     * @param listener
     */
    public void request(Request request, ResponseListener listener)
    {
        sendRequest(this.connectionManager.getConnection(), request, listener);
    }


    /**
     *  register request mapping handler
     *
     * @param command
     * @param requestHandler
     * @return
     */
    public NettyClient registerHandler(String command, RequestHandler requestHandler)
    {
        super.registerRequestHandler(command, requestHandler);
        return this;
    }


    /**
     *
     * @param beforeHandlerInterceptor
     * @return
     */
    public NettyClient registerInterceptor(BeforeHandlerInterceptor beforeHandlerInterceptor)
    {
        super.registerRequestInterceptor(beforeHandlerInterceptor);
        return this;
    }

    public NettyClient registerInterceptor(AfterHandlerInterceptor afterHandlerInterceptor)
    {
        super.registerRequestInterceptor(afterHandlerInterceptor);
        return this;
    }



}
